"use client";
/* eslint-disable @typescript-eslint/no-explicit-any */
import { usePathname } from "next/navigation";
import Drawer from "@mui/material/Drawer";
import IconButton from "@mui/material/IconButton";
import CloseIcon from "@mui/icons-material/Close";
import Typography from "@mui/material/Typography";
import { appsMobile as appList } from "../constants/products";
import ClikkleLogo from "./ClikkleLogo";
import ThemeSwitch from "./themeSwitch";
import { ArrowRight, ArrowRightIcon } from "lucide-react";
import { useTheme } from "../atoms/theme";
import { Button } from "./button";
import { footerColumns } from "../constants/footerText";
import Link from "next/link";
import Image from "next/image";
import { iconMap } from "../utils/iconMap";

const sidebarRoutes = {
  "#features": "Features",
  "/about": "Why Free?",
  "/blogs": "Blog",
};
const externalPages = [
  "/about",
  "/privacy-policy",
  "/terms-of-service",
  "/affiliate-program",
];

const signInMap = {
  "/files": (theme: any) => `https://files.clikkle.com/login?theme=${theme}`,
  "/hr": (theme: any) => `https://hr.clikkle.com/login?theme=${theme}`,
  "/projects": (theme: any) =>
    `https://projects.clikkle.com/login?theme=${theme}`,
  "/esign": (theme: any) => `https://esign.clikkle.com/login?theme=${theme}`,
  "/chat": (theme: any) => `https://chat.clikkle.com/login?theme=${theme}`,
  "/campaigns": (theme: any) =>
    `https://campaigns.clikkle.com/login?theme=${theme}`,
  "/host": (theme: any) => `https://host.clikkle.com/login?theme=${theme}`,
  "/mail": (theme: any) => `https://mail.clikkle.com/login?theme=${theme}`,
  "/crew": (theme: any) => `https://crew.clikkle.com/login?theme=${theme}`,
  "/crm": (theme: any) => `https://crm.clikkle.com/login?theme=${theme}`,
};

const signUpMap = {
  "/files": (theme: any) => `https://files.clikkle.com/signup?theme=${theme}`,
  "/hr": (theme: any) => `https://hr.clikkle.com/signup?theme=${theme}`,
  "/projects": (theme: any) =>
    `https://projects.clikkle.com/signup?theme=${theme}`,
  "/esign": (theme: any) => `https://esign.clikkle.com/signup?theme=${theme}`,
  "/chat": (theme: any) => `https://chat.clikkle.com/signup?theme=${theme}`,
  "/campaigns": (theme: any) =>
    `https://campaigns.clikkle.com/signup?theme=${theme}`,
  "/host": (theme: any) => `https://host.clikkle.com/signup?theme=${theme}`,
  "/mail": (theme: any) => `https://mail.clikkle.com/signup?theme=${theme}`,
  "/crew": (theme: any) => `https://crew.clikkle.com/signup?theme=${theme}`,
  "/crm": (theme: any) => `https://crm.clikkle.com/signup?theme=${theme}`,
};

export interface SidebarDrawerProps {
  open: boolean;
  toggle: () => void;
}

const SidebarDrawer = ({ open, toggle }: SidebarDrawerProps) => {
  const pathname = usePathname();
  const { theme } = useTheme();

  const currentPath = pathname;
  const resolvedSidebarRoutes = Object.fromEntries(
    Object.entries(sidebarRoutes).map(([route, label]) => {
      if (["#features", "#pricing"].includes(route)) {
        if (currentPath && externalPages.includes(currentPath)) {
          return [`https://clikkle.com${route}`, label];
        }
      }
      return [route, label];
    })
  );

  const isSignInKey = (key: string): key is keyof typeof signInMap =>
    key in signInMap;
  const isSignUpKey = (key: string): key is keyof typeof signUpMap =>
    key in signUpMap;

  const signInUrl =
    (currentPath && isSignInKey(currentPath) && signInMap[currentPath](theme)) ||
    (currentPath === "/"
      ? `https://hub.clikkle.com/login?theme=${theme}`
      : `https://hub.clikkle.com/login?referralApp=Clikkle%20Website&&theme=` +
        theme);

  const signUpUrl =
    (currentPath && isSignUpKey(currentPath) && signUpMap[currentPath](theme)) ||
    (currentPath === "/"
      ? `https://hub.clikkle.com/signup?theme=${theme}`
      : `https://hub.clikkle.com/signup?referralApp=Clikkle%20Website&&theme=` +
        theme);

  return (
    <Drawer
      anchor="top"
      open={open}
      onClose={toggle}
      transitionDuration={{ enter: 300, exit: 300 }}
      BackdropProps={{
        style: {
          backgroundColor: "rgba(0,0,0,0.4)",
          transition: "opacity 300ms ease",
        },
      }}
      PaperProps={{
        style: {
          width: "100%",
          height: "100vh",
          borderRadius: 0,
          background: "transparent",
          boxShadow: "none",
          overflow: "hidden",
        },
      }}
      ModalProps={{ keepMounted: true }}
    >
      <div
        className={`w-full h-full transition-all duration-300 ease-in-out overflow-y-auto transform ${
          open ? "opacity-100 translate-y-0" : "opacity-0 -translate-y-4"
        }`}
      >
        <div className="w-full min-h-full bg-white/80 dark:bg-[#0b0b0b]/70 backdrop-blur-sm shadow-sm pb-16">
          <div className="p-2 flex items-center justify-between">
            <div className="flex-grow">
              <ClikkleLogo />
            </div>
            <ThemeSwitch />
            <IconButton onClick={toggle} aria-label="Close sidebar">
              <CloseIcon />
            </IconButton>
          </div>

          <div className="flex justify-center items-center my-3">
            <div className="flex items-center gap-2 rounded-xl border w-fit border-gray-300 dark:border-[#ffffff3a] p-1 shadow-sm">
              <button
                className="cursor-pointer px-3 py-1 text-sm font-medium text-[#253237] dark:text-white"
                onClick={() => {
                  window.location.replace(signInUrl);
                }}
              >
                Login
              </button>

              <button
                className="cursor-pointer flex items-center gap-1 rounded-xl bg-[#253237] dark:bg-white hover:bg-[#1a2428] dark:hover:bg-gray-100 px-4 py-1.5 text-sm font-medium text-white dark:text-[#253237] transition"
                onClick={() => {
                  window.location.replace(signUpUrl);
                }}
              >
                Sign Up
                <span>
                  <ArrowRightIcon className="w-4" />
                </span>
              </button>
            </div>
          </div>
          <div className="m-4 p-4 rounded-lg border border-gray-300 dark:border-[#ffffff3a]">
            <div className="flex flex-col gap-2">
              {Object.entries(resolvedSidebarRoutes).map(([route, label]) =>
                route.startsWith("http") ? (
                  <Link
                    href={route}
                    key={route}
                    onClick={toggle}
                    className="cursor-pointer"
                  >
                    <Typography className="text-gray-700 dark:text-gray-300 hover:text-[#286efb] dark:hover:text-[#286efb]">
                      {label}
                    </Typography>
                  </Link>
                ) : (
                  <Link
                    href={route}
                    key={route}
                    onClick={toggle}
                    style={{ cursor: "pointer" }}
                  >
                    <Typography className="text-gray-700 dark:text-gray-300 hover:text-[#286efb] dark:hover:text-[#286efb]">
                      {label}
                    </Typography>
                  </Link>
                )
              )}
            </div>

            <div className="flex flex-col border-t border-gray-200 dark:border-gray-700  mt-4">
              <h1 className="text-md font-semibold text-black dark:text-white tracking-wide py-4">
                Products
              </h1>
              {appList.map((app) => {
                const IconComponent = iconMap[app.icon];
                return (
                  <div
                    key={app.id}
                    className="flex items-center gap-3 cursor-pointer rounded-md mt-1"
                  >
                    <Link href={app.route} onClick={toggle}>
                      {IconComponent && (
                        <div className="flex items-center justify-center w-10 h-10 rounded-xl bg-white dark:bg-[#1a1a1a] border border-gray-200 dark:border-[#2a2a2a]">
                          <IconComponent className="h-[18px] w-[18px] text-gray-900 dark:text-white" />
                        </div>
                      )}
                    </Link>
                  <Link href={app.route} onClick={toggle}>
                    <Typography
                      className="text-xs sm:text-base md:text-lg flex items-center gap-1 text-gray-900 dark:text-white"
                      sx={{ p: 1 }}
                    >
                      {app.title} <ArrowRight className="w-3" />{" "}
                      <span className="text-[10px]">{app.description}</span>
                    </Typography>
                  </Link>
                </div>
                );
              })}
              <Link href="/products">
                <Button
                  style={{ marginTop: "16px", marginBottom: "20px" }}
                  variant="ghost"
                  className={`cursor-pointer hover:bg-[#286efb] w-44 hover:dark:bg-[#286efb] hover:text-white hover:dark:text-[#eeeef0] font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#1a1a24] text-gray-900 dark:text-[#eeeef0] border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                >
                  View all products
                  <ArrowRightIcon className="h-2 w-2 dark:text-white" />
                </Button>
              </Link>

              <div className="border-t border-gray-200 dark:border-gray-700 pt-4 ">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                  {footerColumns.map((column, index) => (
                    <div key={index} className="space-y-3">
                      <h1 className="text-md font-semibold text-black dark:text-white tracking-wide">
                        {column.title}
                      </h1>
                      <div className="space-y-2 ">
                        {column.links.map((link, linkIndex) => (
                          <div key={linkIndex}>
                            {link.url.startsWith("http") ? (
                              <a
                                href={link.url}
                                target="_blank"
                                rel="noopener noreferrer"
                                onClick={toggle}
                                className="flex items-center gap-1 text-sm text-gray-700 dark:text-gray-300 hover:text-[#286efb] dark:hover:text-[#286efb] transition-colors"
                              >
                                {link.text}
                                {link.isNew && (
                                  <span className="bg-[#253237]/80 text-white dark:bg-white dark:text-black/80 text-xs px-1.5 py-0.5 rounded-md font-medium">
                                    New
                                  </span>
                                )}
                              </a>
                            ) : (
                              <Link
                                href={link.url}
                                onClick={toggle}
                                className="flex items-center gap-1 text-sm text-gray-700 dark:text-gray-300 hover:text-[#286efb] dark:hover:text-[#286efb] transition-colors"
                              >
                                {link.text}
                                {link.isNew && (
                                  <span className="bg-[#253237]/80 text-white dark:bg-white dark:text-black/80 text-xs px-1.5 py-0.5 rounded-md font-medium">
                                    New
                                  </span>
                                )}
                              </Link>
                            )}
                          </div>
                        ))}
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </Drawer>
  );
};

export default SidebarDrawer;
