import * as React from "react";
import { useState, useMemo } from "react";
import { usePathname } from "next/navigation";
import Link from "next/link";
import {
  Card,
  CardContent,
  ToggleButton,
  ToggleButtonGroup,
  Button,
  Typography,
} from "@mui/material";

type PriceInfo = {
  period: string;
  amount: number;
  title: number | string;
};

type PricingPlan = {
  title: string;
  users: string;
  price: {
    monthly: PriceInfo;
    annually: PriceInfo;
  };
  features: string[];
  isBestValue: boolean;
};

type PricingData = {
  [service: string]: {
    [plan: string]: PricingPlan;
  };
};

const CheckIcon = () => (
  <svg
    xmlns="http://www.w3.org/2000/svg"
    enableBackground="new 0 0 24 24"
    height="20px"
    viewBox="0 0 24 24"
    width="20px"
    fill="#5985E1"
  >
    <rect fill="none" height="24" width="24" />
    <path d="M22,5.18L10.59,16.6l-4.24-4.24l1.41-1.41l2.83,2.83l10-10L22,5.18z M19.79,10.22C19.92,10.79,20,11.39,20,12 c0,4.42-3.58,8-8,8s-8-3.58-8-8c0-4.42,3.58-8,8-8c1.58,0,3.04,0.46,4.28,1.25l1.44-1.44C16.1,2.67,14.13,2,12,2C6.48,2,2,6.48,2,12 c0,5.52,4.48,10,10,10s10-4.48,10-10c0-1.19-0.22-2.33-0.6-3.39L19.79,10.22z" />
  </svg>
);

const pricingData: PricingData = {
  hr: {
    Starter: {
      title: "Starter",
      users: "Unlimited Users",
      price: {
        monthly: { period: "mo", amount: 49, title: 49 },
        annually: { period: "an", amount: 294, title: "24.50" },
      },
      features: [
        "Full HR platform",
        "Recruiting / ATS",
        "Time off Management",
        "HR Analytics",
        "Time and Attendance Tracking",
        "Performance Management",
        "New Hire Onboarding",
        "Job Offers",
        "Surveys and Training",
        "24/7 Customer Support",
      ],
      isBestValue: false,
    },
    Private: {
      title: "Private",
      users: "Unlimited Users",
      price: {
        monthly: { period: "mo", amount: 20, title: 20 },
        annually: { period: "an", amount: 144, title: "12" },
      },
      features: [
        "Full HR platform",
        "Recruiting / ATS",
        "Time off Management",
        "HR Analytics",
        "Time and Attendance Tracking",
        "Performance Management",
        "New Hire Onboarding",
        "Job Offers",
        "Surveys and Training",
        "24/7 Customer Support",
      ],
      isBestValue: false,
    },
    business: {
      title: "Business",
      users: "Unlimited Users",
      price: {
        monthly: { period: "mo", amount: 30, title: 30 },
        annually: { period: "an", amount: 216, title: "18" },
      },
      features: [
        "Full HR platform",
        "Recruiting / ATS",
        "Time off Management",
        "HR Analytics",
        "Time and Attendance Tracking",
        "Performance Management",
        "New Hire Onboarding",
        "Job Offers",
        "Surveys and Training",
        "24/7 Customer Support",
      ],
      isBestValue: false,
    },
    enterprise: {
      title: "Enterprise",
      users: "Unlimited Users",
      price: {
        monthly: { period: "mo", amount: 40, title: 40 },
        annually: { period: "an", amount: 288, title: "24" },
      },
      features: [
        "Unlimited Users & Previous Plans Features",
        "Access to all Clikkle Products",
        "Customizable Reports and Analytics",
        "Real Time Notifications and Alerts",
        "Document Management System",
        "Mobile App Access",
        "Automated Payroll Processing",
        "Employee Self-Service Portal",
        "AI Powered Candidate Matching",
        "Integrated Video Interviewing",
        "Project Management Tools",
      ],
      isBestValue: true,
    },
  },
  files: {
    free: {
      title: "Free",
      users: "Free service",
      price: {
        monthly: { period: "mo", amount: 0, title: "0" },
        annually: { period: "an", amount: 0, title: "0" },
      },
      features: [
        "Send files and folders up to 2 GB per transfer*",
        "Unlimited transfers",
      ],
      isBestValue: false,
    },
    pro: {
      title: "Pro",
      users: "1 user",
      price: {
        monthly: { period: "mo", amount: 20, title: 20 },
        annually: { period: "an", amount: 120, title: 10 },
      },
      features: [
        "Send and receive files/folders up to 250 GB per transfer",
        "Unlimited transfers",
        "250 GB of storage",
        "Add custom logos, backgrounds and promotional messages",
        "Outlook Add-in",
      ],
      isBestValue: true,
    },
    team: {
      title: "Team",
      users: "10 users",
      price: {
        monthly: { period: "mo", amount: 28, title: 28 },
        annually: { period: "an", amount: 168, title: "14" },
      },
      features: [
        "Send and receive files/folders up to 500 GB per transfer",
        "Unlimited transfers",
        "1 TB of storage",
        "Add custom logos, backgrounds and promotional messages",
        "Outlook Add-in",
      ],
      isBestValue: false,
    },
    enterprise: {
      title: "Enterprise",
      users: "+ 10 users",
      price: {
        monthly: { period: "mo", amount: 0, title: "$40 " },
        annually: { period: "an", amount: 0, title: "$20" },
      },
      features: [
        "Send and receive files/folders with customized size limit",
        "Unlimited transfers",
        "5 TB of storage",
        "Add custom logos, backgrounds and promotional messages",
        "Outlook Add-in",
        "Single sign-on (SSO)",
        "CNAME Record",
        "Carbon Footprint",
      ],
      isBestValue: false,
    },
  },
  chat: {
    free: {
      title: "Free",
      users: "Forever",
      price: {
        monthly: { period: "mo", amount: 0, title: 0 },
        annually: { period: "an", amount: 0, title: 0 },
      },
      features: [
        "1 Seat",
        "0 Agents",
        "0 Depts",
        "1 Websites",
        "Unlimited Chats",
        "30 day Chat History",
      ],
      isBestValue: false,
    },
    starter: {
      title: "Starter",
      users: "Starts from",
      price: {
        monthly: { period: "mo", amount: 25, title: 25 },
        annually: { period: "an", amount: 150, title: "12.50" },
      },
      features: [
        "1 Seat",
        "1 Agents",
        "0 Depts",
        "Unlimited Websites",
        "Unlimited Chats",
        "30 day Chat History",
      ],
      isBestValue: false,
    },
    basic: {
      title: "Basic",
      users: "Starts from",
      price: {
        monthly: { period: "mo", amount: 50, title: 50 },
        annually: { period: "an", amount: 300, title: "25" },
      },
      features: [
        "10 Seats + $8 per additional seat",
        "Unlimited Agents",
        "0 Depts",
        "Unlimited Websites",
        "Unlimited Chats",
        "5 Year Chat History",
      ],
      isBestValue: true,
    },
    enterprise: {
      title: "Enterprise",
      users: "Starts from",
      price: {
        monthly: { period: "mo", amount: 79, title: 79 },
        annually: { period: "an", amount: 594, title: "49.50" },
      },
      features: [
        "10 Seats + $8 per additional seat",
        "Unlimited Agents",
        "Unlimited Depts",
        "Unlimited Websites",
        "Unlimited Chats",
        "15 Year Chat History",
      ],
      isBestValue: false,
    },
  },
  esign: {
    free: {
      title: "Free",
      users: "Free service",
      price: {
        monthly: { period: "mo", amount: 0, title: 0 },
        annually: { period: "an", amount: 0, title: 0 },
      },
      features: [
        "1 Documents per month",
        "1 Template",
        "2 Signer Authentication Credits",
      ],
      isBestValue: false,
    },
    basic: {
      title: "Basic",
      users: "Small teams",
      price: {
        monthly: { period: "mo", amount: 25, title: "25" },
        annually: { period: "an", amount: 150, title: "12.50" },
      },
      features: [
        "5 Documents",
        "3 Templates",
        "LTV Enabled",
        "10 Signer Authentication Credits",
        "App Integrations",
        "Bulk Document Sending",
        "Basic Support",
      ],
      isBestValue: false,
    },
    professional: {
      title: "Professional",
      users: "1-5 Users",
      price: {
        monthly: { period: "mo", amount: 49.99, title: "49.99" },
        annually: { period: "an", amount: 294, title: "24.50" },
      },
      features: [
        "10 Documents",
        "10 Templates",
        "LTV Enabled",
        "20 Signer Authentication Credits",
        "5 Users",
        "App Integrations",
        "Bulk Document Sending",
        "Premium Support",
        "In-Person Signing",
      ],
      isBestValue: true,
    },
    professionalPlus: {
      title: "Professional Plus",
      users: "1-15 Users",
      price: {
        monthly: { period: "mo", amount: 99.99, title: "99.99" },
        annually: { period: "an", amount: 599.94, title: "49.99" },
      },
      features: [
        "Unlimited Documents",
        "25 Templates",
        "LTV enabled",
        "40 Signer Authentication Credits",
        "15 Users",
        "App Integrations",
        "Bulk Document Sending",
        "Premium Support",
        "In-Person Signing",
        "Custom Branding",
      ],
      isBestValue: false,
    },
  },
  projects: {
    free: {
      title: "Free",
      users: "Task management",
      price: {
        monthly: { period: "mo", amount: 0, title: 0 },
        annually: { period: "an", amount: 0, title: 0 },
      },
      features: [
        "Unlimited spaces, boards, items and views",
        "Member and admin user roles",
        "6 fields per board",
        "Summary row",
        "5GB storage",
        "7 days activity log",
        "Google & Apple SSO",
      ],
      isBestValue: false,
    },
    Standard: {
      title: "Standard",
      users: "Project management",
      price: {
        monthly: { period: "mo", amount: 7.99, title: "$7.99" },
        annually: { period: "an", amount: 47.88, title: "$3.99" },
      },
      features: [
        "Private boards, views and spaces",
        "Gantt view",
        "12 fields per board",
        "Viewer role",
        "Conditional coloring",
        "Bulk actions",
        "Invite/team permissions",
        "5GB storage",
        "6 months activity log",
        "Board and task templates",
      ],
      isBestValue: false,
    },
    Team: {
      title: "Team",
      users: "Privacy & security",
      price: {
        monthly: { period: "mo", amount: 10.99, title: "$10.99" },
        annually: { period: "an", amount: 71.88, title: "$5.99" },
      },
      features: [
        "Board permissions",
        "Board ownership management",
        "Unlimited fields per board",
        "250GB storage",
        "Unlimited activity log",
        "Customized user profiles",
        "Custom subdomain",
        "Custom SSO",
      ],
      isBestValue: false,
    },
    Platinum: {
      title: "Platinum",
      users: "Productivity Suite",
      price: {
        monthly: { period: "mo", amount: 12.99, title: "$12.99" },
        annually: { period: "an", amount: 77.94, title: "$6.49" },
      },
      features: ["Team Communication", "Time Tracking", "Project Management"],
      isBestValue: true,
    },
  },
  campaigns: {
    lite: {
      title: "Lite",
      users: "Ideal for beginners",
      price: {
        monthly: { period: "mo", amount: 17, title: 17 },
        annually: { period: "an", amount: 102, title: "8.50" },
      },
      features: [
        "Live 1:1 onboarding with experts",
        "Live phone and chat support",
        "Drag-and-drop email editor with AI copy generator",
        "Basic email templates",
        "Automatic social media posts",
        "Basic reporting analytics",
        "1 automation template",
        "Easy contact management",
        "Web signup forms",
        "1 custom segment",
        "Social media ads manager",
        "Event hosting with payments",
        "300+ app integrations",
      ],
      isBestValue: false,
    },
    standard: {
      title: "Standard",
      users: "Great for most businesses",
      price: {
        monthly: { period: "mo", amount: 48, title: 48 },
        annually: { period: "an", amount: 288, title: 24 },
      },
      features: [
        "Live 1:1 onboarding with experts",
        "Live phone and chat support",
        "Email scheduling and subject line testing",
        "Advanced email templates",
        "Social media post scheduling",
        "Advanced reporting (opens/clicks)",
        "3 automation templates with AI builder",
        "Automatic resend to non-openers",
        "10 custom segments",
        "Facebook lookalike targeting",
        "Google Ad Manager",
        "SEO recommendations",
        "Event hosting with payments",
        "300+ app integrations",
      ],
      isBestValue: true,
    },
    premium: {
      title: "Premium",
      users: "Ideal for pros",
      price: {
        monthly: { period: "mo", amount: 109, title: 109 },
        annually: { period: "an", amount: 654, title: "54.50" },
      },
      features: [
        "Priority onboarding by dedicated team",
        "Dedicated priority support",
        "Dynamic email content",
        "All Standard email features",
        "Advanced social media tools",
        "Comprehensive reporting",
        "Unlimited automation templates",
        "Custom automations",
        "Ecommerce templates",
        "Auto-send welcome emails",
        "Unlimited custom segments",
        "Advanced ad targeting",
        "Premium SEO tools",
        "Event hosting with payments",
        "300+ app integrations",
      ],
      isBestValue: false,
    },
  },
};

interface PricingCardProps {
  plan: PricingPlan;
  period: "monthly" | "annually";
  theme: string;
  servicePath: string;
}

const PricingCard: React.FC<PricingCardProps> = ({
  plan,
  period,
  theme,
  servicePath,
}) => {
  return (
    <Card
      variant="outlined"
      className={`flex-1 ${plan.isBestValue ? "relative" : ""}`}
      sx={{
        width: 300,
        minHeight: "550px",
        display: "flex",
        flexDirection: "column",
        justifyContent: "space-between",
      }}
    >
      {plan.isBestValue && (
        <div className="text-center py-2 font-medium text-white bg-blue-500 w-full absolute top-0">
          BEST VALUE
        </div>
      )}

      <CardContent className="flex justify-center flex-col items-center">
        <Typography
          sx={{
            fontWeight: "bold",
            margin: "25px 25px 25px 25px",
            fontSize: 25,
          }}
          className="pt-6 md:pt-6 pb-1 text-gray-900 dark:text-white"
          variant="h5"
          component="div"
        >
          {plan.title.includes("+") ? (
            <>
              {plan.title.split("+")[0]}
              <span className="text-blue-500">+</span>
            </>
          ) : (
            plan.title
          )}
        </Typography>

        <Typography className="py-2 dark:text-gray-600" variant="body2">
          {plan.users}
        </Typography>

        <div className="flex py-6 justify-center text-gray-900 items-center w-full dark:text-white">
          <span className="text-3xl">
            {typeof plan.price[period].title === "number" ||
            (typeof plan.price[period].title === "string" &&
              !plan.price[period].title.toString().startsWith("$"))
              ? "$"
              : ""}
            {plan.price[period].title}
          </span>
          {plan.price[period].title !== "Free" &&
            plan.price[period].title !== "Custom" && (
              <span className="mt-3"> &nbsp; per month</span>
            )}
        </div>
        <Button variant="contained" size="small" className="cursor-pointer w-full pt-4">
          <Link
            href={`https://hub.clikkle.com/login?referralApp=Clikkle%20Website&&?referrer=https://www.clikkle.com/${servicePath}&&redirectto=https://${servicePath}.clikkle.com?amount=${plan.price[period].amount}&period=${plan.price[period].period}&theme=${theme}`}
          >
            <span className="text-md p-1">Try for free</span>
          </Link>
        </Button>

        <div className="text-left w-full py-10">
          <div className="text-left font-medium text-gray-900 dark:text-white mb-2">
            {plan.features.length > 1 ? "All Features Included" : "Features"}
          </div>

          <div className="flex flex-col text-sm text-gray-700 dark:text-white">
            {plan.features.map((feature: string, index: number) => (
              <div key={index} className="flex items-start my-1">
                <span className="mt-1 mr-2 flex-shrink-0">
                  <CheckIcon />
                </span>
                <span className="text-left">{feature}</span>
              </div>
            ))}
          </div>
        </div>

        <div className="py-10 w-10 h-20"></div>
      </CardContent>
    </Card>
  );
};

interface DynamicPricingCardProps {
  theme?: "light" | "dark";
}

const DynamicPricingCard: React.FC<DynamicPricingCardProps> = ({
  theme = "light",
}) => {
  const [alignment, setAlignment] = useState<"monthly" | "annually">("monthly");
  const pathname = usePathname();

  const servicePath = useMemo(() => {
    const path = pathname.split("/")[1] || "hr";
    return path;
  }, [pathname]);

  const servicePlans = useMemo(() => {
    return pricingData[servicePath] || pricingData.hr;
  }, [servicePath]);

  const plansArray = useMemo(() => {
    return Object.values(servicePlans);
  }, [servicePlans]);

  const handleChange = (
    _event: React.MouseEvent<HTMLElement>,
    newAlignment: "monthly" | "annually"
  ) => {
    if (newAlignment !== null) {
      setAlignment(newAlignment);
    }
  };

  const borderStyle = {
    marginTop: "18px",
    width: "18px",
    height: "18px",
    borderTop: "2px solid blue",
    borderRight: "2px solid blue",
    borderTopRightRadius: "50px",
  };

  return (
    <div className="flex flex-col justify-center items-center">
      <div className="flex flex-row">
        <div className="text-center m-2 text-blue-800">Save Up to 50%</div>
        <div style={borderStyle}></div>
      </div>

      <ToggleButtonGroup
        color="primary"
        value={alignment}
        exclusive
        onChange={handleChange}
        aria-label="Billing Period"
      >
        <ToggleButton
          sx={{
            px: {
              xs: 2,
              sm: 3,
            },
            fontSize: {
              xs: 14,
              sm: 18,
            },
            borderBottomLeftRadius: 50,
            borderTopLeftRadius: 50,
          }}
          size="medium"
          value="monthly"
        >
          Bill Monthly
        </ToggleButton>
        <ToggleButton
          sx={{
            px: {
              xs: 2,
              sm: 3,
            },
            fontSize: {
              xs: 14,
              sm: 18,
            },
            borderBottomRightRadius: 50,
            borderTopRightRadius: 50,
          }}
          size="medium"
          value="annually"
        >
          Bill Annually
        </ToggleButton>
      </ToggleButtonGroup>

      <div className="flex md:flex-row flex-col my-4 py-4 gap-10 sm:gap-2">
        {plansArray.map((plan, index) => (
          <PricingCard
            key={index}
            plan={plan}
            period={alignment}
            theme={theme}
            servicePath={servicePath}
          />
        ))}
      </div>
    </div>
  );
};

export default DynamicPricingCard;
