import { Button, Typography } from "@mui/material";
import React from "react";
import Link from "next/link";
import Image from "next/image";
import KeyboardArrowDownOutlinedIcon from "@mui/icons-material/KeyboardArrowDownOutlined";
import { useRef } from "react";

export type HomeFeatureProps = {
  illistrationUrl: string;
  name: string;
  appLogoUrl: string;
  description: string;
  freetrialLink: string;
  learnMoreLink: string;
};

export const features: HomeFeatureProps[] = [
  {
    name: "Clikkle Campaigns",
    description:
      "Experience Clikkle's power! Data driven campaigns for real results, from audience collection to optimization. Elevate your brand with out digital marketing expertise",
    illistrationUrl:
      "https://cdn.clikkle.com/images/campaigns/illustration/campaigns1.png",
    appLogoUrl:
      "https://cdn.clikkle.com/images/campaigns/logo/2025/clikkle%20campaigns.png",
    freetrialLink: "https://campaigns.clikkle.com",
    learnMoreLink: "/campaigns",
  },
  {
    name: "Clikkle Host",
    description:
      "Choose Clikkle's trusted, scalable hosting for high-performance webstites. Our optimized infrastructure guarentees uptime, supported by expert assistance.",
    illistrationUrl:
      "https://cdn.clikkle.com/images/host/illustration/host4.png",
    appLogoUrl: "https://cdn.clikkle.com/images/host/logo/2023/host.png",
    freetrialLink: "https://host.clikkle.com",
    learnMoreLink: "/host",
  },
  {
    name: "Clikkle Files",
    description:
      "Welcome to Clikkle, the secure cloud file storage service. Enjoy easy file management, flexible sharing, and advanced encryption.",
    illistrationUrl:
      "https://cdn.clikkle.com/images/files/illustration/files1.png",
    appLogoUrl:
      "https://cdn.clikkle.com/images/files/logo/2025/clikkle%20files.png",
    freetrialLink: "https://files.clikkle.com",
    learnMoreLink: "/files",
  },
  {
    name: "Clikkle eSign",
    description:
      " Easily send, sign, and manage agreements securely with real-time collaboration and mobile access.",
    illistrationUrl:
      "https://cdn.clikkle.com/images/e-sign/illustration/e-sign3.png",
    appLogoUrl:
      "https://cdn.clikkle.com/images/e-sign/logo/2025/clikkle%20esign.png",
    freetrialLink: "https://esign.clikkle.com",
    learnMoreLink: "/esign",
  },
  {
    name: "Clikkle Mail",
    description:
      "C-Mail: Secure, reliable web service for email-like communication. Send/receive messages, attachments and organize your inbox easily.",
    illistrationUrl:
      "https://cdn.clikkle.com/images/cmail/illustration/cmail1.png",
    appLogoUrl: "https://cdn.clikkle.com/images/cmail/logo/2023/cmail.png",
    freetrialLink: "https://mail.clikkle.com",
    learnMoreLink: "/mail",
  },
  {
    name: "Clikkle Projects",
    description:
      "Unleash the AI power in project mangement. Intelligent task tracking, advanced collaboration and customizable workflows for unparalled productivity.",
    illistrationUrl:
      "https://cdn.clikkle.com/images/projects/illustration/projects1.png",
    appLogoUrl:
      "https://cdn.clikkle.com/images/projects/logo/2025/clikkle%20projects.png",
    freetrialLink: "https://projects.clikkle.com",
    learnMoreLink: "/projects",
  },
  {
    name: "Clikkle HR",
    description:
      "Discover our comprehensive HR platform, offering employee management, payroll, performance tracking, and more. Empower your workforce today!",
    illistrationUrl: "/illustrations/hr.png",
    appLogoUrl: "https://cdn.clikkle.com/images/hr/logo/2025/clikkle%20hr.png",
    freetrialLink: "https://hr.clikkle.com",
    learnMoreLink: "/hr",
  },

  {
    name: "Clikkle Chat",
    appLogoUrl:
      "https://cdn.clikkle.com/images/chat/logo/2025/clikkle%20chat.png",
    description:
      "Effortlessly automate routine tasks with our visual automation builder. Create robust automations effortlessly using no-code building blocks like bots, triggers, conditions, and rules.",
    freetrialLink: "https://chat.clikkle.com",
    learnMoreLink: "/chat",
    illistrationUrl:
      "https://cdn.clikkle.com/images/chat/Clikkle%20Chat%20page/pexels-thisisengineering-3861969.jpg",
  },
  {
    name: "Clikkle Crew",
    appLogoUrl: "https://cdn.clikkle.com/images/crew/logo/2023/crew.png",
    description:
      "With live captions powered by Clikkle speech recognition technology, Clikkle Crew makes meetings more accessible. For non-native speakers, hearing impaired participants, or just noisy coffee shops, live captions make it easy for everyone to follow along (available in English only).",
    freetrialLink: "https://crew.clikkle.com",
    learnMoreLink: "/crew",
    illistrationUrl:
      "https://cdn.clikkle.com/images/crew/Clikkle%20crew%20LANDING%20PAGE%20UI/IMG_2738.JPG",
  },
];

const HomeFeature: React.FC<{ feature: HomeFeatureProps; left: boolean }> = ({
  feature,
  left,
}) => {
  const containerRef = useRef<HTMLDivElement | null>(null);


  return (
    <div
      ref={containerRef}
      id={feature.name.replace(/\s+/g, "-").toLowerCase()}
    >
      <h6 className="text-lg font-[500] w-full px-10 pt-1 mb-20  sm:m-0 text-center text-gray-800 dark:text-gray-300 sm:pt-12 lg:pt-20">
        See what you can do with {feature.name}
        <div
          className="flex items-center justify-center mt-1 animate-bounce"
          
        >
          <KeyboardArrowDownOutlinedIcon />
        </div>
      </h6>
      <div
        className={`w-full sm:mt-14 flex flex-col sm:flex-row items-center justify-between gap-16 p-2 ${
          left ? "sm:flex-row" : "sm:flex-row-reverse"
        }`}
      >
        <Image
          src={feature.illistrationUrl}
          className="h-72 w-auto sm:h-[calc(100vh-300px)] sm:w-auto ml-0 sm:ml-10"
          alt={`Illustration of ${feature.name}`}
          width={600}
          height={400}
        />
        <div className="max-w-[850px] w-[80%] flex-grow  m-3 sm:p-16 sm:m-8">
          <div className="flex items-center justify-center gap-4 sm:justify-start">
            <Image
              src={feature.appLogoUrl}
              alt={feature.name}
              className="h-[40px] w-[40px]"
              width={40}
              height={40}
            />
            <Typography
              className="hidden font-light text-center text-gray-800 sm:flex sm:text-left dark:text-gray-300"
              style={{ fontSize: "2.0rem" }}
            >
              {feature.name}
            </Typography>
            <Typography
              className="font-normal text-center text-gray-800 sm:hidden sm:text-left dark:text-gray-300 sm:mt-0"
              style={{ fontSize: "1.4rem" }}
            >
              {feature.name}
            </Typography>
          </div>

          <br className="hidden sm:block" />
          <Typography
            className="hidden font-light text-center text-gray-800 sm:flex sm:text-left dark:text-gray-300"
            style={{ fontSize: "1.2rem" }}
          >
            {feature.description}
          </Typography>

          <Typography
            className="font-normal text-center text-gray-800 sm:hidden sm:text-left dark:text-gray-300 sm:mt-0"
            style={{ fontSize: "0.8rem" }}
          >
            {feature.description}
          </Typography>

          <br className="hidden sm:block" />

          <div className="flex items-center justify-center gap-4 sm:justify-start sm:mb-0">
            {feature.freetrialLink.startsWith("http") ? (
              <Link
                href={feature.freetrialLink}
                target="_blank"
                rel="noopener noreferrer"
                style={{ textDecoration: "none" }}
              >
                <Button size="small" variant="contained">
                  Start Free Trial
                </Button>
              </Link>
            ) : (
              <Link
                href={feature.freetrialLink}
                style={{ textDecoration: "none" }}
              >
                <Button size="small" variant="contained">
                  Start Free Trial
                </Button>
              </Link>
            )}
            <Link href={feature.learnMoreLink} style={{ textDecoration: "none" }}>
              <Button size="small" variant="outlined">
                Learn More
              </Button>
            </Link>
          </div>
        </div>
      </div>
    </div>
  );
};

export default HomeFeature;
