'use client'

import { ArrowRightIcon, Sparkles } from "lucide-react";
import React, { useState } from "react";
import Link from "next/link";
import { appssection } from "../constants/products";
import { iconMap } from "../utils/iconMap";
import { useTheme } from "../atoms/theme";
import { motion } from "framer-motion";

interface AppsSectionProps {
  headingPrefix?: string;
}

function AppsSection({ headingPrefix }: AppsSectionProps) {
  const { theme } = useTheme();
  const [hoveredApp, setHoveredApp] = useState<number | null>(null);

  // Determine if this is being called from a product page or the homepage
  const isProductPage = !!headingPrefix;

  return (
    <section className="w-full py-16 md:py-24 bg-white dark:bg-transparent relative overflow-hidden px-6 md:px-0">
      {/* Background gradient effect */}
      <div className="absolute inset-0 bg-transparent via-gray-50/50 dark:bg-transparent pointer-events-none" />
      
      <div className="container max-w-[1233px] mx-auto relative z-10">
        {/* Header Section */}
        <div className="text-center mb-12 md:mb-16">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.5 }}
            className="inline-flex items-center gap-2 bg-gray-100 dark:bg-[#16171B] px-4 py-2 rounded-full mb-6"
          >
            <Sparkles className="w-4 h-4 text-[#253237] dark:text-white" />
            <span className="text-sm font-medium text-gray-900 dark:text-white">
              {isProductPage ? "Integration Hub" : "Powerful Tools"}
            </span>
          </motion.div>

          <motion.h2
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.5, delay: 0.1 }}
            className="text-3xl md:text-5xl font-bold text-gray-900 dark:text-white mb-4 tracking-tight"
          >
            {isProductPage ? (
              <>
                {headingPrefix} <span className="text-[#253237] dark:text-white"></span>
              </>
            ) : (
              <>
                Choose Your <span className="text-[#253237] dark:text-white">Perfect Tool</span>
              </>
            )}
          </motion.h2>

          <motion.p
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.5, delay: 0.2 }}
            className="text-gray-600 dark:text-[#94A3B8] max-w-2xl mx-auto text-base md:text-lg"
          >
            {isProductPage 
              ? "Seamlessly integrate with our entire suite of business tools to maximize productivity and streamline your workflows"
              : "Everything you need to run your business efficiently, all in one powerful suite"
            }
          </motion.p>

          <motion.div
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.5, delay: 0.3 }}
            className="mt-6"
          >
            <Link
              href="/products"
              className="inline-flex items-center gap-2 text-[#253237] dark:text-white hover:gap-3 transition-all duration-300 font-medium group"
            >
              {isProductPage ? "Explore all integrations" : "View all products"}
              <ArrowRightIcon className="w-4 h-4 group-hover:translate-x-1 transition-transform" />
            </Link>
          </motion.div>
        </div>

        {/* Apps Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {appssection.map((app, index) => {
            const IconComponent = iconMap[app.icon];
            const isHovered = hoveredApp === app.id;

            return (
              <motion.div
                key={app.id}
                initial={{ opacity: 0, y: 30 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ duration: 0.5, delay: index * 0.1 }}
                onMouseEnter={() => setHoveredApp(app.id)}
                onMouseLeave={() => setHoveredApp(null)}
                onClick={() => {
                  window.location.replace(app.to + "/signup?theme=" + theme);
                }}
                className="group relative bg-white dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E] rounded-2xl p-6 hover:shadow-xl dark:hover:shadow-2xl dark:hover:shadow-white/5 transition-all duration-300 cursor-pointer overflow-hidden"
              >
                {/* Hover gradient effect */}
                <div className="absolute inset-0 bg-gradient-to-br from-[#253237]/0 via-transparent to-transparent dark:from-white/5 dark:via-transparent dark:to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300 pointer-events-none" />
                
                {/* Content */}
                <div className="relative z-10">
                  {/* Icon Container */}
                  <div className={`inline-flex items-center justify-center w-14 h-14 rounded-xl mb-4 transition-all duration-300 ${
                    isHovered 
                      ? 'bg-[#253237] dark:bg-white scale-110' 
                      : 'bg-gray-100 dark:bg-[#0B0C0E]'
                  }`}>
                    {IconComponent && (
                      <IconComponent className={`w-7 h-7 transition-colors duration-300 ${
                        isHovered 
                          ? 'text-white dark:text-[#253237]' 
                          : 'text-gray-900 dark:text-white'
                      }`} />
                    )}
                  </div>

                  {/* Title */}
                  <h3 className="text-xl font-bold text-gray-900 dark:text-white mb-2 transition-colors">
                    {app.title}
                  </h3>

                  {/* Description */}
                  <p className="text-sm text-gray-600 dark:text-[#94A3B8] leading-relaxed mb-4">
                    {app.description}
                  </p>

                  {/* CTA */}
                  <div className="flex items-center gap-2 text-sm font-medium text-[#253237] dark:text-white group-hover:gap-3 transition-all">
                    Connect now
                    <ArrowRightIcon className="w-4 h-4 group-hover:translate-x-1 transition-transform" />
                  </div>
                </div>

                {/* Decorative corner element */}
                <div className="absolute top-0 right-0 w-20 h-20 bg-gradient-to-br from-[#253237]/10 to-transparent dark:from-white/20 dark:to-transparent rounded-bl-full opacity-0 group-hover:opacity-100 transition-opacity duration-300" />
              </motion.div>
            );
          })}
        </div>
      </div>
    </section>
  );
}

export default AppsSection;
