"use client";
import React, { useState } from "react";
import Link from "next/link";
import { apps } from "../constants/products";
import { iconMap } from "../utils/iconMap";
import { ArrowRight, Sparkles } from "lucide-react";

export default function AppsShowcase({ className = "" }) {
  const [hoveredApp, setHoveredApp] = useState<number | null>(null);
  const [hoveredCta, setHoveredCta] = useState(false);

  return (
    <section className={`py-20 md:py-28 ${className} relative z-10 w-full bg-white dark:bg-transparent`}>
      <div className="max-w-7xl mx-auto px-4 md:px-6 w-full">
        {/* Header */}
        <div className="text-center mb-12 md:mb-16">
          <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-gray-100 dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E] mb-6">
            <Sparkles className="w-4 h-4 text-[#253237] dark:text-white" />
            <span className="text-sm font-semibold text-gray-900 dark:text-white">
              Complete Business Suite
            </span>
          </div>
          
          <h2 className="text-3xl md:text-5xl font-bold mb-4 leading-tight">
            <span className="text-gray-900 dark:text-white">
              Everything your business needs,{" "}
            </span>
            <span className="text-[#253237] dark:text-white">
              in one place
            </span>
          </h2>
          
          <p className="text-lg text-gray-600 dark:text-[#94A3B8] max-w-2xl mx-auto">
            Powerful tools that work together seamlessly. Start with what you need, 
            scale as you grow.
          </p>
        </div>

        {/* Desktop Grid */}
        <div className="hidden md:grid md:grid-cols-4 gap-6 mb-12">
          {apps.map((app) => {
            const IconComponent = iconMap[app.icon];
            const isHovered = hoveredApp === app.id;
            
            return (
              <Link
                key={app.id}
                href={app.route}
                onMouseEnter={() => setHoveredApp(app.id)}
                onMouseLeave={() => setHoveredApp(null)}
                className="group relative h-full"
              >
                <div
                  className={`
                    relative overflow-hidden rounded-2xl p-6 h-full
                    transition-all duration-300 flex flex-col
                    bg-white dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E]
                    hover:shadow-xl dark:hover:shadow-2xl dark:hover:shadow-white/5
                    ${isHovered ? "-translate-y-2" : ""}
                  `}
                >
                  {/* Hover gradient effect */}
                  <div className="absolute inset-0 bg-gradient-to-br from-[#253237]/0 via-transparent to-transparent dark:from-white/5 dark:via-transparent dark:to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300 pointer-events-none" />
                  
                  {/* Icon Container */}
                  <div className="flex flex-col items-center text-center space-y-4 flex-1 relative z-10">
                    <div
                      className={`
                        w-20 h-20 rounded-xl flex items-center justify-center
                        transition-all duration-300
                        ${
                          isHovered
                            ? "bg-[#253237] dark:bg-white scale-110"
                            : "bg-gray-100 dark:bg-[#0B0C0E]"
                        }
                      `}
                    >
                      {IconComponent && (
                        <IconComponent
                          className={`
                            w-10 h-10 transition-colors duration-300
                            ${
                              isHovered
                                ? "text-white dark:text-[#253237]"
                                : "text-gray-900 dark:text-white"
                            }
                          `}
                        />
                      )}
                    </div>

                    {/* App Title */}
                    <div className="space-y-1 flex-1 flex flex-col justify-start">
                      <h3 className="text-lg font-bold transition-colors duration-300 text-gray-900 dark:text-white">
                        {app.title}
                      </h3>
                      <p className="text-sm transition-colors duration-300 line-clamp-2 text-gray-600 dark:text-[#94A3B8]">
                        {app.description}
                      </p>
                    </div>

                    {/* Arrow Icon */}
                    <div
                      className={`
                        transition-all duration-300 mt-auto flex items-center gap-2 text-sm font-medium
                        ${
                          isHovered
                            ? "opacity-100 gap-3"
                            : "opacity-70"
                        }
                      `}
                    >
                      <span className="text-[#253237] dark:text-white">Connect now</span>
                      <ArrowRight className="w-4 h-4 text-[#253237] dark:text-white group-hover:translate-x-1 transition-transform" />
                    </div>
                  </div>

                  {/* Decorative corner element */}
                  <div className="absolute top-0 right-0 w-20 h-20 bg-gradient-to-br from-[#253237]/10 to-transparent dark:from-white/20 dark:to-transparent rounded-bl-full opacity-0 group-hover:opacity-100 transition-opacity duration-300" />
                </div>
              </Link>
            );
          })}
          
          {/* CTA Card - fills remaining 2 slots */}
          <Link
            href="/products"
            onMouseEnter={() => setHoveredCta(true)}
            onMouseLeave={() => setHoveredCta(false)}
            className="group relative md:col-span-2 h-full"
          >
            <div
              className={`
                relative overflow-hidden rounded-2xl p-8 h-full
                transition-all duration-300 flex flex-col items-center justify-center text-center
                bg-white dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E]
                hover:shadow-xl dark:hover:shadow-2xl dark:hover:shadow-white/5
                ${hoveredCta ? "-translate-y-2" : ""}
              `}
            >
              {/* Hover gradient effect */}
              <div className="absolute inset-0 bg-gradient-to-br from-[#253237]/0 via-transparent to-transparent dark:from-white/5 dark:via-transparent dark:to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300 pointer-events-none" />
              
              <div className="relative z-10">
                <Sparkles
                  className={`
                    w-12 h-12 mb-4 mx-auto transition-colors duration-300
                    ${hoveredCta ? "text-[#253237] dark:text-white scale-110" : "text-[#253237] dark:text-white"}
                  `}
                />
                <h3 className="text-xl font-bold mb-3 transition-colors duration-300 text-gray-900 dark:text-white">
                  Each app is built to work seamlessly together
                </h3>
                <p className="text-sm mb-6 transition-colors duration-300 max-w-md text-gray-600 dark:text-[#94A3B8]">
                  Pick what you need and scale as you grow
                </p>
                <div className="inline-flex items-center gap-2 px-6 py-3 rounded-xl text-sm font-semibold bg-[#253237] dark:bg-white text-white dark:text-[#253237] group-hover:gap-3 transition-all hover:scale-105">
                  Explore all products
                  <ArrowRight className="w-4 h-4 group-hover:translate-x-1 transition-transform" />
                </div>
              </div>

              {/* Decorative corner element */}
              <div className="absolute top-0 right-0 w-20 h-20 bg-gradient-to-br from-[#253237]/10 to-transparent dark:from-white/20 dark:to-transparent rounded-bl-full opacity-0 group-hover:opacity-100 transition-opacity duration-300" />
            </div>
          </Link>
        </div>

        {/* Mobile 2-Column Grid with 3 Rows */}
        <div className="md:hidden mb-12">
          <div className="grid grid-cols-2 gap-4 mb-4">
            {apps.map((app) => {
              const IconComponent = iconMap[app.icon];
              
              return (
                <Link
                  key={app.id}
                  href={app.route}
                  className="group relative"
                >
                  <div className="relative overflow-hidden rounded-2xl p-4 bg-white dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E] transition-all duration-300 h-full min-h-[140px] flex flex-col hover:-translate-y-1">
                    {/* Hover gradient effect */}
                    <div className="absolute inset-0 bg-gradient-to-br from-[#253237]/0 via-transparent to-transparent dark:from-white/5 dark:via-transparent dark:to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300 pointer-events-none" />
                    
                    <div className="flex flex-col items-center text-center space-y-2 flex-1 relative z-10">
                      <div className="w-12 h-12 rounded-xl bg-gray-100 dark:bg-[#0B0C0E] flex items-center justify-center flex-shrink-0 group-hover:bg-[#253237] group-hover:dark:bg-white transition-all duration-300 group-hover:scale-110">
                        {IconComponent && (
                          <IconComponent className="w-6 h-6 text-gray-900 dark:text-white group-hover:text-white group-hover:dark:text-[#253237] transition-colors duration-300" />
                        )}
                      </div>
                      <div className="flex-1 flex flex-col justify-center">
                        <h3 className="text-xs font-bold text-gray-900 dark:text-white mb-1">
                          {app.title}
                        </h3>
                        <p className="text-[10px] text-gray-600 dark:text-[#94A3B8] line-clamp-2">
                          {app.description}
                        </p>
                      </div>
                    </div>
                    
                    {/* Decorative corner element */}
                    <div className="absolute top-0 right-0 w-12 h-12 bg-gradient-to-br from-[#253237]/10 to-transparent dark:from-white/20 dark:to-transparent rounded-bl-full opacity-0 group-hover:opacity-100 transition-opacity duration-300" />
                  </div>
                </Link>
              );
            })}
          </div>
          
          {/* CTA Card for Mobile - Full Width Below Grid */}
          <Link href="/products" className="block">
            <div className="relative overflow-hidden rounded-2xl p-6 bg-[#253237] dark:bg-white border border-[#253237] dark:border-white transition-all duration-300 text-center hover:-translate-y-1">
              {/* Decorative element */}
              <div className="absolute top-0 right-0 w-24 h-24 bg-white/10 dark:bg-[#253237]/10 rounded-full -mr-12 -mt-12" />
              
              <div className="relative z-10">
                <Sparkles className="w-8 h-8 mb-3 mx-auto text-white dark:text-[#253237]" />
                <h3 className="text-base font-bold text-white dark:text-[#253237] mb-2">
                  Work seamlessly together
                </h3>
                <p className="text-xs text-white/80 dark:text-[#253237]/80 mb-3">
                  Pick what you need and scale as you grow
                </p>
                <div className="inline-flex items-center gap-2 text-sm font-semibold text-white dark:text-[#253237]">
                  <span>Explore all products</span>
                  <ArrowRight className="w-4 h-4" />
                </div>
              </div>
            </div>
          </Link>
        </div>
      </div>

      {/* Background Decoration */}
      <div className="absolute inset-0 -z-10 overflow-hidden pointer-events-none">
        <div className="absolute top-0 left-1/4 w-96 h-96 bg-gray-100/50 dark:bg-transparent rounded-full blur-3xl" />
        <div className="absolute bottom-0 right-1/4 w-96 h-96 bg-gray-100/50 dark:bg-transparent rounded-full blur-3xl" />
      </div>
    </section>
  );
}
