"use client";
import React, { useState } from "react";
import { usePathname } from "next/navigation";
import Button from "@mui/material/Button";
import Popover from "@mui/material/Popover";
import Apps from "@mui/icons-material/Apps";
import IconButton from "@mui/material/IconButton";
import Link from "next/link";
import { 
  MessageSquare, 
  FolderClosed, 
  FileSignature, 
  Users, 
  Kanban,
  Megaphone,
  ArrowRight,
  Sparkles
} from "lucide-react";

type App = {
  logo: string;
  name: string;
  link: string;
  url?: string;
  description: string;
  icon: string;
};

export const appList: App[] = [
  {
    logo: "https://cdn.clikkle.com/images/campaigns/logo/2025/clikkle%20campaigns.png",
    name: "Campaigns",
    link: "/campaigns",
    description: "Customer engagement",
    icon: "Megaphone",
  },
  {
    logo: "https://cdn.clikkle.com/images/files/logo/2025/clikkle%20files.png",
    name: "Files",
    link: "/files",
    url: "https://files.clikkle.com",
    description: "Cloud storage",
    icon: "FolderClosed",
  },
  {
    logo: "https://cdn.clikkle.com/images/e-sign/logo/2025/clikkle%20esign.png",
    name: "eSign",
    link: "/esign",
    url: "https://esign.clikkle.com",
    description: "Digital signatures",
    icon: "FileSignature",
  },
  {
    logo: "https://cdn.clikkle.com/images/projects/logo/2025/clikkle%20projects.png",
    name: "Projects",
    link: "/projects",
    url: "https://projects.clikkle.com",
    description: "Task management",
    icon: "Kanban",
  },
  {
    logo: "https://cdn.clikkle.com/images/hr/logo/2025/clikkle%20hr.png",
    name: "HR",
    link: "/hr",
    url: "https://hr.clikkle.com",
    description: "HR management",
    icon: "Users",
  },
  {
    logo: "https://cdn.clikkle.com/images/chat/logo/2025/clikkle%20chat.png",
    name: "Chat",
    link: "/chat",
    description: "Team communication",
    icon: "MessageSquare",
  },
];

const iconMap: { [key: string]: React.ReactNode } = {
  MessageSquare: <MessageSquare className="w-5 h-5" />,
  FolderClosed: <FolderClosed className="w-5 h-5" />,
  FileSignature: <FileSignature className="w-5 h-5" />,
  Users: <Users className="w-5 h-5" />,
  Kanban: <Kanban className="w-5 h-5" />,
  Megaphone: <Megaphone className="w-5 h-5" />,
};

const AppListToggler: React.FC = () => {
  const [anchorEl, setAnchorEl] = useState<HTMLButtonElement | null>(null);
  const [hoveredApp, setHoveredApp] = useState<string | null>(null);
  const pathname = usePathname();

  const isCrewPage = pathname === "/crew";

  const redirectMap: { [key: string]: string } = {
    "/hr": "https://hr.clikkle.com",
    "/host": "https://host.clikkle.com",
    "/mail": "https://mail.clikkle.com",
    "/chat": "https://chat.clikkle.com",
    "/crew": "https://crew.clikkle.com",
    "/campaigns": "https://campaigns.clikkle.com",
    "/files": "https://files.clikkle.com",
    "/projects": "https://projects.clikkle.com",
    "/esign": "https://esign.clikkle.com",
  };

  const handleButtonClick = (event: React.MouseEvent<HTMLButtonElement>) => {
    setAnchorEl(event.currentTarget);
  };

  const handleClose = () => {
    setAnchorEl(null);
    setHoveredApp(null);
  };

  const handleLinkClick = () => {
    handleClose();
  };

  const currentPath = pathname;
  const redirectUrl =
    redirectMap[currentPath] ||
    "https://hub.clikkle.com/login?referralApp=Clikkle%20Website&&";

  return (
    <div className="sm:flex items-center gap-2">
      <IconButton onClick={handleButtonClick}>
        <Apps />
      </IconButton>
      {!isCrewPage && (
        <Button variant="contained" component="a" href={redirectUrl}>
          Sign In
        </Button>
      )}

      <Popover
        anchorEl={anchorEl}
        open={Boolean(anchorEl)}
        onClose={handleClose}
        anchorOrigin={{
          vertical: "bottom",
          horizontal: "left",
        }}
        transformOrigin={{
          vertical: "top",
          horizontal: "left",
        }}
        slotProps={{
          paper: {
            sx: {
              mt: 1,
              borderRadius: 2,
              boxShadow: "0 10px 40px rgba(0,0,0,0.1)",
              overflow: "visible",
            },
          },
        }}
      >
        <div className="w-[480px] bg-white dark:bg-[#1a1924]">
          {/* Header */}
          <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-800">
            <div className="flex items-center justify-between">
              <div>
                <h3 className="text-lg font-bold text-gray-900 dark:text-white flex items-center gap-2">
                  <Sparkles className="w-5 h-5 text-blue-500" />
                  Clikkle Apps
                </h3>
                <p className="text-sm text-gray-600 dark:text-gray-400 mt-0.5">
                  Choose an app to get started
                </p>
              </div>
            </div>
          </div>

          {/* Apps Grid */}
          <div className="p-4">
            <div className="grid grid-cols-2 gap-3">
              {appList.map((app) => (
                <Link
                  key={app.name}
                  href={app.link}
                  onClick={handleLinkClick}
                  onMouseEnter={() => setHoveredApp(app.name)}
                  onMouseLeave={() => setHoveredApp(null)}
                  className="group relative"
                >
                  <div
                    className={`
                      relative overflow-hidden rounded-xl border-2 p-4 
                      transition-all duration-300 cursor-pointer
                      ${
                        hoveredApp === app.name
                          ? "border-blue-500 shadow-lg shadow-blue-500/20 -translate-y-1"
                          : "border-gray-200 dark:border-gray-800 hover:border-gray-300 dark:hover:border-gray-700"
                      }
                      bg-white dark:bg-white/5
                    `}
                  >
                    {/* Icon Background */}
                    <div className="flex items-start justify-between mb-3">
                      <div
                        className={`
                          w-12 h-12 rounded-lg flex items-center justify-center
                          transition-all duration-300
                          ${
                            hoveredApp === app.name
                              ? "bg-blue-500 text-white scale-110"
                              : "bg-gray-100 dark:bg-white/10 text-gray-700 dark:text-gray-300"
                          }
                        `}
                      >
                        {iconMap[app.icon]}
                      </div>
                      
                      {/* Arrow Icon */}
                      <div
                        className={`
                          transition-all duration-300
                          ${
                            hoveredApp === app.name
                              ? "opacity-100 translate-x-0"
                              : "opacity-0 -translate-x-2"
                          }
                        `}
                      >
                        <ArrowRight className="w-4 h-4 text-blue-500" />
                      </div>
                    </div>

                    {/* Content */}
                    <div>
                      <h4 className="text-base font-bold text-gray-900 dark:text-white mb-1">
                        {app.name}
                      </h4>
                      <p className="text-sm text-gray-600 dark:text-gray-400 line-clamp-1">
                        {app.description}
                      </p>
                    </div>

                    {/* Hover Gradient */}
                    <div
                      className={`
                        absolute inset-0 bg-gradient-to-br from-blue-500/5 to-purple-500/5 
                        transition-opacity duration-300 pointer-events-none rounded-xl
                        ${hoveredApp === app.name ? "opacity-100" : "opacity-0"}
                      `}
                    />
                  </div>
                </Link>
              ))}
            </div>
          </div>

          {/* Footer */}
          <div className="px-6 py-4 border-t border-gray-200 dark:border-gray-800 bg-gray-50 dark:bg-black/20">
            <Link
              href="/products"
              onClick={handleLinkClick}
              className="flex items-center justify-between text-sm font-medium text-blue-500 hover:text-blue-600 dark:hover:text-blue-400 transition-colors group"
            >
              <span>Explore all products</span>
              <ArrowRight className="w-4 h-4 transition-transform group-hover:translate-x-1" />
            </Link>
          </div>
        </div>
      </Popover>
    </div>
  );
};

export default AppListToggler;
