"use client";

import React, { useState } from "react";
import Image, { StaticImageData } from "next/image";
import { IoIosArrowBack, IoIosArrowForward } from "react-icons/io";

interface Testimonial {
  text: string;
  name: string;
  title: string;
  date?: string;
  logo: string | StaticImageData;
  recommendNote?: string;
}

interface TestimonialSectionProps {
  testimonials: Testimonial[];
  title?: string;
}

const TestimonialSection: React.FC<TestimonialSectionProps> = ({
  testimonials,
  title = "What Our\nClients Are\nSaying",
}) => {
  const [currentTestimonial, setCurrentTestimonial] = useState(0);
  const [touchStart, setTouchStart] = useState(0);
  const [touchEnd, setTouchEnd] = useState(0);

  return (
    <>
      {/* Mobile Feedback Section */}
      <section className="block md:hidden bg-transparent px-4 py-12 md:py-20 transition-colors duration-300">
        <div className="max-w-7xl mx-auto">
          <div className="text-center mb-8">
            <h2 className="text-3xl md:text-4xl font-bold text-[#253237] dark:text-white mb-4">
              {title.split("\n").map((line, i) => (
                <React.Fragment key={i}>
                  {line}
                  {i < title.split("\n").length - 1 && <br />}
                </React.Fragment>
              ))}
            </h2>
          </div>

          <div className="relative max-w-sm mx-auto px-10">
            <div
              className="overflow-hidden"
              onTouchStart={(e) => {
                const touch = e.targetTouches[0];
                if (touch) setTouchStart(touch.clientX);
              }}
              onTouchMove={(e) => {
                const touch = e.targetTouches[0];
                if (touch) setTouchEnd(touch.clientX);
              }}
              onTouchEnd={() => {
                const diff = touchStart - touchEnd;
                if (Math.abs(diff) > 75) {
                  if (diff > 0 && currentTestimonial < testimonials.length - 1) {
                    setCurrentTestimonial((prev) => prev + 1);
                  } else if (diff < 0 && currentTestimonial > 0) {
                    setCurrentTestimonial((prev) => prev - 1);
                  }
                }
                setTouchStart(0);
                setTouchEnd(0);
              }}
            >
              <div
                className="flex transition-transform duration-500 ease-out"
                style={{
                  transform: `translateX(-${currentTestimonial * 100}%)`,
                }}
              >
                {testimonials.map((testimonial, index) => (
                  <div
                    key={index}
                    className="w-full flex-shrink-0 px-1 transition-all duration-1000 opacity-100 translate-y-0"
                  >
                    <div className="bg-white dark:bg-[#1a1a1a] rounded-xl p-4 shadow-lg h-full flex flex-col">
                      {/* Star Rating */}
                      <div className="flex gap-0.5 mb-2">
                        {[...Array(5)].map((_, i) => (
                          <svg
                            key={i}
                            className="w-3 h-3 text-[#253237] dark:text-white"
                            viewBox="0 0 24 24"
                            fill="currentColor"
                          >
                            <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z" />
                          </svg>
                        ))}
                      </div>

                      {/* Feedback */}
                      <p className="text-gray-700 dark:text-gray-300 text-xs leading-relaxed mb-3 flex-grow">
                        {testimonial.text}
                      </p>

                      {/* Author Info */}
                      <div className="flex items-center gap-2 pt-2 border-t border-gray-200 dark:border-gray-700">
                        <div className="w-8 h-8 rounded-full overflow-hidden flex-shrink-0">
                          <Image
                            src={testimonial.logo}
                            alt={testimonial.name}
                            className="w-full h-full object-cover"
                            width={32}
                            height={32}
                          />
                        </div>

                        <div className="flex-1 min-w-0">
                          <p className="font-bold text-xs text-[#253237] dark:text-white truncate">
                            {testimonial.name}
                          </p>
                          <p className="text-[10px] text-gray-600 dark:text-gray-400 truncate">
                            {testimonial.title}
                          </p>
                        </div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>

            {/* Navigation Arrows */}
            <div className="absolute top-1/2 -translate-y-1/2 left-0 right-0 flex justify-between pointer-events-none px-1">
              <button
                onClick={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                  setCurrentTestimonial((prev) => Math.max(0, prev - 1));
                }}
                disabled={currentTestimonial === 0}
                className={`pointer-events-auto w-8 h-8 rounded-full flex items-center justify-center transition-all duration-300 ${
                  currentTestimonial === 0
                    ? "bg-gray-200 dark:bg-gray-800 text-gray-400 dark:text-gray-600 cursor-not-allowed opacity-50"
                    : "bg-[#253237] dark:bg-white text-white dark:text-[#253237] shadow-lg active:scale-95"
                }`}
                aria-label="Previous feedback"
              >
                <IoIosArrowBack size={16} />
              </button>

              <button
                onClick={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                  setCurrentTestimonial((prev) =>
                    Math.min(testimonials.length - 1, prev + 1)
                  );
                }}
                disabled={currentTestimonial >= testimonials.length - 1}
                className={`pointer-events-auto w-8 h-8 rounded-full flex items-center justify-center transition-all duration-300 ${
                  currentTestimonial >= testimonials.length - 1
                    ? "bg-gray-200 dark:bg-gray-800 text-gray-400 dark:text-gray-600 cursor-not-allowed opacity-50"
                    : "bg-[#253237] dark:bg-white text-white dark:text-[#253237] shadow-lg active:scale-95"
                }`}
                aria-label="Next feedback"
              >
                <IoIosArrowForward size={16} />
              </button>
            </div>
          </div>

          {/* Dots Indicator */}
          <div className="flex justify-center gap-2 mt-6">
            {testimonials.map((_, index) => (
              <button
                key={index}
                onClick={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                  setCurrentTestimonial(index);
                }}
                className={`transition-all duration-300 rounded-full ${
                  index === currentTestimonial
                    ? "w-8 h-2 bg-[#253237] dark:bg-white"
                    : "w-2 h-2 bg-gray-300 dark:bg-gray-700"
                }`}
                aria-label={`Go to feedback ${index + 1}`}
              />
            ))}
          </div>
        </div>
      </section>

      {/* Desktop Feedback Section */}
      <section className="hidden md:block bg-transparent px-8 md:px-12 lg:px-20 py-20 md:py-28 transition-colors duration-300">
        <div className="max-w-[1233px] mx-auto">
          <div className="flex flex-col lg:flex-row gap-12 items-start">
            {/* Left Side - Title and Navigation */}
            <div className="lg:w-1/4 flex-shrink-0 transition-all duration-1000 opacity-100 translate-x-0">
              <div className="relative">
                <h2 className="text-4xl md:text-5xl font-bold text-[#253237] dark:text-white mb-8 leading-tight">
                  {title.split("\n").map((line, i) => (
                    <React.Fragment key={i}>
                      {line}
                      {i < title.split("\n").length - 1 && <br />}
                    </React.Fragment>
                  ))}
                </h2>
              </div>

              {/* Navigation Arrows */}
              <div className="flex gap-4 mt-8">
                <button
                  onClick={() => {
                    if (currentTestimonial > 0) {
                      setCurrentTestimonial((prev) => prev - 1);
                    }
                  }}
                  disabled={currentTestimonial === 0}
                  className={`p-4 rounded-full transition-all duration-300 ${
                    currentTestimonial === 0
                      ? "bg-gray-200 dark:bg-gray-800 text-gray-400 dark:text-gray-600 cursor-not-allowed"
                      : "bg-[#253237] dark:bg-white text-white dark:text-[#253237] hover:scale-110"
                  }`}
                  aria-label="Previous testimonials"
                >
                  <IoIosArrowBack size={20} />
                </button>

                <button
                  onClick={() => {
                    if (currentTestimonial < testimonials.length - 1) {
                      setCurrentTestimonial((prev) => prev + 1);
                    }
                  }}
                  disabled={currentTestimonial >= testimonials.length - 3}
                  className={`p-4 rounded-full transition-all duration-300 ${
                    currentTestimonial >= testimonials.length - 3
                      ? "bg-gray-200 dark:bg-gray-800 text-gray-400 dark:text-gray-600 cursor-not-allowed"
                      : "bg-[#253237] dark:bg-white text-white dark:text-[#253237] hover:scale-110"
                  }`}
                  aria-label="Next testimonials"
                >
                  <IoIosArrowForward size={20} />
                </button>
              </div>
            </div>

            {/* Right Side - Testimonial Cards */}
            <div className="lg:w-3/4 overflow-hidden pb-6">
              <div
                className="flex gap-6 transition-transform duration-700 ease-in-out"
                style={{
                  transform: `translateX(-${currentTestimonial * (100 / 3)}%)`,
                }}
              >
                {testimonials.map((testimonial, index) => (
                  <div
                    key={index}
                    className="min-w-[calc(33.333%-16px)] transition-all duration-1000 opacity-100 translate-y-0"
                  >
                    <div className="bg-white dark:bg-[#1a1a1a] rounded-2xl p-6 shadow-lg hover:shadow-xl transition-shadow duration-300 h-full flex flex-col">
                      <div className="flex gap-1 mb-4">
                        {[...Array(5)].map((_, i) => (
                          <svg
                            key={i}
                            className="w-4 h-4 text-[#253237] dark:text-white"
                            viewBox="0 0 24 24"
                            fill="currentColor"
                          >
                            <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z" />
                          </svg>
                        ))}
                      </div>

                      <p className="text-gray-700 dark:text-gray-300 text-sm leading-relaxed mb-6 flex-grow">
                        {testimonial.text}
                      </p>

                      <div className="flex items-center gap-3 border-t border-gray-200 dark:border-gray-700 pt-4">
                        <div className="w-10 h-10 rounded-full overflow-hidden flex-shrink-0">
                          <Image
                            src={testimonial.logo}
                            alt={testimonial.name}
                            className="w-full h-full object-cover"
                            width={40}
                            height={40}
                          />
                        </div>
                        <div>
                          <p className="font-bold text-[#253237] dark:text-white text-sm mb-1">
                            {testimonial.name}
                          </p>
                          <p className="text-xs text-gray-600 dark:text-gray-400">
                            {testimonial.title}
                          </p>
                        </div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>
        </div>
      </section>
    </>
  );
};

export default TestimonialSection;
