'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { CheckCircle2, Circle, Clock } from 'lucide-react'
import Image from 'next/image'

interface Task {
  id: number
  title: string
  assignee: string
  avatarUrl: string
  status: 'todo' | 'inprogress' | 'done'
  color: string
}

const tasksData: Task[] = [
  {
    id: 1,
    title: 'Design new landing page',
    assignee: 'Sarah',
    avatarUrl: 'https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=100&h=100&fit=crop&crop=faces',
    status: 'todo',
    color: 'bg-gray-300 dark:bg-gray-600'
  },
  {
    id: 2,
    title: 'Review customer feedback',
    assignee: 'Mike',
    avatarUrl: 'https://images.unsplash.com/photo-1500648767791-00dcc994a43e?w=100&h=100&fit=crop&crop=faces',
    status: 'todo',
    color: 'bg-gray-200 dark:bg-gray-500'
  },
  {
    id: 3,
    title: 'Update documentation',
    assignee: 'Emma',
    avatarUrl: 'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=100&h=100&fit=crop&crop=faces',
    status: 'todo',
    color: 'bg-gray-400 dark:bg-gray-700'
  }
]

export default function TaskCollaborationAnimationPanel() {
  const [tasks, setTasks] = useState<Task[]>([])
  const [isVisible, setIsVisible] = useState(false)
  const panelRef = useRef<HTMLDivElement>(null)
  const taskIndexRef = useRef(0)

  // Intersection observer
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  // Animation sequence
  useEffect(() => {
    if (!isVisible) return

    const animateTaskFlow = async () => {
      // Reset
      setTasks([])
      taskIndexRef.current = 0
      
      await new Promise(resolve => setTimeout(resolve, 500))

      // Add tasks to "To Do" one by one
      for (let i = 0; i < tasksData.length; i++) {
        setTasks(prev => [...prev, { ...tasksData[i], id: taskIndexRef.current++, status: 'todo' }])
        await new Promise(resolve => setTimeout(resolve, 800))
      }

      await new Promise(resolve => setTimeout(resolve, 1000))

      // Move first task to "In Progress"
      setTasks(prev => prev.map((task, idx) => 
        idx === 0 ? { ...task, status: 'inprogress' } : task
      ))

      await new Promise(resolve => setTimeout(resolve, 1200))

      // Move first task to "Done"
      setTasks(prev => prev.map((task, idx) => 
        idx === 0 ? { ...task, status: 'done' } : task
      ))

      await new Promise(resolve => setTimeout(resolve, 800))

      // Move second task to "In Progress"
      setTasks(prev => prev.map((task, idx) => 
        idx === 1 ? { ...task, status: 'inprogress' } : task
      ))

      await new Promise(resolve => setTimeout(resolve, 1200))

      // Move second task to "Done"
      setTasks(prev => prev.map((task, idx) => 
        idx === 1 ? { ...task, status: 'done' } : task
      ))

      await new Promise(resolve => setTimeout(resolve, 1500))

      // Loop
      animateTaskFlow()
    }

    animateTaskFlow()
  }, [isVisible])

  const getTasksByStatus = (status: 'todo' | 'inprogress' | 'done') => {
    return tasks.filter(task => task.status === status)
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'todo':
        return <Circle className="w-4 h-4 md:w-5 md:h-5 text-gray-400 dark:text-gray-500" />
      case 'inprogress':
        return <Clock className="w-4 h-4 md:w-5 md:h-5 text-[#286efb] dark:text-[#A9C5FF]" />
      case 'done':
        return <CheckCircle2 className="w-4 h-4 md:w-5 md:h-5 text-green-500" />
      default:
        return <Circle className="w-4 h-4 md:w-5 md:h-5 text-gray-400" />
    }
  }

  return (
    <div 
      ref={panelRef}
      className={`w-[310px] md:w-[450px] mx-auto mb-8 md:mb-0 transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Kanban Board */}
      <div className="w-full bg-white dark:bg-[#1a1a1a] backdrop-blur-md rounded-xl md:rounded-2xl shadow-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
        {/* Header */}
        <div className="bg-gray-50 dark:bg-[#1a1a1a] px-3 md:px-5 py-2.5 md:py-4 border-b border-gray-200 dark:border-gray-700">
          <div className="flex items-center justify-between">
            <h3 className="text-gray-700 dark:text-white font-semibold text-xs md:text-base">Task Board</h3>
            <div className="flex items-center gap-1.5 md:gap-2">
              <span className="text-gray-600 dark:text-gray-300 text-[10px] md:text-sm">
                {tasks.filter(t => t.status === 'done').length}/{tasks.length} completed
              </span>
            </div>
          </div>
        </div>

        {/* Kanban Columns */}
        <div className="p-3 md:p-5 grid grid-cols-3 gap-2 md:gap-3 h-[255px] md:h-[340px]">
          {/* To Do Column */}
          <div className="flex flex-col">
            <div className="flex items-center gap-1.5 mb-2 md:mb-3">
              <Circle className="w-3 h-3 md:w-4 md:h-4 text-gray-400 dark:text-gray-500" />
              <h4 className="text-gray-700 dark:text-white font-semibold text-[10px] md:text-sm">To Do</h4>
              <span className="text-gray-500 dark:text-gray-400 text-[9px] md:text-xs ml-auto">
                {getTasksByStatus('todo').length}
              </span>
            </div>
            <div className="space-y-2 flex-1 overflow-hidden">
              <AnimatePresence mode="sync">
                {getTasksByStatus('todo').map((task) => (
                  <motion.div
                    key={`todo-${task.id}`}
                    layout
                    initial={{ opacity: 0, y: -20, scale: 0.8 }}
                    animate={{ opacity: 1, y: 0, scale: 1 }}
                    exit={{ opacity: 0, x: 100, scale: 0.8 }}
                    transition={{
                      type: 'spring',
                      stiffness: 400,
                      damping: 25
                    }}
                    className="bg-gray-50 dark:bg-transparent border border-gray-200 dark:border-gray-700 rounded-lg p-2 md:p-2.5"
                  >
                    <p className="text-gray-700 dark:text-white text-[9px] md:text-xs font-medium mb-1.5 md:mb-2 line-clamp-2">
                      {task.title}
                    </p>
                    <div className="flex items-center gap-1.5">
                      <div className={`w-4 h-4 md:w-5 md:h-5 rounded-full ${task.color} overflow-hidden relative flex-shrink-0`}>
                        <Image
                          src={task.avatarUrl}
                          alt={task.assignee}
                          fill
                          className="object-cover"
                          sizes="20px"
                        />
                      </div>
                      <span className="text-gray-600 dark:text-gray-400 text-[8px] md:text-[10px]">{task.assignee}</span>
                    </div>
                  </motion.div>
                ))}
              </AnimatePresence>
            </div>
          </div>

          {/* In Progress Column */}
          <div className="flex flex-col">
            <div className="flex items-center gap-1.5 mb-2 md:mb-3">
              <Clock className="w-3 h-3 md:w-4 md:h-4 text-[#286efb] dark:text-[#A9C5FF]" />
              <h4 className="text-gray-700 dark:text-white font-semibold text-[10px] md:text-sm">In Progress</h4>
              <span className="text-gray-500 dark:text-gray-400 text-[9px] md:text-xs ml-auto">
                {getTasksByStatus('inprogress').length}
              </span>
            </div>
            <div className="space-y-2 flex-1 overflow-hidden">
              <AnimatePresence mode="sync">
                {getTasksByStatus('inprogress').map((task) => (
                  <motion.div
                    key={`inprogress-${task.id}`}
                    layout
                    initial={{ opacity: 0, x: -100, scale: 0.8 }}
                    animate={{ opacity: 1, x: 0, scale: 1 }}
                    exit={{ opacity: 0, x: 100, scale: 0.8 }}
                    transition={{
                      type: 'spring',
                      stiffness: 400,
                      damping: 25
                    }}
                    className="bg-blue-50 dark:bg-[#286efb]/10 border border-[#286efb]/30 dark:border-[#A9C5FF]/30 rounded-lg p-2 md:p-2.5"
                  >
                    <p className="text-gray-700 dark:text-white text-[9px] md:text-xs font-medium mb-1.5 md:mb-2 line-clamp-2">
                      {task.title}
                    </p>
                    <div className="flex items-center gap-1.5">
                      <div className={`w-4 h-4 md:w-5 md:h-5 rounded-full ${task.color} overflow-hidden relative flex-shrink-0`}>
                        <Image
                          src={task.avatarUrl}
                          alt={task.assignee}
                          fill
                          className="object-cover"
                          sizes="20px"
                        />
                      </div>
                      <span className="text-gray-600 dark:text-gray-400 text-[8px] md:text-[10px]">{task.assignee}</span>
                    </div>
                  </motion.div>
                ))}
              </AnimatePresence>
            </div>
          </div>

          {/* Done Column */}
          <div className="flex flex-col">
            <div className="flex items-center gap-1.5 mb-2 md:mb-3">
              <CheckCircle2 className="w-3 h-3 md:w-4 md:h-4 text-green-500" />
              <h4 className="text-gray-700 dark:text-white font-semibold text-[10px] md:text-sm">Done</h4>
              <span className="text-gray-500 dark:text-gray-400 text-[9px] md:text-xs ml-auto">
                {getTasksByStatus('done').length}
              </span>
            </div>
            <div className="space-y-2 flex-1 overflow-hidden">
              <AnimatePresence mode="sync">
                {getTasksByStatus('done').map((task) => (
                  <motion.div
                    key={`done-${task.id}`}
                    layout
                    initial={{ opacity: 0, x: -100, scale: 0.8 }}
                    animate={{ opacity: 1, x: 0, scale: 1 }}
                    exit={{ opacity: 0, scale: 0.8 }}
                    transition={{
                      type: 'spring',
                      stiffness: 400,
                      damping: 25
                    }}
                    className="bg-green-50 dark:bg-green-900/10 border border-green-500/30 rounded-lg p-2 md:p-2.5"
                  >
                    <p className="text-gray-700 dark:text-white text-[9px] md:text-xs font-medium mb-1.5 md:mb-2 line-clamp-2 opacity-70">
                      {task.title}
                    </p>
                    <div className="flex items-center gap-1.5">
                      <div className={`w-4 h-4 md:w-5 md:h-5 rounded-full ${task.color} overflow-hidden relative flex-shrink-0`}>
                        <Image
                          src={task.avatarUrl}
                          alt={task.assignee}
                          fill
                          className="object-cover"
                          sizes="20px"
                        />
                      </div>
                      <span className="text-gray-600 dark:text-gray-400 text-[8px] md:text-[10px]">{task.assignee}</span>
                    </div>
                  </motion.div>
                ))}
              </AnimatePresence>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
