"use client";

import React, { useState, useRef, useEffect } from "react";
import Image, { StaticImageData } from "next/image";
import { motion, AnimatePresence } from "framer-motion";
import { useTheme } from "../atoms/theme";

export interface TabItem {
  id: string | number;
  label: string;
  icon: React.ReactNode;
  lightImage: StaticImageData | string;
  darkImage: StaticImageData | string;
}

interface TabbedFeatureProps {
  tabs: TabItem[];
  decorator?: React.ReactNode;
  classicGradients?: boolean; // If true, use fixed edge gradients (for many tabs)
}

const TabbedFeature: React.FC<TabbedFeatureProps> = ({ tabs, decorator, classicGradients = false }) => {
  const [activeTab, setActiveTab] = useState(0);
  const { theme } = useTheme();
  const scrollContainerRef = useRef<HTMLDivElement>(null);
  const tabsRef = useRef<(HTMLButtonElement | null)[]>([]);
  const [manualChange, setManualChange] = useState(false);
  const [firstTabLeft, setFirstTabLeft] = useState(0);
  const [lastTabRight, setLastTabRight] = useState(0);


  useEffect(() => {
    if (manualChange) return;

    const interval = setInterval(() => {
      setActiveTab((prev) => (prev + 1) % tabs.length);
    }, 3500);

    return () => clearInterval(interval);
  }, [manualChange, tabs.length]);

  useEffect(() => {
    if (scrollContainerRef.current && tabsRef.current[activeTab]) {
      const container = scrollContainerRef.current;
      const tab = tabsRef.current[activeTab];

      if (tab) {
        const containerRect = container.getBoundingClientRect();
        const tabRect = tab.getBoundingClientRect();
        const scrollLeft =
          container.scrollLeft +
          (tabRect.left - containerRect.left) -
          containerRect.width / 2 +
          tabRect.width / 2;

        container.scrollTo({
          left: scrollLeft,
          behavior: "smooth",
        });
      }
    }
  }, [activeTab]);

  // Calculate positions for gradient overlays
  useEffect(() => {
    const updateGradientPositions = () => {
      // Skip calculation if using classic gradients
      if (classicGradients) {
        setFirstTabLeft(0);
        setLastTabRight(0);
        return;
      }

      if (scrollContainerRef.current && tabsRef.current.length > 0) {
        const container = scrollContainerRef.current;
        const firstTab = tabsRef.current[0];
        const lastTab = tabsRef.current[tabsRef.current.length - 1];

        if (firstTab && lastTab && container.parentElement) {
          const parentRect = container.parentElement.getBoundingClientRect();
          const firstTabRect = firstTab.getBoundingClientRect();
          const lastTabRect = lastTab.getBoundingClientRect();

          // Calculate positions relative to the parent container
          // Left gradient should be positioned at the container's left edge
          const leftPos = 0;
          // Right gradient should be positioned at the container's right edge
          const rightPos = 0;
          
          setFirstTabLeft(leftPos);
          setLastTabRight(rightPos);
        }
      }
    };

    // Add a small delay to ensure DOM is fully rendered
    const timer = setTimeout(updateGradientPositions, 100);
    
    window.addEventListener("resize", updateGradientPositions);
    
    return () => {
      window.removeEventListener("resize", updateGradientPositions);
      clearTimeout(timer);
    };
  }, [tabs.length, classicGradients]);

  return (
    <div className="w-full flex flex-col items-center px-6 md:px-0">
      {/* Single Large Rectangle with Image Background */}
      <div className="relative w-full max-w-[1233px]">
        <div className="relative w-full aspect-[16/11.25] md:aspect-[21/11.25] rounded-2xl md:rounded-3xl overflow-hidden shadow-2xl border border-gray-200/50 dark:border-white/10">
          {/* Background Image - Full Rectangle */}
          <AnimatePresence mode="wait">
            <motion.div
              key={activeTab}
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              exit={{ opacity: 0 }}
              transition={{ duration: 0.4, ease: "easeInOut" }}
              className="absolute inset-0 w-full h-full"
            >
              <Image
                src={
                  theme === "dark"
                    ? tabs[activeTab].darkImage
                    : tabs[activeTab].lightImage
                }
                alt={tabs[activeTab].label}
                className="w-full h-full object-cover"
                fill
                priority
              />
            </motion.div>
          </AnimatePresence>

          {/* Overlay Grid - Text Options on Top */}
          <div className="absolute inset-0 grid md:grid-cols-2">
            {/* Left Half - Empty (shows image) */}
            <div className="hidden md:block"></div>

            {/* Right Half - Text Options Overlay */}
            <div className="flex flex-col justify-center p-6 md:p-8 space-y-3 md:space-y-4 bg-gradient-to-r from-black via-black/80 to-transparent dark:from-black dark:via-black/80 dark:to-transparent md:bg-white/70 md:dark:bg-[#0B0C0E]/90 md:backdrop-blur-sm overflow-y-auto scrollbar-hide">
              {tabs.map((tab, index) => {
                const isActive = activeTab === index
                
                return (
                  <motion.button
                    key={tab.id}
                    onClick={() => {
                      setActiveTab(index);
                      setManualChange(true);
                    }}
                    className={`text-left transition-all duration-300 ${
                      isActive ? 'opacity-100' : 'opacity-50 hover:opacity-75'
                    }`}
                    whileHover={{ x: isActive ? 0 : 8 }}
                    transition={{ duration: 0.2 }}
                  >
                    <h3 className={`text-sm md:text-base font-semibold transition-colors ${
                      isActive 
                        ? 'text-gray-900 dark:text-white' 
                        : 'text-gray-600 dark:text-gray-400'
                    }`}>
                      {tab.label}
                    </h3>
                  </motion.button>
                )
              })}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default TabbedFeature;
