'use client'

import { useState, useEffect, useRef } from 'react'
import Image from 'next/image'

export default function ProjectsAnimationPanel() {
  const [isVisible, setIsVisible] = useState(false)
  const [isHovered, setIsHovered] = useState(false)
  const [hasBeenHovered, setHasBeenHovered] = useState(false)
  const panelRef = useRef<HTMLDivElement>(null)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  // Use isVisible for mobile animation, isHovered for desktop hover effect
  const shouldAnimate = isVisible || isHovered

  return (
    <div 
      ref={panelRef}
      className={`relative flex items-center justify-end w-full transition-all duration-100 delay-[100ms] ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
      onMouseEnter={() => {
        setIsHovered(true)
        setHasBeenHovered(true)
      }}
      onMouseLeave={() => setIsHovered(false)}
    >

      <div className="relative w-full max-w-[320px] md:max-w-md h-[260px] md:h-[480px] ml-auto">
        {/* Back Card */}
        <div className={`absolute top-4 md:top-16 left-[15%] w-[85%] bg-white dark:bg-[#1a1a1a] rounded-xl md:rounded-2xl shadow-xl border border-gray-200 dark:border-gray-700 p-3 md:p-6 transition-all duration-500 ${
          shouldAnimate ? 'translate-x-4 translate-y-4' : 'translate-x-0 translate-y-0'
        }`}>
          <div className="flex items-center justify-between mb-2 md:mb-4">
            <h3 className="font-bold text-xs md:text-base text-gray-900 dark:text-white">Dashboard Refresh</h3>
            <span className="px-2 py-0.5 md:px-3 md:py-1 bg-blue-100 dark:bg-blue-900/30 text-black dark:text-gray-700 text-[9px] md:text-xs font-semibold rounded-full">
              In Progress
            </span>
          </div>
          
          {/* Progress bar */}
          <div className="mb-2 md:mb-4">
            <div className="flex justify-between text-[9px] md:text-xs text-gray-600 dark:text-gray-400 mb-1 md:mb-2">
              <span>Progress</span>
              <span className={`transition-all duration-1000 delay-300 ${shouldAnimate ? 'opacity-100' : 'opacity-0'}`}>
                65%
              </span>
            </div>
            <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-1 md:h-2 overflow-hidden">
              <div 
                className="h-full bg-gradient-to-r from-blue-500 to-cyan-500 rounded-full transition-all duration-1000 delay-300"
                style={{ width: shouldAnimate ? '65%' : '0%' }}
              ></div>
            </div>
          </div>

          {/* Tasks */}
          <div className="space-y-1 md:space-y-2">
            <div className={`flex items-center gap-1.5 md:gap-2 text-[10px] md:text-sm transition-all duration-500 delay-400 ${
              shouldAnimate ? 'opacity-100 translate-x-0' : 'opacity-0 -translate-x-4'
            }`}>
              <div className="w-3 h-3 md:w-4 md:h-4 rounded border-2 border-green-500 bg-green-500 flex items-center justify-center flex-shrink-0">
                <svg className="w-2 h-2 md:w-3 md:h-3 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={3} d="M5 13l4 4L19 7" />
                </svg>
              </div>
              <span className="text-gray-700 dark:text-gray-300 truncate">Visual mockups</span>
            </div>
            <div className={`flex items-center gap-1.5 md:gap-2 text-[10px] md:text-sm transition-all duration-500 delay-500 ${
              shouldAnimate ? 'opacity-100 translate-x-0' : 'opacity-0 -translate-x-4'
            }`}>
              <div className="w-3 h-3 md:w-4 md:h-4 rounded border-2 border-green-500 bg-green-500 flex items-center justify-center flex-shrink-0">
                <svg className="w-2 h-2 md:w-3 md:h-3 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={3} d="M5 13l4 4L19 7" />
                </svg>
              </div>
              <span className="text-gray-700 dark:text-gray-300 truncate">Stakeholder buy-in</span>
            </div>
            <div className={`flex items-center gap-1.5 md:gap-2 text-[10px] md:text-sm transition-all duration-500 delay-[600ms] ${
              shouldAnimate ? 'opacity-100 translate-x-0' : 'opacity-0 -translate-x-4'
            }`}>
              <div className="w-3 h-3 md:w-4 md:h-4 rounded border-2 border-gray-400 dark:border-gray-500 flex-shrink-0"></div>
              <span className="text-gray-700 dark:text-gray-300 truncate">Code integration</span>
            </div>
          </div>
        </div>

        {/* Front Card */}
        <div className={`absolute top-0 right-0 w-[85%] bg-white dark:bg-[#1a1a1a] rounded-xl md:rounded-2xl shadow-2xl border border-gray-200 dark:border-gray-700 p-3 md:p-6 transition-all duration-500 ${
          shouldAnimate ? '-translate-x-4 -translate-y-4' : 'translate-x-0 translate-y-0'
        }`}>
          <div className="flex items-center justify-between mb-2 md:mb-4">
            <h3 className="font-bold text-xs md:text-base text-gray-900 dark:text-white">Feature Launch 2.0</h3>
            <span className="px-2 py-0.5 md:px-3 md:py-1 bg-green-100 dark:bg-green-900/30 text-green-600 dark:text-green-400 text-[9px] md:text-xs font-semibold rounded-full">
              Active
            </span>
          </div>
          
          {/* Progress bar */}
          <div className="mb-2 md:mb-4">
            <div className="flex justify-between text-[9px] md:text-xs text-gray-600 dark:text-gray-400 mb-1 md:mb-2">
              <span>Progress</span>
              <span className={`transition-all duration-1000 delay-300 ${shouldAnimate ? 'opacity-100' : 'opacity-0'}`}>
                82%
              </span>
            </div>
            <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-1 md:h-2 overflow-hidden">
              <div 
                className="h-full bg-gradient-to-r from-green-500 to-emerald-500 rounded-full transition-all duration-1000 delay-300"
                style={{ width: shouldAnimate ? '82%' : '0%' }}
              ></div>
            </div>
          </div>

          {/* Tasks */}
          <div className="space-y-1 md:space-y-2">
            <div className={`flex items-center gap-1.5 md:gap-2 text-[10px] md:text-sm transition-all duration-500 delay-400 ${
              shouldAnimate ? 'opacity-100 translate-x-0' : 'opacity-0 -translate-x-4'
            }`}>
              <div className="w-3 h-3 md:w-4 md:h-4 rounded border-2 border-green-500 bg-green-500 flex items-center justify-center flex-shrink-0">
                <svg className="w-2 h-2 md:w-3 md:h-3 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={3} d="M5 13l4 4L19 7" />
                </svg>
              </div>
              <span className="text-gray-700 dark:text-gray-300 truncate">Testing phase</span>
            </div>
            <div className={`flex items-center gap-1.5 md:gap-2 text-[10px] md:text-sm transition-all duration-500 delay-500 ${
              shouldAnimate ? 'opacity-100 translate-x-0' : 'opacity-0 -translate-x-4'
            }`}>
              <div className="w-3 h-3 md:w-4 md:h-4 rounded border-2 border-green-500 bg-green-500 flex items-center justify-center flex-shrink-0">
                <svg className="w-2 h-2 md:w-3 md:h-3 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={3} d="M5 13l4 4L19 7" />
                </svg>
              </div>
              <span className="text-gray-700 dark:text-gray-300 truncate">Bug fixing</span>
            </div>
            <div className={`flex items-center gap-1.5 md:gap-2 text-[10px] md:text-sm transition-all duration-500 delay-[600ms] ${
              shouldAnimate ? 'opacity-100 translate-x-0' : 'opacity-0 -translate-x-4'
            }`}>
              <div className="w-3 h-3 md:w-4 md:h-4 rounded border-2 border-black bg-black flex items-center justify-center flex-shrink-0">
                <div className="w-1 h-1 md:w-2 md:h-2 bg-white rounded-full"></div>
              </div>
              <span className="text-gray-700 dark:text-gray-300 truncate">Launch preparation</span>
            </div>
          </div>

          {/* Team avatars */}
          <div className={`flex items-center gap-1.5 md:gap-2 mt-2 md:mt-4 pt-2 md:pt-4 border-t border-gray-200 dark:border-gray-700 transition-all duration-500 delay-700 ${
            shouldAnimate ? 'opacity-100' : 'opacity-0'
          }`}>
            <div className="flex -space-x-1.5 md:-space-x-2">
              <Image 
                src="https://images.unsplash.com/photo-1534528741775-53994a69daeb?w=200&h=200&fit=crop&crop=faces" 
                alt="Team member" 
                width={20}
                height={20}
                className="w-4 h-4 md:w-6 md:h-6 rounded-full border-2 border-white dark:border-gray-800 object-cover"
              />
              <Image 
                src="https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=200&h=200&fit=crop&crop=faces" 
                alt="Team member" 
                width={20}
                height={20}
                className="w-4 h-4 md:w-6 md:h-6 rounded-full border-2 border-white dark:border-gray-800 object-cover"
              />
              <Image 
                src="https://images.unsplash.com/photo-1573497019940-1c28c88b4f3e?w=200&h=200&fit=crop&crop=faces" 
                alt="Team member" 
                width={20}
                height={20}
                className="w-4 h-4 md:w-6 md:h-6 rounded-full border-2 border-white dark:border-gray-800 object-cover"
              />
            </div>
            <span className="text-[9px] md:text-xs text-gray-600 dark:text-gray-400">3 team members</span>
          </div>
        </div>

        {/* Floating badge */}
        <div className={`absolute -bottom-2 md:bottom-6 left-1/2 -translate-x-1/2 bg-white dark:bg-[#1a1a1a] rounded-xl md:rounded-2xl shadow-lg px-3 md:px-6 py-1.5 md:py-3 border border-gray-200 dark:border-gray-600 transition-all duration-500 delay-800 ${
          shouldAnimate ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-4'
        }`}>
          <div className="flex items-center gap-2">
            <div className="w-1.5 h-1.5 md:w-2 md:h-2 rounded-full bg-green-500 animate-pulse"></div>
            <span className="text-xs md:text-sm font-semibold text-gray-900 dark:text-white">Live Sync </span>
          </div>
        </div>
      </div>
    </div>
  )
}
