"use client";

import React, { useState } from "react";
import { ArrowRightIcon } from "lucide-react";
import { LucideIcon } from "lucide-react";
import { IconType } from "react-icons";

interface PricingFeature {
  icon: LucideIcon | IconType;
  text: string;
  subtext: string;
}

interface PricingPlan {
  name: string;
  description: string;
  price: string;
  pricePeriod: string;
  buttonText: string;
  buttonVariant: "primary" | "outline";
  badge: string | null;
  features: PricingFeature[];
  highlighted: boolean;
  backgroundImage?: boolean;
}

interface PricingSectionProps {
  title?: string;
  subtitle?: string;
  monthlyPlans: PricingPlan[];
  annualPlans: PricingPlan[];
  onPlanSelect: (planName: string, cycle: "monthly" | "annual") => void;
  contactSalesUrl?: string;
}

const PricingSection: React.FC<PricingSectionProps> = ({
  title = "Plans and Pricing",
  subtitle = 'Receive <span class="font-semibold">unlimited storage</span> when you pay yearly, and save on your plan.',
  monthlyPlans,
  annualPlans,
  onPlanSelect,
  contactSalesUrl = "/contact",
}) => {
  const [isAnnual, setIsAnnual] = useState(true);

  return (
    <>
      {/* Pricing Section */}
      <section
        className="w-full bg-white dark:bg-[#0B0C0E] py-20 md:py-32"
        id="pricing"
      >
        <div className="relative max-w-[1233px] mx-auto px-8 md:px-12 lg:px-20">
          {/* Header */}
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl lg:text-6xl font-bold mb-6 text-gray-900 dark:text-white">
              {title}
            </h2>
            <p
              className="text-lg text-gray-600 dark:text-gray-400 max-w-2xl mx-auto mb-8"
              dangerouslySetInnerHTML={{ __html: subtitle }}
            />

            {/* Billing Toggle */}
            <div className="inline-flex items-center gap-4 bg-white dark:bg-[#1a1a1a] p-2 rounded-full shadow-lg border border-gray-200 dark:border-white/10">
              <button
                onClick={() => setIsAnnual(false)}
                className={`px-6 py-2 rounded-full transition-all duration-300 font-medium ${
                  !isAnnual
                    ? "bg-gray-900 text-white dark:bg-white dark:text-black shadow-md"
                    : "text-gray-600 dark:text-gray-400"
                }`}
              >
                Monthly
              </button>
              <button
                onClick={() => setIsAnnual(true)}
                className={`px-6 py-2 rounded-full transition-all duration-300 font-medium flex items-center gap-2 ${
                  isAnnual
                    ? "bg-gray-900 text-white dark:bg-white dark:text-black shadow-md"
                    : "text-gray-600 dark:text-gray-400"
                }`}
              >
                Annual
                <span className="bg-gray-900 text-white dark:bg-white dark:text-black px-2 py-0.5 rounded-full text-xs font-bold">
                  Save 50%
                </span>
              </button>
            </div>
          </div>

          {/* Pricing Cards */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8 max-w-7xl mx-auto">
            {(isAnnual ? annualPlans : monthlyPlans).map((plan) => {
              return (
                <div
                  key={plan.name}
                  className={`relative group mx-0 md:mx-0 ${
                    plan.highlighted ? "lg:scale-105 z-10" : ""
                  }`}
                >
                  {/* Badge for highlighted plan */}
                  {plan.backgroundImage && (
                    <div className="absolute -top-4 left-1/2 transform -translate-x-1/2 z-20">
                      <span className="bg-gradient-to-r from-gray-900 to-gray-700 dark:from-white dark:to-gray-200 text-white dark:text-black text-xs font-bold px-4 py-1.5 rounded-full shadow-lg">
                        MOST POPULAR
                      </span>
                    </div>
                  )}

                  <div
                    className={`relative bg-white dark:bg-[#1a1a1a] border-2 ${
                      plan.highlighted
                        ? "border-gray-900 dark:border-white shadow-2xl"
                        : "border-gray-200 dark:border-white/10"
                    } rounded-2xl md:rounded-3xl p-6 md:p-8 hover:shadow-xl transition-all duration-500 h-full flex flex-col`}
                  >
                    {/* Plan Name */}
                    <div className="mb-6">
                      <h3 className="text-2xl md:text-3xl font-bold text-gray-900 dark:text-white mb-2">
                        {plan.name}
                      </h3>
                      <p className="text-sm text-gray-600 dark:text-gray-400 whitespace-pre-line">
                        {plan.description}
                      </p>
                    </div>

                    {/* Price */}
                    <div className="mb-8">
                      <div className="flex items-baseline gap-2">
                        <span className="text-5xl md:text-6xl font-bold text-gray-900 dark:text-white">
                          {plan.price}
                        </span>
                        <span className="text-gray-600 dark:text-gray-400 text-lg">
                          /month
                        </span>
                      </div>
                      {isAnnual && (
                        <p className="text-sm text-gray-500 dark:text-gray-500 mt-2">
                          Billed annually
                        </p>
                      )}
                    </div>

                    {/* Features */}
                    <div className="space-y-4 flex-grow mb-8">
                      {plan.features.map((feature, featureIndex) => {
                        const FeatureIcon = feature.icon;
                        return (
                          <div
                            key={featureIndex}
                            className="flex items-start gap-3"
                          >
                            <div className="flex-shrink-0 w-5 h-5 rounded-full bg-gray-100 dark:bg-white/10 flex items-center justify-center mt-0.5">
                              <FeatureIcon className="w-3 h-3 text-gray-900 dark:text-white" />
                            </div>
                            <div className="flex-1">
                              <p className="text-gray-900 dark:text-white font-medium text-sm">
                                {feature.subtext}
                              </p>
                            </div>
                          </div>
                        );
                      })}
                    </div>

                    {/* CTA Button - Now at bottom */}
                    <button
                      onClick={() =>
                        onPlanSelect(plan.name, isAnnual ? "annual" : "monthly")
                      }
                      className={`w-full py-4 px-6 rounded-xl font-semibold text-lg transition-all duration-300 mt-auto ${
                        plan.buttonVariant === "primary"
                          ? "bg-gray-900 text-white dark:bg-white dark:text-black hover:bg-gray-800 dark:hover:bg-gray-100 shadow-lg hover:shadow-xl"
                          : "bg-white dark:bg-transparent border-2 border-gray-900 dark:border-white text-gray-900 dark:text-white hover:bg-gray-50 dark:hover:bg-white/5"
                      }`}
                    >
                      {plan.buttonText}
                    </button>
                  </div>
                </div>
              );
            })}
          </div>

          {/* Bottom CTA */}
          <div className="text-center mt-16">
            <p className="text-gray-600 dark:text-gray-400 mb-6">
              Need a custom plan for your organization?
            </p>
            <button
              onClick={() => {
                window.location.href = contactSalesUrl;
              }}
              className="inline-flex items-center gap-2 px-8 py-4 bg-gray-900 dark:bg-white text-white dark:text-black rounded-xl font-semibold hover:bg-gray-800 dark:hover:bg-gray-100 transition-all duration-300 shadow-lg hover:shadow-xl"
            >
              Contact Sales
              <ArrowRightIcon className="w-5 h-5" />
            </button>
          </div>
        </div>
      </section>
    </>
  );
};

export default PricingSection;
