"use client";
import { useState, useEffect, useRef } from "react";
import Link from "next/link";
import { Button } from "./button";
import { ArrowRightIcon, Ellipsis, Logs } from "lucide-react";
import ThemeSwitch from "./themeSwitch";
import { useTheme } from "../atoms/theme";
import { apps } from "../constants/products";
import { footerColumns } from "../constants/footerText";
import { FaArrowRight } from "react-icons/fa";
import AnnouncementBar from "./AnnouncementBar";
import { getCookie } from "../utils/cookies";
import ClikkleLogo from "./ClikkleLogo";
import { iconMap } from "../utils/iconMap";
import SidebarDrawer from "./sidebarDrawer";
import CRMAnimationPanel from "./CRMAnimationPanel";
import ChatAnimationPanel from "./ChatAnimationPanel";
import FilesAnimationPanel from "./FilesAnimationPanel";
import EsignWorkflowAnimationPanel from "./EsignWorkflowAnimationPanel";
import HRAnimationPanel from "./HRAnimationPanel";
import ProjectsAnimationPanel from "./ProjectsAnimationPanel";
import CampaignsAnimationPanel from "./CampaignsAnimationPanel";

// Product rotation configuration
const productRotation = [
  {
    title: "Chat",
    description: "Real-time messaging for your team and customers",
    route: "/chat",
    component: ChatAnimationPanel,
  },
  {
    title: "Files",
    description: "Secure cloud storage for all your documents",
    route: "/files",
    component: FilesAnimationPanel,
  },
  {
    title: "eSign",
    description: "Sign and send documents digitally in seconds",
    route: "/esign",
    component: EsignWorkflowAnimationPanel,
  },
  {
    title: "HR",
    description: "Streamline hiring, onboarding, and HR operations",
    route: "/hr",
    component: HRAnimationPanel,
  },
  {
    title: "Campaigns",
    description: "Create and manage marketing campaigns",
    route: "/campaigns",
    component: CampaignsAnimationPanel,
  },
];

export default function MainHeader() {
  const { theme } = useTheme();
  const [scrolled, setScrolled] = useState(false);
  const [leftDrawerOpen, setLeftDrawerOpen] = useState(false);
  const [productsOpen, setProductsOpen] = useState(false);
  const [dotsOpen, setDotsOpen] = useState(false);
  const [announcementOpen, setAnnouncementOpen] = useState(false);
  const [currentProductIndex, setCurrentProductIndex] = useState(0);

  const productsRef = useRef<HTMLDivElement>(null);
  const dotsRef = useRef<HTMLDivElement>(null);

  const toggleLeftDrawer = () => setLeftDrawerOpen((prev) => !prev);

  // Pick a random product once on mount and stick with it
  useEffect(() => {
    const randomIndex = Math.floor(Math.random() * productRotation.length);
    setCurrentProductIndex(randomIndex);
  }, []);

  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 0);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (productsRef.current && !productsRef.current.contains(event.target as Node)) {
        setProductsOpen(false);
      }
      if (dotsRef.current && !dotsRef.current.contains(event.target as Node)) {
        setDotsOpen(false);
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  useEffect(() => {
    const checkHeight = () => {
      const el = document.querySelector("[data-announcement]") as HTMLElement;
      if (!el) return;
      if (el.offsetHeight === 0) {
        setAnnouncementOpen(false);
      }
    };
    checkHeight();
    const observer = new MutationObserver(checkHeight);
    observer.observe(document.body, { childList: true, subtree: true });
    return () => observer.disconnect();
  }, []);

  useEffect(() => {
    const checkDismissedAnnouncements = () => {
      const dismissed = getCookie("dismissedAnnouncements");
      if (dismissed) {
        try {
          const dismissedIds = JSON.parse(dismissed);
          if (dismissedIds.length > 0) {
            setAnnouncementOpen(false);
          }
        } catch (error) {
          console.error("Error parsing dismissed announcements:", error);
        }
      }
    };

    checkDismissedAnnouncements();
    const interval = setInterval(checkDismissedAnnouncements, 1000);
    return () => clearInterval(interval);
  }, []);

  return (
    <>
      <section className="w-full">
        <section className="relative w-full overflow-hidden">
          <div
            className={`fixed top-0 inset-x-0 z-50 ${
              scrolled
                ? "backdrop-blur-md bg-white/90 dark:bg-[#0B0C0E]/90"
                : productsOpen || dotsOpen
                ? "backdrop-blur-md bg-white/90 dark:bg-[#0B0C0E]/90"
                : ""
            }`}
          >
            <div
              ref={(el) => {
                if (el) {
                  const height = el.getBoundingClientRect().height;
                  if (height !== 0) {
                    setAnnouncementOpen(true);
                  }
                }
              }}
            >
              <AnnouncementBar onClose={() => setAnnouncementOpen(false)} />
            </div>
            <div className="container max-w-[1233px] mx-auto max-[1300px]:px-14 max-[1100px]:px-4">
              <nav className={`relative flex items-center justify-between py-4`}>
                <ClikkleLogo />

                <div className="flex md:hidden items-center">
                  <div
                    className="flex items-center gap-0 rounded-xl border border-gray-300 dark:border-[#ffffff3a] p-[8px] shadow-sm"
                    onClick={toggleLeftDrawer}
                  >
                    <p className="text-sm font-medium text-gray-900 dark:text-white">
                      <Logs className="w-5 h-5 text-gray-900 dark:text-white mr-3" />
                    </p>
                    <p className="flex items-center rounded-xl text-sm font-bold">Menu</p>
                  </div>
                </div>

                <div className="hidden md:flex items-center gap-12 font-inter font-medium text-[18px]">
                  <Link
                    href="/#features"
                    className="text-gray-900 dark:text-white hover:text-gray-600 dark:hover:text-[#94A3B8] transition-colors"
                  >
                    Features
                  </Link>
                  <Link
                    href="/about"
                    className="text-gray-900 dark:text-white hover:text-gray-600 dark:hover:text-[#94A3B8] transition-colors"
                  >
                    About Us
                  </Link>
                  <Link
                    href="/blogs"
                    className="text-gray-900 dark:text-white hover:text-gray-600 dark:hover:text-[#94A3B8] transition-colors"
                  >
                    Blog
                  </Link>
                  <div className="flex flex-row justify-center items-center">
                    <div className="relative" ref={productsRef}>
                      <button
                        onClick={() => setProductsOpen((o) => !o)}
                        className="cursor-pointer flex items-center gap-1 text-gray-900 dark:text-white hover:text-gray-600 dark:hover:text-[#94A3B8] transition-colors"
                      >
                        Products
                      </button>
                      {productsOpen && (
                        <div
                          className={`fixed ${
                            announcementOpen ? "top-[115px]" : "top-[75px]"
                          } 
              left-1/2 transform -translate-x-1/2 w-full bg-white/90 dark:bg-[#0B0C0E]/90 backdrop-blur-md 
              shadow-lg p-6 border-b-2 border-gray-300 dark:border-[#26282E] z-50 flex justify-center items-center pb-0`}
                        >
                          <div className="container max-w-[1233px] max-[1300px]:px-14 max-[1100px]:px-4 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                            <div className="flex flex-col gap-3">
                              {apps
                                .slice(0, Math.ceil(apps.length / 2))
                                .map((product) => {
                                  const IconComponent = iconMap[product.icon];
                                  return (
                                    <Link
                                      key={product.id}
                                      href={product.route}
                                      className="flex items-center gap-4 p-[0.35rem] rounded-lg hover:bg-gray-100 dark:hover:bg-white/10 transition"
                                    >
                                      <div className="bg-gray-100 dark:bg-gray-900/50 rounded-xl p-2 flex items-center justify-center">
                                        {IconComponent && (
                                          <IconComponent className="w-[26px] h-[26px] text-gray-900 dark:text-white" />
                                        )}
                                      </div>
                                      <div>
                                        <p className="font-bold text-gray-900 dark:text-white text-sm">
                                          {product.title}
                                        </p>
                                        <p className="text-xs text-gray-600 dark:text-gray-300">
                                          {product.description}
                                        </p>
                                      </div>
                                    </Link>
                                  );
                                })}
                            </div>

                            <div className="flex flex-col gap-3">
                              {apps
                                .slice(
                                  Math.ceil(apps.length / 2),
                                  Math.ceil(apps.length / 2) + 2
                                )
                                .map((product) => {
                                  const IconComponent = iconMap[product.icon];
                                  return (
                                    <Link
                                      key={product.id}
                                      href={product.route}
                                      className="flex items-center gap-4 p-[0.35rem] rounded-lg hover:bg-gray-100 dark:hover:bg-white/10 transition"
                                    >
                                      <div className="bg-gray-100 dark:bg-gray-900/50 rounded-xl p-2 flex items-center justify-center">
                                        {IconComponent && (
                                          <IconComponent className="w-[26px] h-[26px] text-gray-900 dark:text-white" />
                                        )}
                                      </div>
                                      <div>
                                        <p className="font-bold text-gray-900 dark:text-white text-sm">
                                          {product.title}
                                        </p>
                                        <p className="text-xs text-gray-600 dark:text-gray-300">
                                          {product.description}
                                        </p>
                                      </div>
                                    </Link>
                                  );
                                })}

                              <Link href="/products">
                                <Button
                                  style={{ marginTop: "5px" }}
                                  variant="ghost"
                                  className={`cursor-pointer hover:bg-gray-800 w-44 hover:dark:bg-[#16171B] hover:text-white hover:dark:text-white font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#16171B] text-gray-900 dark:text-white border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                                >
                                  View all products
                                  <ArrowRightIcon className="h-2 w-2 dark:text-white" />
                                </Button>
                              </Link>
                            </div>

                            <div className="hidden lg:flex flex-col items-center justify-center text-center gap-4 py-6">
                              {/* Animation Panel */}
                              <div className="w-[280px] h-[160px] rounded-xl overflow-hidden bg-gray-50 dark:bg-gray-900/50 flex items-center justify-center">
                                <div className="scale-[0.35] origin-center">
                                  {(() => {
                                    const CurrentPanel = productRotation[currentProductIndex].component;
                                    return <CurrentPanel />;
                                  })()}
                                </div>
                              </div>

                              {/* Dynamic Get Started Button */}
                              <Button
                                variant="ghost"
                                onClick={() => {
                                  window.location.href = productRotation[currentProductIndex].route;
                                }}
                                className={`cursor-pointer hover:bg-gray-800 hover:dark:bg-[#16171B] hover:text-white hover:dark:text-white font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#16171B] text-gray-900 dark:text-white border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                              >
                                Try {productRotation[currentProductIndex].title} for Free
                                <ArrowRightIcon className={`h-2 w-2 dark:text-white`} />
                              </Button>
                            </div>
                          </div>
                        </div>
                      )}
                    </div>

                    <div className="relative" ref={dotsRef}>
                      <Ellipsis
                        onClick={() => setDotsOpen((o) => !o)}
                        className="w-8 h-6 rounded-full p-[1px] cursor-pointer"
                      />
                      {dotsOpen && (
                        <div
                          className={`fixed ${
                            announcementOpen ? "top-[115px]" : "top-[75px]"
                          } left-1/2 transform -translate-x-1/2 w-full 
    bg-white/90 dark:bg-[#0B0C0E]/90 backdrop-blur-md 
    shadow-lg border-b-2 border-gray-300 dark:border-[#26282E] 
    z-50 flex justify-center`}
                        >
                          <div className="container max-w-[1233px] max-[1300px]:px-14 max-[1100px]:px-4 py-6">
                            <div className="flex flex-col md:flex-row justify-between items-start gap-12">
                              <div className="space-y-12">
                                <div className="max-w-[285px] space-y-2 mb-0">
                                  <h3 className="font-instrument-sans font-medium text-[18px] leading-[20px] text-gray-900 dark:text-white">
                                    Get Started For Free
                                  </h3>
                                  <p className="font-body-medium text-[#51515E] dark:text-white">
                                    Streamline your Business with our Apps. Effortlessly for every team.
                                  </p>
                                </div>
                                <Button
                                  style={{ marginTop: "20px" }}
                                  variant="ghost"
                                  onClick={() => {
                                    window.location.replace(
                                      "https://hub.clikkle.com/signup?theme=" + theme
                                    );
                                  }}
                                  className={`cursor-pointer hover:bg-gray-800 w-44 hover:dark:bg-[#16171B] hover:text-white hover:dark:text-white font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#16171B] text-gray-900 dark:text-white border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                                >
                                  See How It Works
                                  <ArrowRightIcon className={`h-2 w-2 dark:text-white`} />
                                </Button>
                              </div>

                              <div className="flex flex-wrap gap-x-14 gap-y-8 text-xs sm:text-sm">
                                {footerColumns.map((column, index) => (
                                  <div key={index} className="flex flex-col gap-5">
                                    <h4 className="font-label-medium text-gray-900 dark:text-white">
                                      {column.title}
                                    </h4>
                                    <ul className="flex flex-col gap-4">
                                      {column.links.map((link, linkIndex) => (
                                        <li key={linkIndex}>
                                          <div className="flex items-start gap-3">
                                            <Link
                                              href={link.url}
                                              target={
                                                link.url.startsWith("http") ? "_blank" : "_self"
                                              }
                                              rel={
                                                link.url.startsWith("http")
                                                  ? "noopener noreferrer"
                                                  : undefined
                                              }
                                              className="font-body-medium text-[#51515E] dark:text-[#d7d7de] hover:text-gray-900 dark:hover:text-white transition-colors whitespace-nowrap flex items-center gap-2"
                                            >
                                              {link.text}
                                              {column.title === "Social" && (
                                                <FaArrowRight className="w-2 h-2" />
                                              )}
                                            </Link>

                                            {link.isNew && (
                                              <div className="hidden md:flex items-center gap-2">
                                                <div className="w-4 h-4 flex items-center justify-center">
                                                  <div className="w-[13px] h-[13px] bg-[#EDEFFC] rounded-[6.67px] flex items-center justify-center">
                                                    <div className="w-1 h-1 bg-[#253237] rounded-sm" />
                                                  </div>
                                                </div>
                                                <span className="font-body-medium text-gray-900 dark:text-white whitespace-nowrap">
                                                  New
                                                </span>
                                              </div>
                                            )}
                                          </div>
                                        </li>
                                      ))}
                                    </ul>
                                  </div>
                                ))}
                              </div>
                            </div>
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                </div>

                <div className="hidden md:flex items-center gap-4">
                  <div className="hidden md:flex items-center gap-4">
                    <div className="w-6">
                      <ThemeSwitch />
                    </div>
                    <div className="flex items-center gap-2 rounded-xl border border-gray-300 dark:border-[#ffffff3a] p-1 shadow-sm">
                      <button
                        className="cursor-pointer px-3 py-1 text-sm font-medium text-[#253237] dark:text-white"
                        onClick={() => {
                          window.location.replace(
                            "https://hub.clikkle.com/login?theme=" + theme
                          );
                        }}
                      >
                        Login
                      </button>

                      <button
                        className="cursor-pointer flex items-center gap-1 rounded-xl bg-[#253237] dark:bg-white hover:bg-[#1a2428] dark:hover:bg-gray-100 px-4 py-1.5 text-sm font-medium text-white dark:text-[#253237] transition"
                        onClick={() => {
                          window.location.replace(
                            "https://hub.clikkle.com/signup?theme=" + theme
                          );
                        }}
                      >
                        Try for free
                        <span>
                          <ArrowRightIcon className="w-4" />
                        </span>
                      </button>
                    </div>
                  </div>
                </div>
              </nav>
            </div>
          </div>
        </section>
      </section>
      <SidebarDrawer open={leftDrawerOpen} toggle={toggleLeftDrawer} />
    </>
  );
}
