"use client";
import React, { useEffect, useMemo, useRef, useState } from "react";
import { Box, Button, Typography, useMediaQuery } from "@mui/material";
import { useTheme } from "@mui/material/styles";
import { motion, AnimatePresence } from "framer-motion";
import { usePathname } from "next/navigation";
import { getDismissedOffers } from "../utils/cookies";
import { ImArrowUpRight2 } from "react-icons/im";

const DEFAULT_API_BASE_URL = "https://admin.clikkle.com:5100";

type LimitedOffer = {
  _id: string;
  title?: string;
  pageName?: string;
  ctaText?: string;
  ctaUrl?: string;
  endTime?: string;
  isEnabled?: boolean;
  accentColor?: string;
};

function useCountdown(endTime?: string) {
  const [remaining, setRemaining] = useState<number>(0);

  useEffect(() => {
    const parse = (t?: string) => (t ? new Date(t).getTime() : 0);
    const target = parse(endTime);
    const tick = () => {
      const now = Date.now();
      setRemaining(Math.max(0, target - now));
    };
    tick();
    const id = setInterval(tick, 1000);
    return () => clearInterval(id);
  }, [endTime]);

  const parts = useMemo(() => {
    let ms = remaining;
    const days = Math.floor(ms / (24 * 60 * 60 * 1000));
    ms -= days * 24 * 60 * 60 * 1000;
    const hours = Math.floor(ms / (60 * 60 * 1000));
    ms -= hours * 60 * 60 * 1000;
    const minutes = Math.floor(ms / (60 * 1000));
    ms -= minutes * 60 * 1000;
    const seconds = Math.floor(ms / 1000);
    return { days, hours, minutes, seconds };
  }, [remaining]);

  return parts;
}

const LimitedOfferBar = ({
  apiBaseUrl = DEFAULT_API_BASE_URL,
  pageName: explicitPageName,
}: {
  apiBaseUrl?: string;
  pageName?: string;
}) => {
  const pathname = usePathname();
  const pageName = explicitPageName ?? pathname;
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down("sm"));
  const [offers, setOffers] = useState<LimitedOffer[]>([]);
  const [dismissedIds, setDismissedIds] = useState<Set<string>>(new Set());
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [show, setShow] = useState(false);
  const lastYRef = useRef<number>(0);

  useEffect(() => {
    setDismissedIds(getDismissedOffers());
  }, []);

  useEffect(() => {
    const onScroll = () => {
      const y = window.scrollY;
      const dirDown = y > lastYRef.current;
      setShow(dirDown && y > 150);
      lastYRef.current = y;
    };
    onScroll();
    window.addEventListener("scroll", onScroll, { passive: true });
    return () => window.removeEventListener("scroll", onScroll);
  }, []);

  useEffect(() => {
    const fetchOffers = async () => {
      try {
        setIsLoading(true);
        setError(null);
        const res = await fetch(
          `${apiBaseUrl}/api/limited-offers/active?pageName=${encodeURIComponent(
            pageName || ""
          )}`
        );
        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        const data = await res.json();
        setOffers(data?.data || []);
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
      } catch (e: any) {
        console.error("Error fetching limited offers:", e);
        setError(e.message);
        setOffers([]);
      } finally {
        setIsLoading(false);
      }
    };
    fetchOffers();
    const id = setInterval(fetchOffers, 5 * 60 * 1000);
    return () => clearInterval(id);
  }, [apiBaseUrl, pageName]);

  const visibleOffers = offers.filter((o) => !dismissedIds.has(o._id));

  if (isLoading || error || visibleOffers.length === 0) return null;

  const OfferItem = ({ offer }: { offer: LimitedOffer }) => {
    const { days, hours, minutes, seconds } = useCountdown(offer.endTime);
    const accent = offer.accentColor || "#0B0C0E";
    const expired = days === 0 && hours === 0 && minutes === 0 && seconds === 0;
    if (expired) return null;
    return (
      <Box
        sx={{
          backgroundColor: "transparent",
          color: theme.palette.text.primary,
          borderTop: `1px solid ${theme.palette.divider}`,
          py: { xs: 1, md: 1.25 },
          px: { xs: 1.25, md: 3 },
          display: "flex",
          alignItems: "center",
          width: "100%",
          justifyContent: "center",
          gap: { xs: 1.25, md: 2 },
          flexDirection: isMobile ? "column" : "row",
          boxShadow:
            theme.palette.mode === "dark"
              ? "0 -6px 12px rgba(0,0,0,0.4)"
              : "0 -6px 12px rgba(0,0,0,0.06)",
        }}
      >
        <Typography
          sx={{
            fontWeight: 800,
            fontSize: { xs: 15, md: 20 },
            fontFamily: "var(--font-inter), sans-serif",
          }}
        >
          {offer.title || "Special Offer Closing In..."}
        </Typography>
        <Box
          sx={{
            display: "flex",
            alignItems: "center",
            gap: { xs: 1, md: 2 },
            justifyContent: "center",
          }}
        >
          <Box
            sx={{
              display: "flex",
              alignItems: "center",
              gap: { xs: 1, md: 2 },
              flexWrap: "wrap",
            }}
          >
            {[
              { value: days, label: "days" },
              { value: hours, label: "hours" },
              { value: minutes, label: "minutes" },
              { value: seconds, label: "seconds" },
            ].map((d) => (
              <Box
                key={d.label}
                sx={{
                  minWidth: 56,
                  px: 1.5,
                  py: 1,
                  height: 70,
                  borderRadius: 1.5,
                  border: `1px solid ${theme.palette.divider}`,
                  backgroundColor: "bg-white dark:bg-[#1a1924]",
                  display: "flex",
                  flexDirection: "column",
                  alignItems: "center",
                  justifyContent: "center",
                  boxShadow:
                    theme.palette.mode === "dark"
                      ? "0 1px 4px rgba(0,0,0,0.6)"
                      : "0 1px 4px rgba(0,0,0,0.08)",
                }}
              >
                <AnimatePresence mode="popLayout" initial={false}>
                  <motion.div
                    key={d.value}
                    initial={{ rotateX: -90, opacity: 0 }}
                    animate={{ rotateX: 0, opacity: 1 }}
                    exit={{ rotateX: 90, opacity: 0 }}
                    transition={{ duration: 0.4 }}
                    style={{ transformOrigin: "bottom" }}
                  >
                    <Typography
                      sx={{ fontSize: 18, fontWeight: 800, lineHeight: 1 }}
                    >
                      {d.value}
                    </Typography>
                  </motion.div>
                </AnimatePresence>
                <Typography sx={{ fontSize: 11, opacity: 0.7 }}>
                  {d.label}
                </Typography>
              </Box>
            ))}
          </Box>
        </Box>
        <Box
          sx={{
            display: "flex",
            alignItems: "center",
            gap: 1.25,
            mt: { xs: 1.25, md: 0 },
            width: isMobile ? "100%" : "auto",
            justifyContent: "center",
            order: isMobile ? -1 : 0,
          }}
        >
          {offer.ctaText && (
            <button
              onClick={() => {
                try {
                  fetch(`${apiBaseUrl}/api/limited-offers/${offer._id}/click`, {
                    method: "POST",
                    headers: { "Content-Type": "application/json" },
                    body: JSON.stringify({ pageName }),
                  }).catch(() => {});
                } catch {}
                if (offer.ctaUrl) window.open(offer.ctaUrl, "_blank");
              }}
              className="cursor-pointer font-inter font-extrabold px-4 md:px-4 py-1.5 rounded-2xl text-white border-2 border-white flex items-center justify-center gap-2"
              style={{
                backgroundColor: theme.palette.mode === "dark" ? "#0B0C0E" : "#253237",
                outline: theme.palette.mode === "dark" ? "2px solid #0B0C0E" : "2px solid #253237",
                outlineOffset: "2px",
                width: isMobile ? "100%" : "auto",
              }}
            >
              {offer.ctaText}
              <span className="text-base">
                <ImArrowUpRight2 />
              </span>
            </button>
          )}
        </Box>
      </Box>
    );
  };

  return (
    <Box
      sx={{
        position: "fixed",
        bottom: 0,
        left: 0,
        right: 0,
        zIndex: 1200,
        transform: show ? "translateY(0)" : "translateY(100%)",
        opacity: show ? 1 : 0,
        transition: "transform 300ms ease, opacity 300ms ease",
        backdropFilter: "blur(12px)",
        backgroundColor:
          theme.palette.mode === "dark"
            ? "rgba(26,25,36,0.55)"
            : "rgba(255,255,255,0.6)",
        borderTop: `1px solid ${theme.palette.divider}`,
      }}
    >
      {visibleOffers.map((offer) => (
        <OfferItem key={offer._id} offer={offer} />
      ))}
    </Box>
  );
};

export default LimitedOfferBar;
