'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { Mail, User, Phone, CheckCircle2, Database } from 'lucide-react'

interface FormField {
  id: number
  icon: React.ReactNode
  label: string
  value: string
  validated: boolean
}

const fieldsData: FormField[] = [
  { id: 1, icon: <User className="w-4 h-4 md:w-5 md:h-5" />, label: 'Name', value: 'Sarah Johnson', validated: false },
  { id: 2, icon: <Mail className="w-4 h-4 md:w-5 md:h-5" />, label: 'Email', value: 'sarah@company.com', validated: false },
  { id: 3, icon: <Phone className="w-4 h-4 md:w-5 md:h-5" />, label: 'Phone', value: '+1 555-0123', validated: false }
]

export default function LeadCaptureAnimationPanel() {
  const [fields, setFields] = useState<FormField[]>([])
  const [currentFieldIndex, setCurrentFieldIndex] = useState(0)
  const [typingValue, setTypingValue] = useState('')
  const [capturedLeads, setCapturedLeads] = useState(0)
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [showSuccess, setShowSuccess] = useState(false)
  const [isVisible, setIsVisible] = useState(false)
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  useEffect(() => {
    if (!isVisible) return

    const runAnimation = async () => {
      // Reset
      setFields([])
      setCurrentFieldIndex(0)
      setTypingValue('')
      setCapturedLeads(0)
      setIsSubmitting(false)
      setShowSuccess(false)
      await new Promise(resolve => setTimeout(resolve, 500))

      // Add fields one by one and type values
      for (let i = 0; i < fieldsData.length; i++) {
        // Add field
        setCurrentFieldIndex(i)
        setFields(prev => [...prev, { ...fieldsData[i], value: '', validated: false }])
        await new Promise(resolve => setTimeout(resolve, 400))

        // Type value character by character
        const targetValue = fieldsData[i].value
        for (let j = 0; j <= targetValue.length; j++) {
          setTypingValue(targetValue.slice(0, j))
          setFields(prev => prev.map((field, idx) => 
            idx === i ? { ...field, value: targetValue.slice(0, j) } : field
          ))
          await new Promise(resolve => setTimeout(resolve, 50))
        }

        await new Promise(resolve => setTimeout(resolve, 200))

        // Validate field
        setFields(prev => prev.map((field, idx) => 
          idx === i ? { ...field, validated: true } : field
        ))
        await new Promise(resolve => setTimeout(resolve, 400))
      }

      // Submit animation
      await new Promise(resolve => setTimeout(resolve, 600))
      setIsSubmitting(true)
      await new Promise(resolve => setTimeout(resolve, 800))

      // Lead captured
      setIsSubmitting(false)
      setShowSuccess(true)
      setCapturedLeads(1)
      await new Promise(resolve => setTimeout(resolve, 1000))

      // Show more leads being captured
      for (let i = 2; i <= 5; i++) {
        await new Promise(resolve => setTimeout(resolve, 400))
        setCapturedLeads(i)
      }

      await new Promise(resolve => setTimeout(resolve, 1500))

      // Restart
      runAnimation()
    }

    runAnimation()
  }, [isVisible])

  return (
    <div 
      ref={panelRef}
      className={`relative w-full max-w-2xl h-[400px] md:h-[550px] mx-auto flex items-center justify-center overflow-visible transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Left: Form */}
      <motion.div
        className="absolute left-0 md:left-12 w-48 md:w-64"
        initial={{ opacity: 0, x: -100 }}
        animate={{ opacity: 1, x: 0 }}
        transition={{ type: 'spring', stiffness: 200, damping: 20 }}
      >
        <div className="bg-white/90 dark:bg-[#1a1a1a]/90 backdrop-blur-md rounded-xl md:rounded-2xl border-2 border-gray-300 dark:border-gray-700 shadow-2xl p-4 md:p-6">
          {/* Form Header */}
          <div className="mb-4 md:mb-6">
            <h3 className="text-sm md:text-base font-bold text-gray-800 dark:text-white mb-1">Lead Capture Form</h3>
            <p className="text-[10px] md:text-xs text-gray-600 dark:text-gray-400">Fill to get access</p>
          </div>

          {/* Form Fields */}
          <div className="space-y-3 md:space-y-4">
            <AnimatePresence mode="popLayout">
              {fields.map((field, index) => (
                <motion.div
                  key={field.id}
                  layout
                  initial={{ opacity: 0, y: -20, scale: 0.9 }}
                  animate={{ opacity: 1, y: 0, scale: 1 }}
                  exit={{ opacity: 0, scale: 0.9 }}
                  transition={{ type: 'spring', stiffness: 300, damping: 25 }}
                >
                  <div className="relative">
                    <label className="text-[10px] md:text-xs text-gray-600 dark:text-gray-400 mb-1 block font-medium">
                      {field.label}
                    </label>
                    <div className="relative flex items-center">
                      <div className="absolute left-2 md:left-3 text-gray-500 dark:text-gray-400">
                        {field.icon}
                      </div>
                      <input
                        type="text"
                        value={field.value}
                        readOnly
                        className="w-full pl-8 md:pl-10 pr-8 md:pr-10 py-1.5 md:py-2 bg-gray-100 dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg text-[11px] md:text-sm text-gray-800 dark:text-white focus:outline-none"
                      />
                      {field.validated && (
                        <motion.div
                          initial={{ scale: 0, rotate: -180 }}
                          animate={{ scale: 1, rotate: 0 }}
                          transition={{ type: 'spring', stiffness: 300 }}
                          className="absolute right-2 md:right-3"
                        >
                          <CheckCircle2 className="w-4 h-4 md:w-5 md:h-5 text-green-600 dark:text-green-400" />
                        </motion.div>
                      )}
                      {index === currentFieldIndex && !field.validated && field.value.length < fieldsData[index].value.length && (
                        <motion.div
                          className="absolute right-2 md:right-3 w-0.5 h-4 bg-gray-600 dark:bg-gray-300"
                          animate={{ opacity: [1, 0] }}
                          transition={{ duration: 0.8, repeat: Infinity }}
                        />
                      )}
                    </div>
                  </div>
                </motion.div>
              ))}
            </AnimatePresence>
          </div>

          {/* Submit Button */}
          {fields.length === fieldsData.length && (
            <motion.button
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.3 }}
              className={`mt-4 md:mt-6 w-full py-2 md:py-3 rounded-lg font-semibold text-xs md:text-sm transition-all ${
                isSubmitting
                  ? 'bg-gray-400 dark:bg-gray-600 text-gray-200 dark:text-gray-400 cursor-wait'
                  : showSuccess
                  ? 'bg-green-600 dark:bg-green-700 text-white'
                  : 'bg-gray-800 dark:bg-gray-200 text-white dark:text-gray-900 hover:bg-gray-700 dark:hover:bg-gray-300'
              }`}
            >
              {isSubmitting ? 'Submitting...' : showSuccess ? '✓ Submitted' : 'Submit Form'}
            </motion.button>
          )}
        </div>
      </motion.div>

      {/* Center: Flow Arrow */}
      <AnimatePresence>
        {showSuccess && (
          <motion.div
            className="absolute left-1/2 -translate-x-1/2"
            initial={{ opacity: 0, scale: 0 }}
            animate={{ opacity: 1, scale: 1 }}
            exit={{ opacity: 0, scale: 0 }}
            transition={{ type: 'spring', stiffness: 300 }}
          >
            <motion.div
              animate={{ x: [0, 10, 0] }}
              transition={{ duration: 1, repeat: Infinity, ease: 'easeInOut' }}
            >
              <svg width="60" height="40" viewBox="0 0 60 40" className="text-gray-600 dark:text-gray-400">
                <motion.path
                  d="M 0 20 L 60 20"
                  stroke="currentColor"
                  strokeWidth="3"
                  fill="none"
                  strokeDasharray="5 5"
                  initial={{ pathLength: 0 }}
                  animate={{ pathLength: 1 }}
                  transition={{ duration: 1, repeat: Infinity }}
                />
              </svg>
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Right: Database */}
      <motion.div
        className="absolute right-0 md:right-12 w-48 md:w-64"
        initial={{ opacity: 0, x: 100 }}
        animate={{ opacity: 1, x: 0 }}
        transition={{ type: 'spring', stiffness: 200, damping: 20 }}
      >
        <div className="bg-white/90 dark:bg-[#1a1a1a]/90 backdrop-blur-md rounded-xl md:rounded-2xl border-2 border-gray-300 dark:border-gray-700 shadow-2xl p-4 md:p-6">
          {/* Database Icon */}
          <div className="flex flex-col items-center mb-4 md:mb-6">
            <motion.div
              animate={capturedLeads > 0 ? {
                scale: [1, 1.1, 1],
                rotate: [0, 5, -5, 0]
              } : {}}
              transition={{ duration: 0.4 }}
              className="w-16 h-16 md:w-20 md:h-20 mb-3 bg-gray-200 dark:bg-gray-800 rounded-2xl flex items-center justify-center"
            >
              <Database className="w-8 h-8 md:w-10 md:h-10 text-gray-700 dark:text-gray-300" />
            </motion.div>
            <h3 className="text-sm md:text-base font-bold text-gray-800 dark:text-white">Leads Database</h3>
            <p className="text-[10px] md:text-xs text-gray-600 dark:text-gray-400">Secure storage</p>
          </div>

          {/* Counter */}
          <div className="text-center">
            <motion.div
              key={capturedLeads}
              initial={{ scale: 1.5, opacity: 0 }}
              animate={{ scale: 1, opacity: 1 }}
              transition={{ type: 'spring', stiffness: 300 }}
              className="text-3xl md:text-5xl font-bold text-gray-800 dark:text-white mb-2"
            >
              {capturedLeads}
            </motion.div>
            <p className="text-xs md:text-sm text-gray-600 dark:text-gray-400">Leads Captured</p>
          </div>

          {/* Growth Indicator */}
          {capturedLeads > 1 && (
            <motion.div
              initial={{ opacity: 0, y: 10 }}
              animate={{ opacity: 1, y: 0 }}
              className="mt-3 md:mt-4 flex items-center justify-center gap-2 text-green-600 dark:text-green-400"
            >
              <motion.div
                animate={{ y: [-2, 2, -2] }}
                transition={{ duration: 1, repeat: Infinity }}
              >
                ▲
              </motion.div>
              <span className="text-xs md:text-sm font-semibold">Growing</span>
            </motion.div>
          )}
        </div>
      </motion.div>

      {/* Success Overlay */}
      <AnimatePresence>
        {capturedLeads >= 5 && (
          <motion.div
            className="absolute top-0 md:top-8 left-1/2 -translate-x-1/2 z-30"
            initial={{ opacity: 0, y: -50, scale: 0.8 }}
            animate={{ opacity: 1, y: 0, scale: 1 }}
            exit={{ opacity: 0, y: -50, scale: 0.8 }}
            transition={{ type: 'spring', stiffness: 300, damping: 20 }}
          >
            <div className="bg-white dark:bg-[#1a1a1a] border-2 border-gray-800 dark:border-gray-200 rounded-xl md:rounded-2xl px-4 md:px-6 py-2 md:py-3 shadow-xl">
              <div className="flex items-center gap-2 md:gap-3">
                <CheckCircle2 className="w-5 h-5 md:w-6 md:h-6 text-gray-800 dark:text-white" />
                <span className="text-sm md:text-base font-bold text-gray-800 dark:text-white">
                  Audience Growing!
                </span>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  )
}
