"use client";

import { useState, useEffect, useRef } from "react";

interface Square {
  id: number;
  opacity: number;
}

export default function InteractiveGrid() {
  const [mousePosition, setMousePosition] = useState({ x: -200, y: -200 });
  const [isMouseInside, setIsMouseInside] = useState(false);
  const [squares, setSquares] = useState<Square[]>([]);
  const gridRef = useRef<HTMLDivElement>(null);

  // Three opacity levels: 0.4, 0.6, 0.8
  const opacityLevels = [0.4, 0.6, 0.8];

  // Generate squares with 3 repeating opacity levels
  useEffect(() => {
    const desktopSquares = 24 * 12; // 288 squares (reduced by half)
    const generatedSquares: Square[] = [];
    
    for (let i = 0; i < desktopSquares; i++) {
      generatedSquares.push({
        id: i,
        opacity: opacityLevels[i % 3], // Cycles through 3 levels
      });
    }
    
    setSquares(generatedSquares);
  }, []);

  // Track mouse position relative to grid
  const handleMouseMove = (e: React.MouseEvent<HTMLDivElement>) => {
    if (gridRef.current) {
      const rect = gridRef.current.getBoundingClientRect();
      setMousePosition({
        x: e.clientX - rect.left,
        y: e.clientY - rect.top,
      });
      setIsMouseInside(true);
    }
  };

  const handleMouseLeave = () => {
    setIsMouseInside(false);
    setMousePosition({ x: -200, y: -200 });
  };

  // Calculate border opacity based on distance from cursor
  const getBorderOpacity = (index: number, cols: number) => {
    if (!gridRef.current) return 0.05; // Default 5% opacity
    
    if (!isMouseInside) return 0.05; // 5% when mouse not inside
    
    const rect = gridRef.current.getBoundingClientRect();
    const squareWidth = rect.width / cols;
    const squareHeight = rect.height / (squares.length / cols);
    
    const col = index % cols;
    const row = Math.floor(index / cols);
    
    const squareCenterX = col * squareWidth + squareWidth / 2;
    const squareCenterY = row * squareHeight + squareHeight / 2;
    
    const distance = Math.sqrt(
      Math.pow(mousePosition.x - squareCenterX, 2) +
      Math.pow(mousePosition.y - squareCenterY, 2)
    );
    
    if (distance > 200) return 0.05; // 5% when outside hover radius
    
    // Smooth fade: closer = more visible (from 5% to 45%)
    const hoverOpacity = 0.05 + (1 - distance / 200) * 0.4; // 5% base + up to 40% = 45% max
    return hoverOpacity;
  };

  return (
    <div className="w-full flex items-center justify-center px-6 md:px-6">
      <div
        ref={gridRef}
        onMouseMove={handleMouseMove}
        onMouseLeave={handleMouseLeave}
        className="relative rounded-3xl overflow-hidden w-full"
        style={{
          backgroundColor: "#1A1B1E",
          maxWidth: "1233px",
          height: "616.5px",
          borderTop: "1px solid rgba(255, 255, 255, 0.2)",
          borderLeft: "1px solid rgba(255, 255, 255, 0.2)",
          borderRight: "1px solid transparent",
          borderBottom: "1px solid transparent",
        }}
      >
        {/* Desktop Grid: 24 columns × 12 rows */}
        <div 
          className="hidden md:grid w-full h-full absolute inset-0" 
          style={{ 
            gridTemplateColumns: 'repeat(24, 1fr)', 
            gridTemplateRows: 'repeat(12, 1fr)' 
          }}
        >
          {squares.map((square, index) => {
            const borderOpacity = getBorderOpacity(index, 24);
            return (
              <div
                key={square.id}
                className="w-full h-full transition-all duration-300 ease-out"
                style={{
                  backgroundColor: "#1A1B1E",
                  border: `1px solid rgba(255, 255, 255, ${borderOpacity})`,
                }}
              />
            );
          })}
        </div>

        {/* Mobile Grid: 12 columns × 24 rows */}
        <div 
          className="grid md:hidden w-full h-full absolute inset-0" 
          style={{ 
            gridTemplateColumns: 'repeat(12, 1fr)', 
            gridTemplateRows: 'repeat(24, 1fr)',
          }}
        >
          {squares.map((square, index) => {
            const borderOpacity = getBorderOpacity(index, 12);
            return (
              <div
                key={square.id}
                className="w-full h-full transition-all duration-300 ease-out"
                style={{
                  backgroundColor: "#1A1B1E",
                  border: `1px solid rgba(255, 255, 255, ${borderOpacity})`,
                }}
              />
            );
          })}
        </div>

        {/* Content Section - Layered on top */}
        <div className="relative z-10 w-full h-full flex items-center justify-center p-6 md:p-10">
          {/* Three Animated Slashes - Background Layer */}
          <div className="absolute inset-0 flex items-center justify-center gap-8 md:gap-32">
            <div className="w-[12px] md:w-[20px] h-[200px] md:h-[380px] bg-white dark:bg-black rounded-full transform rotate-[25deg] animate-blink" style={{ animationDelay: '0s' }} />
            <div className="w-[12px] md:w-[20px] h-[200px] md:h-[380px] bg-white dark:bg-black rounded-full transform rotate-[25deg] animate-blink" style={{ animationDelay: '1s' }} />
            <div className="w-[12px] md:w-[20px] h-[200px] md:h-[380px] bg-white dark:bg-black rounded-full transform rotate-[25deg] animate-blink" style={{ animationDelay: '2s' }} />
          </div>

          {/* Glassmorphism Card - Independent */}
          <div className="relative z-10 w-full max-w-[90%] md:max-w-[850px] mx-auto">
            <div className="bg-white/80 dark:bg-black/30 backdrop-blur-xl border border-[#253237]/30 dark:border-white/10 rounded-2xl md:rounded-[40px] shadow-2xl transition-all duration-500 p-4 md:p-10">
              
              {/* Avatar Stack */}
              <div className="flex justify-center mb-3 md:mb-7">
                <div className="flex -space-x-3 md:-space-x-4">
                  <img src="https://images.unsplash.com/photo-1573496359142-b8d87734a5a2?w=100&h=100&fit=crop" alt="User" className="w-8 h-8 md:w-14 md:h-14 rounded-full border-2 md:border-3 border-white dark:border-black object-cover shadow-lg transition-transform duration-300 hover:scale-110 hover:z-10" />
                  <img src="https://images.unsplash.com/photo-1560250097-0b93528c311a?w=100&h=100&fit=crop" alt="User" className="w-8 h-8 md:w-14 md:h-14 rounded-full border-2 md:border-3 border-white dark:border-black object-cover shadow-lg transition-transform duration-300 hover:scale-110 hover:z-10" />
                  <img src="https://images.unsplash.com/photo-1573497019940-1c28c88b4f3e?w=100&h=100&fit=crop" alt="User" className="w-8 h-8 md:w-14 md:h-14 rounded-full border-2 md:border-3 border-white dark:border-black object-cover shadow-lg transition-transform duration-300 hover:scale-110 hover:z-10" />
                  <img src="https://images.unsplash.com/photo-1519345182560-3f2917c472ef?w=100&h=100&fit=crop" alt="User" className="w-8 h-8 md:w-14 md:h-14 rounded-full border-2 md:border-3 border-white dark:border-black object-cover shadow-lg transition-transform duration-300 hover:scale-110 hover:z-10" />
                  <img src="https://images.unsplash.com/photo-1580489944761-15a19d654956?w=100&h=100&fit=crop" alt="User" className="w-8 h-8 md:w-14 md:h-14 rounded-full border-2 md:border-3 border-white dark:border-black object-cover shadow-lg transition-transform duration-300 hover:scale-110 hover:z-10" />
                  <img src="https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=100&h=100&fit=crop" alt="User" className="w-8 h-8 md:w-14 md:h-14 rounded-full border-2 md:border-3 border-white dark:border-black object-cover shadow-lg transition-transform duration-300 hover:scale-110 hover:z-10" />
                </div>
              </div>

              {/* Main Heading */}
              <h2 className="text-center text-xl md:text-[44px] font-bold text-[#253237] dark:text-white leading-tight mb-2 md:mb-3">
                51,000+ Businesses Trust Us
              </h2>

              {/* Stats Row */}
              <div className="grid grid-cols-3 gap-2 md:gap-8 mb-3 md:mb-6">
                <div className="text-center">
                  <div className="flex items-center justify-center gap-0.5 md:gap-1 mb-1 md:mb-1.5">
                    {[...Array(5)].map((_, i) => (
                      <svg key={i} className="w-2.5 h-2.5 md:w-4 md:h-4 text-[#253237] dark:text-white" fill="currentColor" viewBox="0 0 20 20">
                        <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                      </svg>
                    ))}
                  </div>
                  <div className="text-xl md:text-4xl font-bold text-[#253237] dark:text-white">4.9</div>
                  <div className="text-[10px] md:text-sm text-gray-600 dark:text-gray-400 font-medium">Rating</div>
                </div>
                
                <div className="text-center border-x border-[#253237]/20 dark:border-white/20">
                  <div className="text-xl md:text-4xl font-bold text-[#253237] dark:text-white mb-1">150+</div>
                  <div className="text-[10px] md:text-sm text-gray-600 dark:text-gray-400 font-medium">Countries</div>
                </div>
                
                <div className="text-center">
                  <div className="text-xl md:text-4xl font-bold text-[#253237] dark:text-white mb-1">99%</div>
                  <div className="text-[10px] md:text-sm text-gray-600 dark:text-gray-400 font-medium">Satisfied</div>
                </div>
              </div>

              {/* Bottom Text */}
              <div className="flex items-center justify-center gap-1.5 md:gap-2 text-xs md:text-base text-gray-600 dark:text-gray-400 font-medium">
                <span>Growing businesses worldwide</span>
                <svg className="w-4 h-4 md:w-5 md:h-5 text-[#253237] dark:text-white" fill="currentColor" viewBox="0 0 20 20">
                  <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                </svg>
              </div>
            </div>
          </div>
        </div>

        <style jsx>{`
          @keyframes blink {
            0%, 100% { 
              background-color: white;
            }
            50% { 
              background-color: black;
            }
          }
          
          @media (prefers-color-scheme: dark) {
            @keyframes blink {
              0%, 100% { 
                background-color: black;
              }
              50% { 
                background-color: white;
              }
            }
          }
          
          .animate-blink {
            animation: blink 6s ease-in-out infinite;
          }
        `}</style>
      </div>
    </div>
  );
}
