"use client";

import Link from "next/link";
import Image from "next/image";
import { CheckCircle2 } from "lucide-react";
import { FaStar } from "react-icons/fa";
import { useTheme } from "../atoms/theme";
import { motion, AnimatePresence } from "framer-motion";
import logosDark from "../../public/images/logosDark.png";
import logosLight from "../../public/images/logosLight.png";

interface HeroSectionProps {
  announcementOpen?: boolean;
  badge?: {
    text: string;
    showPulse?: boolean;
    href?: string;
  };
  title:
    | {
        beforeHighlight?: string;
        highlight: string;
        afterHighlight?: string;
        animated?: false;
      }
    | {
        animated: true;
        headings: Array<{
          line1: React.ReactNode;
          line2: React.ReactNode;
        }>;
        currentIndex: number;
      };
  description: {
    desktop: React.ReactNode;
    mobile: React.ReactNode;
  };
  cta: {
    primary: {
      text: string;
      onClick?: () => void;
      href?: string;
    };
    secondary?: {
      text: string;
      href: string;
    };
  };
  trustIndicators?: {
    icon?: React.ComponentType<{ className?: string }>;
    text: string;
  }[];
  additionalContent?: React.ReactNode;
}

export default function HeroSection({
  announcementOpen = false,
  badge = {
    text: "Built by creators, for creators",
    showPulse: true,
  },
  title,
  description,
  cta,
  trustIndicators = [
    { text: "Start free" },
    { text: "Pay as you use" },
    { text: "Cancel anytime" },
  ],
  additionalContent,
}: HeroSectionProps) {
  const { theme } = useTheme();

  const handlePrimaryClick = () => {
    if (cta.primary.onClick) {
      cta.primary.onClick();
    } else if (cta.primary.href) {
      window.location.replace(cta.primary.href);
    }
  };

  const isAnimated = title.animated === true;

  return (
    <section
      className={`px-4 md:px-6 ${announcementOpen ? "pt-[14rem]" : "pt-[108px] md:pt-[156px]"} pb-12 md:pb-7 max-w-7xl mx-auto container relative overflow-hidden`}
    >
      {/* Background gradient effects */}
      <div className="absolute inset-0 -z-10">
        <div className="absolute top-0 left-1/4 w-96 h-96 bg-[radial-gradient(circle,_rgba(37,50,55,0.08)_0%,transparent_70%)] dark:bg-[radial-gradient(circle,_rgba(255,255,255,0.03)_0%,transparent_70%)]" />
        <div className="absolute top-20 right-1/4 w-96 h-96 bg-[radial-gradient(circle,_rgba(37,50,55,0.06)_0%,transparent_70%)] dark:bg-[radial-gradient(circle,_rgba(255,255,255,0.02)_0%,transparent_70%)]" />
      </div>

      <div className="text-center max-w-5xl mx-auto">
        {/* Badge */}
        {badge.href ? (
          <Link href={badge.href} className="inline-block">
            <div className="inline-flex items-center gap-2 bg-gray-100 dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E] rounded-full px-3 md:px-4 py-1.5 md:py-1 mb-4 md:mb-6">
              {badge.showPulse && (
                <span className="relative flex h-2 w-2">
                  <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-[#253237] dark:bg-white opacity-75 scale-125"></span>
                  <span className="relative inline-flex rounded-full h-2 w-2 bg-[#253237] dark:bg-white"></span>
                </span>
              )}
              <p className="text-xs md:text-sm text-gray-700 dark:text-gray-300 font-medium">
                {badge.text}
              </p>
            </div>
          </Link>
        ) : (
          <div className="inline-flex items-center gap-2 bg-gray-100 dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E] rounded-full px-3 md:px-4 py-1.5 md:py-2 mb-4 md:mb-6">
            {badge.showPulse && (
              <span className="relative flex h-2 w-2">
                <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-[#253237] dark:bg-white opacity-75 scale-125"></span>
                <span className="relative inline-flex rounded-full h-2 w-2 bg-[#253237] dark:bg-white"></span>
              </span>
            )}
            <p className="text-xs md:text-sm text-gray-700 dark:text-gray-300 font-medium">
              {badge.text}
            </p>
          </div>
        )}

        {/* Title - Static or Animated */}
        {isAnimated ? (
          <h1 className="text-3xl sm:text-4xl md:text-5xl lg:text-[3.5rem] font-bold font-eina text-[#253237] dark:text-white mb-4 sm:mb-6 leading-[1.2] relative min-h-[120px] sm:min-h-[150px] md:min-h-[180px] w-full text-center px-2">
            <AnimatePresence mode="wait">
              <motion.span
                key={title.currentIndex}
                initial={{ opacity: 0, y: 30 }}
                animate={{ opacity: 1, y: 0 }}
                exit={{ opacity: 0, y: -30 }}
                transition={{
                  duration: 0.6,
                  ease: [0.25, 0.46, 0.45, 0.94],
                }}
                className="inline-block w-full"
              >
                {title.headings[title.currentIndex].line1}
                <br />
                {title.headings[title.currentIndex].line2}
              </motion.span>
            </AnimatePresence>
          </h1>
        ) : (
          <h1 className="text-[28px] sm:text-[36px] md:text-[4rem] lg:text-[4.5rem] font-extrabold text-gray-900 dark:text-white leading-[1.1] mb-4 md:mb-6 px-2">
            {title.beforeHighlight && <>{title.beforeHighlight} </>}
            <span className="relative inline-block">
              <span className="text-[#253237] dark:text-white">{title.highlight}</span>
              <svg
                className="absolute -bottom-1 md:-bottom-2 left-0 w-full"
                height="8"
                viewBox="0 0 200 8"
                fill="none"
                xmlns="http://www.w3.org/2000/svg"
              >
                <path
                  d="M1 5.5C50 2.5 150 2.5 199 5.5"
                  stroke="#253237"
                  className="dark:stroke-white"
                  strokeWidth="3"
                  strokeLinecap="round"
                />
              </svg>
            </span>
            {title.afterHighlight && (
              <>
                <br className="hidden md:block" />
                {title.afterHighlight}
              </>
            )}
          </h1>
        )}

        {/* Description */}
        <p className="text-sm md:text-lg lg:text-xl text-gray-600 dark:text-gray-400 leading-relaxed max-w-3xl mx-auto mb-6 md:mb-8 px-2">
          <span className="hidden md:inline">{description.desktop}</span>
          <span className="md:hidden">{description.mobile}</span>
        </p>

        {/* Trust Factor */}
        <div className="flex items-center gap-6 flex-wrap justify-center mb-6 md:mb-5">
          <div className="flex items-center gap-2">
            <div className="flex -space-x-2">
              <img src="https://i.pravatar.cc/150?img=12" alt="User" className="w-8 h-8 rounded-full border-2 border-white dark:border-black object-cover" />
              <img src="https://i.pravatar.cc/150?img=26" alt="User" className="w-8 h-8 rounded-full border-2 border-white dark:border-black object-cover" />
              <img src="https://i.pravatar.cc/150?img=47" alt="User" className="w-8 h-8 rounded-full border-2 border-white dark:border-black object-cover" />
            </div>
            <span className="text-xs font-semibold text-gray-900 dark:text-white">100,000+ users</span>
          </div>
          <div className="flex items-center gap-1">
            {[...Array(5)].map((_, i) => (
              <svg key={i} className="w-4 h-4 text-[#253237] dark:text-white" fill="currentColor" viewBox="0 0 20 20">
                <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
              </svg>
            ))}
            <span className="text-xs font-semibold text-gray-900 dark:text-white ml-1">4.9/5</span>
          </div>
        </div>

        {/* CTA Buttons */}
        <div className="flex flex-col sm:flex-row items-center justify-center gap-3 md:gap-4 mb-8 md:mb-12 px-2">
          <button
            onClick={handlePrimaryClick}
            className="w-full sm:w-auto bg-[#253237] dark:bg-white text-white dark:text-[#253237] px-6 md:px-8 py-3 md:py-4 rounded-lg font-semibold hover:bg-[#1a2428] dark:hover:bg-gray-100 transition-all shadow-lg hover:shadow-xl text-base md:text-lg"
          >
            {cta.primary.text}
          </button>
          {cta.secondary && (
            <Link
              href={cta.secondary.href}
              className="w-full sm:w-auto text-gray-900 dark:text-white px-6 md:px-8 py-3 md:py-4 rounded-lg font-semibold border-2 border-gray-300 dark:border-[#ffffff3a] hover:bg-gray-100 dark:hover:bg-[#16171B] transition-colors text-base md:text-lg text-center"
            >
              {cta.secondary.text}
            </Link>
          )}
        </div>
         

        {/* Additional Content (e.g., email signup, custom CTAs) */}
        {additionalContent && <div className="mb-8 md:mb-12">{additionalContent}</div>}

        {/* Trust indicators */}
        {trustIndicators && trustIndicators.length > 0 && (
          <div className="flex flex-wrap items-center justify-center gap-4 md:gap-8 text-xs md:text-sm text-gray-600 dark:text-gray-400 px-2">
            {trustIndicators.map((indicator, index) => {
              const IconComponent = indicator.icon || CheckCircle2;
              return (
                <div key={index} className="flex items-center gap-1.5 md:gap-2">
                  <IconComponent className="w-4 md:w-5 h-4 md:h-5 text-[#253237] dark:text-white flex-shrink-0" />
                  <span className="whitespace-nowrap">{indicator.text}</span>
                </div>
              );
            })}
          </div>
        )}
      </div>
    </section>
  );
}
