'use client'

import { useState, useEffect } from 'react'
import { Check, X, User } from 'lucide-react'
import { motion, AnimatePresence } from 'framer-motion'

const candidatesData = [
  {
    id: 1,
    name: 'Sarah Johnson',
    role: 'Senior Frontend Developer',
    skills: 'React, TypeScript, Tailwind',
    experience: '5 years'
  },
  {
    id: 2,
    name: 'Michael Chen',
    role: 'Product Designer',
    skills: 'UI/UX, Figma, Prototyping',
    experience: '4 years'
  },
  {
    id: 3,
    name: 'Emily Davis',
    role: 'Backend Engineer',
    skills: 'Node.js, PostgreSQL, AWS',
    experience: '6 years'
  },
  {
    id: 4,
    name: 'David Wilson',
    role: 'DevOps Engineer',
    skills: 'Docker, Kubernetes, CI/CD',
    experience: '3 years'
  }
]

function HRTinderCardStack() {
  const [cards, setCards] = useState(candidatesData)
  const [currentIndex, setCurrentIndex] = useState(0)
  const [direction, setDirection] = useState<'accept' | 'decline' | null>(null)

  useEffect(() => {
    const interval = setInterval(() => {
      // Alternate between accept and decline
      const isAccept = currentIndex % 2 === 0
      setDirection(isAccept ? 'accept' : 'decline')
      
      setTimeout(() => {
        setCurrentIndex((prev) => prev + 1)
        setDirection(null)
        
        // Move the card to the end of the array
        setCards((prevCards) => {
          const newCards = [...prevCards]
          const removedCard = newCards.shift()
          if (removedCard) {
            newCards.push(removedCard)
          }
          return newCards
        })
      }, 1200)
    }, 1250)

    return () => clearInterval(interval)
  }, [currentIndex])

  return (
    <div className="relative w-full h-[400px] flex items-center justify-center">
      <AnimatePresence>
        {cards.slice(0, 4).map((candidate, index) => {
          const isTop = index === 0
          const zIndex = 4 - index
          
          // Calculate scale and position for stacked effect
          const scale = 1 - index * 0.05
          const yOffset = index * 10
          
          return (
            <motion.div
              key={`${candidate.id}-${currentIndex}`}
              className="absolute w-60 h-72 md:w-80 md:h-96"
              style={{
                zIndex,
              }}
              initial={
                index === 0 
                  ? { x: 300, opacity: 0, scale: 0.8, rotate: 10 }
                  : { scale, y: yOffset }
              }
              animate={
                isTop && direction
                  ? direction === 'accept'
                    ? { 
                        x: 400, 
                        y: -150, 
                        rotate: 20, 
                        opacity: 0,
                        scale: 0.9
                      }
                    : { 
                        x: -400, 
                        y: 150, 
                        rotate: -20, 
                        opacity: 0,
                        scale: 0.9
                      }
                  : { 
                      x: 0, 
                      y: yOffset, 
                      scale, 
                      rotate: 0, 
                      opacity: 1 
                    }
              }
              transition={{
                type: 'spring',
                stiffness: 260,
                damping: 20,
                opacity: { duration: 0.3 }
              }}
            >
              {/* Card with glassmorphism */}
              <div className={`relative w-full h-full rounded-2xl overflow-hidden shadow-lg border transition-all duration-300 ${
                isTop && direction === 'accept'
                  ? 'border-green-500 shadow-green-500/50'
                  : isTop && direction === 'decline'
                  ? 'border-red-500 shadow-red-500/50'
                  : 'border-gray-200/50 dark:border-white/10'
              }`}>
                {/* Glassmorphism background */}
                <div className="absolute inset-0 bg-white/80 dark:bg-[#1a1a1a]/80 backdrop-blur-md" />
                
                {/* Green/Red overlay on decision */}
                {isTop && direction && (
                  <motion.div
                    className={`absolute inset-0 ${
                      direction === 'accept' ? 'bg-green-500/30' : 'bg-red-500/30'
                    }`}
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    transition={{ duration: 0.3 }}
                  />
                )}

                {/* Decision Icon */}
                {isTop && direction && (
                  <motion.div
                    className={`absolute top-4 right-4 md:top-8 md:right-8 w-14 h-14 md:w-20 md:h-20 rounded-full flex items-center justify-center z-20 ${
                      direction === 'accept' 
                        ? 'bg-green-500 shadow-lg shadow-green-500/50' 
                        : 'bg-red-500 shadow-lg shadow-red-500/50'
                    }`}
                    initial={{ scale: 0, rotate: -180 }}
                    animate={{ scale: 1, rotate: 0 }}
                    transition={{ 
                      type: 'spring',
                      stiffness: 300,
                      damping: 20
                    }}
                  >
                    {direction === 'accept' ? (
                      <Check className="w-7 h-7 md:w-10 md:h-10 text-white" strokeWidth={3} />
                    ) : (
                      <X className="w-7 h-7 md:w-10 md:h-10 text-white" strokeWidth={3} />
                    )}
                  </motion.div>
                )}

                {/* Card Content */}
                <div className="relative h-full p-4 md:p-6 flex flex-col">
                  {/* Profile Icon */}
                  <div className="flex-1 flex items-center justify-center mb-3 md:mb-6">
                    <div className="w-28 h-28 md:w-40 md:h-40 rounded-full flex items-center justify-center bg-gray-100 dark:bg-[#1a1a1a] border-2 md:border-4 border-gray-200 dark:border-white/20 shadow-xl">
                      <User className="w-14 h-14 md:w-20 md:h-20 text-gray-900 dark:text-gray-300" strokeWidth={1.5} />
                    </div>
                  </div>

                  {/* Candidate Info */}
                  <div className="text-center space-y-2 md:space-y-3">
                    <h3 className="text-lg md:text-2xl font-bold text-gray-900 dark:text-white">
                      {candidate.name}
                    </h3>
                    <p className="text-sm md:text-lg text-gray-700 dark:text-gray-300">
                      {candidate.role}
                    </p>
                    <div className="pt-2 md:pt-4 space-y-1 md:space-y-2">
                      <div className="flex items-center justify-center gap-1 md:gap-2 text-xs md:text-sm">
                        <span className="text-gray-600 dark:text-gray-400">Skills:</span>
                        <span className="text-gray-800 dark:text-gray-200">{candidate.skills}</span>
                      </div>
                      <div className="flex items-center justify-center gap-1 md:gap-2 text-xs md:text-sm">
                        <span className="text-gray-600 dark:text-gray-400">Experience:</span>
                        <span className="text-gray-800 dark:text-gray-200">{candidate.experience}</span>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </motion.div>
          )
        })}
      </AnimatePresence>
    </div>
  )
}

export default function HRAnimationPanel() {
  return (
    <div className="relative flex items-center justify-center w-full h-full">
      <div className="absolute w-[380px] h-[380px] rounded-full bg-[radial-gradient(circle,_rgba(0,0,0,0.15)_0%,rgba(31,98,255,0.1)_30%,_rgba(0,0,0,0)_80%)] dark:bg-[radial-gradient(circle,_rgba(255,255,255,0.15)_0%,rgba(165,197,255,0.1)_30%,_rgba(255,255,255,0)_80%)] filter blur-3xl" />
      <div className="relative z-10">
        <HRTinderCardStack />
      </div>
    </div>
  )
}
