'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { Clock, Smartphone, MessageCircle } from 'lucide-react'
import Image from 'next/image'

interface CustomerMessage {
  id: number
  name: string
  location: string
  time: string
  channel: 'web' | 'mobile' | 'chat'
  message: string
  avatarUrl: string
  color: string
}

const messagesData: CustomerMessage[] = [
  {
    id: 1,
    name: 'Alex Johnson',
    location: 'New York, USA',
    time: '2:30 PM',
    channel: 'web',
    message: 'Need help with my order',
    avatarUrl: 'https://images.unsplash.com/photo-1500648767791-00dcc994a43e?w=100&h=100&fit=crop&crop=faces',
    color: 'bg-gray-300 dark:bg-gray-600'
  },
  {
    id: 2,
    name: 'Yuki Tanaka',
    location: 'Tokyo, Japan',
    time: '3:30 AM',
    channel: 'mobile',
    message: 'Product inquiry',
    avatarUrl: 'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=100&h=100&fit=crop&crop=faces',
    color: 'bg-gray-200 dark:bg-gray-500'
  },
  {
    id: 3,
    name: 'Sophie Martin',
    location: 'Paris, France',
    time: '8:30 PM',
    channel: 'chat',
    message: 'Question about shipping',
    avatarUrl: 'https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=100&h=100&fit=crop&crop=faces',
    color: 'bg-gray-400 dark:bg-gray-700'
  }
]

const getChannelIcon = (channel: string) => {
  switch (channel) {
    case 'web':
      return <svg className="w-3 h-3 md:w-4 md:h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 12a9 9 0 01-9 9m9-9a9 9 0 00-9-9m9 9H3m9 9a9 9 0 01-9-9m9 9c1.657 0 3-4.03 3-9s-1.343-9-3-9m0 18c-1.657 0-3-4.03-3-9s1.343-9 3-9m-9 9a9 9 0 019-9" /></svg>
    case 'mobile':
      return <Smartphone className="w-3 h-3 md:w-4 md:h-4" />
    case 'chat':
      return <MessageCircle className="w-3 h-3 md:w-4 md:h-4" />
    default:
      return <svg className="w-3 h-3 md:w-4 md:h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 12a9 9 0 01-9 9m9-9a9 9 0 00-9-9m9 9H3m9 9a9 9 0 01-9-9m9 9c1.657 0 3-4.03 3-9s-1.343-9-3-9m0 18c-1.657 0-3-4.03-3-9s1.343-9 3-9m-9 9a9 9 0 019-9" /></svg>
  }
}

export default function GlobalSupportAnimationPanel() {
  const [messages, setMessages] = useState<CustomerMessage[]>([])
  const [isVisible, setIsVisible] = useState(false)
  const messageCounterRef = useRef(0)
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer to detect when component is visible
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  // Animation logic - only starts when visible
  useEffect(() => {
    if (!isVisible) return

    let messageIndex = 0
    let timerId: NodeJS.Timeout

    const addMessage = () => {
      if (messageIndex < messagesData.length) {
        const messageWithUniqueId = {
          ...messagesData[messageIndex],
          id: messageCounterRef.current++
        }
        setMessages((prev) => [...prev, messageWithUniqueId])
        messageIndex++
        timerId = setTimeout(addMessage, 900)
      } else {
        // Hold for 1.5 seconds then reset
        timerId = setTimeout(() => {
          setMessages([])
          messageIndex = 0
          timerId = setTimeout(addMessage, 500)
        }, 1500)
      }
    }

    timerId = setTimeout(addMessage, 500)

    return () => clearTimeout(timerId)
  }, [isVisible])

  return (
    <div 
      ref={panelRef}
      className={`w-[310px] md:w-[450px] mx-auto mb-8 md:mb-0 transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Support Dashboard Window */}
      <div className="w-full bg-white dark:bg-[#1a1a1a] backdrop-blur-md rounded-xl md:rounded-2xl shadow-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
        {/* Header */}
        <div className="bg-gray-50 dark:bg-[#1a1a1a] px-3 md:px-5 py-2.5 md:py-4 border-b border-gray-200 dark:border-gray-700">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-2 md:gap-3">
              <h3 className="text-gray-700 dark:text-white font-semibold text-xs md:text-base">Global Support</h3>
            </div>
            
            {/* 24/7 indicator with pulse */}
            <div className="flex items-center gap-1.5 md:gap-2">
              <div className="relative">
                <div className="w-1.5 h-1.5 md:w-2 md:h-2 bg-green-500 rounded-full"></div>
                <div className="absolute inset-0 w-1.5 h-1.5 md:w-2 md:h-2 bg-green-500 rounded-full animate-ping"></div>
              </div>
              <span className="text-gray-600 dark:text-gray-300 text-[10px] md:text-sm">24/7 active</span>
            </div>
          </div>
        </div>

        {/* Messages Container */}
        <div className="p-3 md:p-5 space-y-2.5 md:space-y-4 h-[255px] md:h-[340px] overflow-hidden">
          <AnimatePresence mode="popLayout">
            {messages.map((message) => (
              <motion.div
                key={message.id}
                initial={{ x: -100, opacity: 0, scale: 0.8 }}
                animate={{ x: 0, opacity: 1, scale: 1 }}
                exit={{ opacity: 0, scale: 0.8 }}
                transition={{
                  type: 'spring',
                  stiffness: 400,
                  damping: 25
                }}
                className="flex items-start gap-3"
              >
                {/* Avatar */}
                <div className={`w-7 h-7 md:w-10 md:h-10 rounded-full ${message.color} flex items-center justify-center flex-shrink-0 overflow-hidden shadow-md relative`}>
                  <Image
                    src={message.avatarUrl}
                    alt={message.name}
                    fill
                    className="object-cover"
                    sizes="40px"
                  />
                </div>

                {/* Message Bubble */}
                <div className="flex-1">
                  <div className="flex items-baseline gap-1.5 md:gap-2 mb-0.5 md:mb-1">
                    <span className="text-gray-700 dark:text-white font-medium text-[10px] md:text-sm">{message.name}</span>
                    <span className="text-gray-500 dark:text-gray-400 text-[8px] md:text-xs">{message.time}</span>
                  </div>
                  
                  <div className="flex items-center gap-1.5 mb-1">
                    <span className="text-gray-500 dark:text-gray-400 text-[8px] md:text-xs">
                      {message.location}
                    </span>
                    <span className="text-gray-300 dark:text-gray-600">•</span>
                    <div className="flex items-center gap-1 text-gray-600 dark:text-gray-400">
                      {getChannelIcon(message.channel)}
                      <span className="text-[8px] md:text-xs capitalize">{message.channel}</span>
                    </div>
                  </div>
                  
                  <div className="bg-gray-200 dark:bg-white/10 text-gray-800 dark:text-white px-2.5 md:px-4 py-1.5 md:py-2.5 rounded-xl md:rounded-2xl rounded-tl-sm shadow-sm">
                    <p className="text-[10px] md:text-sm leading-relaxed">
                      {message.message}
                    </p>
                  </div>
                </div>
              </motion.div>
            ))}
          </AnimatePresence>
        </div>
      </div>
    </div>
  )
}
