'use client'

import { useState, useEffect, useRef } from 'react'
import { Cloud, FolderOpen, Check } from 'lucide-react'
import { motion, useAnimation } from 'framer-motion'

interface Folder {
  id: number
  label: string
  initialX: number
  initialY: number
  initialRotate: number
}

const folders: Folder[] = [
  { id: 1, label: 'Payroll', initialX: -120, initialY: -80, initialRotate: -15 },
  { id: 2, label: 'Contracts', initialX: 140, initialY: -70, initialRotate: 12 },
  { id: 3, label: 'Resume', initialX: -130, initialY: 90, initialRotate: 8 },
  { id: 4, label: 'Legal', initialX: 150, initialY: 100, initialRotate: -10 }
]

export default function FilesAnimationPanel() {
  const [currentFolderIndex, setCurrentFolderIndex] = useState(0)
  const [isDragging, setIsDragging] = useState(false)
  const [isComplete, setIsComplete] = useState(false)
  const [droppedFolders, setDroppedFolders] = useState<number[]>([])
  const [cloudGlow, setCloudGlow] = useState(false)
  const [isVisible, setIsVisible] = useState(false)
  
  const cursorControls = useAnimation()
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer to detect when component is visible
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  useEffect(() => {
    if (!isVisible) return

    const animateSequence = async () => {
      if (currentFolderIndex < folders.length) {
        const folder = folders[currentFolderIndex]
        
        // Move cursor to folder
        await cursorControls.start({
          x: folder.initialX,
          y: folder.initialY,
          transition: { type: 'spring', stiffness: 100, damping: 15 }
        })
        
        // Grab the folder
        setIsDragging(true)
        await new Promise(resolve => setTimeout(resolve, 200))
        
        // Drag to center
        await cursorControls.start({
          x: 0,
          y: 0,
          transition: { type: 'spring', stiffness: 100, damping: 15, duration: 1 }
        })
        
        // Drop the folder
        setIsDragging(false)
        setDroppedFolders(prev => [...prev, currentFolderIndex])
        setCloudGlow(true)
        
        await new Promise(resolve => setTimeout(resolve, 400))
        setCloudGlow(false)
        
        await new Promise(resolve => setTimeout(resolve, 300))
        setCurrentFolderIndex(prev => prev + 1)
        
      } else if (currentFolderIndex === folders.length && !isComplete) {
        // All folders collected - show completion
        setIsComplete(true)
        await new Promise(resolve => setTimeout(resolve, 2000))
        
        // Reset
        setIsComplete(false)
        setDroppedFolders([])
        setCurrentFolderIndex(0)
        await cursorControls.start({ x: -200, y: -200 })
        await new Promise(resolve => setTimeout(resolve, 500))
      }
    }

    animateSequence()
  }, [currentFolderIndex, cursorControls, isComplete, isVisible])

  return (
    <div 
      ref={panelRef}
      className={`relative w-full max-w-lg h-[450px] mx-auto flex items-center justify-center transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Central Cloud Container */}
      <motion.div
        className={`relative w-64 h-64 rounded-3xl backdrop-blur-md border transition-all duration-300 flex items-center justify-center ${
          cloudGlow 
            ? 'bg-white/90 dark:bg-[#1a1a1a]/90 border-gray-400 dark:border-gray-500 shadow-[0_0_40px_rgba(0,0,0,0.3)] dark:shadow-[0_0_40px_rgba(255,255,255,0.2)]' 
            : 'bg-white/80 dark:bg-[#1a1a1a]/80 border-gray-300/60 dark:border-gray-700/50 shadow-xl'
        }`}
        animate={cloudGlow ? { scale: [1, 1.05, 1] } : {}}
        transition={{ duration: 0.4 }}
      >
        <div className="text-center">
          <Cloud className={`w-20 h-20 mx-auto mb-3 transition-colors duration-300 ${
            cloudGlow ? 'text-gray-700 dark:text-gray-300' : 'text-gray-500 dark:text-gray-400'
          }`} strokeWidth={1.5} />
          <p className="text-sm font-medium text-[#253237] dark:text-white">Wurknest Files</p>
          <p className="text-xs text-gray-600 dark:text-gray-400 mt-1">{droppedFolders.length}/4 files</p>
        </div>

        {/* Completion Checkmark */}
        {isComplete && (
          <motion.div
            className="absolute inset-0 flex items-center justify-center bg-gray-200/30 dark:bg-gray-700/30 rounded-3xl"
            initial={{ opacity: 0, scale: 0.5 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ type: 'spring', stiffness: 200 }}
          >
            <div className="w-24 h-24 bg-gray-800 dark:bg-gray-200 rounded-full flex items-center justify-center">
              <Check className="w-14 h-14 text-white dark:text-gray-900" strokeWidth={3} />
            </div>
          </motion.div>
        )}
      </motion.div>

      {/* Scattered Folders */}
      {folders.map((folder, index) => {
        const isBeingDragged = isDragging && index === currentFolderIndex
        const isDropped = droppedFolders.includes(index)

        if (isDropped && !isBeingDragged) return null

        return (
          <motion.div
            key={folder.id}
            className="absolute"
            initial={{
              x: folder.initialX,
              y: folder.initialY,
              rotate: folder.initialRotate,
              scale: 1,
              opacity: 1
            }}
            animate={
              isBeingDragged
                ? {
                    x: 0,
                    y: 0,
                    rotate: 0,
                    scale: 0.8,
                    opacity: 1
                  }
                : isDropped
                ? {
                    scale: 0,
                    opacity: 0,
                    transition: { duration: 0.3 }
                  }
                : {
                    x: folder.initialX,
                    y: folder.initialY,
                    rotate: folder.initialRotate,
                    scale: 1,
                    opacity: 1
                  }
            }
            transition={{ type: 'spring', stiffness: 100, damping: 15 }}
          >
            <div className="bg-white/90 dark:bg-[#1a1a1a]/80 border border-gray-300 dark:border-gray-700 rounded-lg p-3 shadow-lg backdrop-blur-sm">
              <FolderOpen className="w-10 h-10 text-gray-700 dark:text-gray-300 mb-1" />
              <p className="text-xs font-medium text-[#253237] dark:text-white whitespace-nowrap">{folder.label}</p>
            </div>
          </motion.div>
        )
      })}

      {/* Custom Cursor */}
      <motion.div
        className="absolute pointer-events-none z-50"
        animate={cursorControls}
        initial={{ x: -200, y: -200 }}
      >
        <svg
          width="24"
          height="24"
          viewBox="0 0 24 24"
          fill="none"
          className="drop-shadow-lg"
        >
          <path
            d="M3 3L10.07 19.97L12.58 12.58L19.97 10.07L3 3Z"
            fill="white"
            stroke="black"
            strokeWidth="1"
          />
        </svg>
      </motion.div>
    </div>
  )
}
