"use client";
import React, { useState, useRef, useMemo } from "react";

interface UploadedFile {
  id: string;
  name: string;
  size: number;
  type: string;
  file: File;
}

interface FileTransferFormProps {
  initialFiles?: UploadedFile[];
}

export default function FileTransferForm({
  initialFiles = [],
}: FileTransferFormProps) {
  const [tab, setTab] = useState<"email" | "link">("link");
  const [showSettingsModal, setShowSettingsModal] = useState(false);
  const [settings, setSettings] = useState({
    expiration: 7,
    password: "",
    notifyCreation: true,
    notifyDownloads: true,
    notifyNonDownload: true,
    previewFiles: true,
  });

  const expirationDate = useMemo(() => {
    // eslint-disable-next-line react-hooks/purity
    return new Date(Date.now() + settings.expiration * 24 * 60 * 60 * 1000);
  }, [settings.expiration]);
  const [uploadedFiles, setUploadedFiles] = useState<UploadedFile[]>(
    initialFiles.length > 0
      ? initialFiles
      : [
          {
            id: "1",
            name: "market-analysis-report (4).pdf",
            size: 142208,
            type: "application/pdf",
            file: new File([], "market-analysis-report (4).pdf", {
              type: "application/pdf",
            }),
          },
        ]
  );
  const [isDragOver, setIsDragOver] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return "0 Bytes";
    const k = 1024;
    const sizes = ["Bytes", "KB", "MB", "GB"];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
  };

  const getFileExtension = (filename: string) => {
    return "." + filename.split(".").pop()?.toLowerCase();
  };

  const handleFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = Array.from(event.target.files || []);
    const newFiles: UploadedFile[] = files.map((file) => ({
      id: Date.now() + Math.random().toString(),
      name: file.name,
      size: file.size,
      type: file.type,
      file: file,
    }));

    setUploadedFiles((prev) => [...prev, ...newFiles]);
  };

  const handleDrop = (event: React.DragEvent) => {
    event.preventDefault();
    setIsDragOver(false);

    const files = Array.from(event.dataTransfer.files);
    const newFiles: UploadedFile[] = files.map((file) => ({
      id: Date.now() + Math.random().toString(),
      name: file.name,
      size: file.size,
      type: file.type,
      file: file,
    }));

    setUploadedFiles((prev) => [...prev, ...newFiles]);
  };

  const handleDragOver = (event: React.DragEvent) => {
    event.preventDefault();
    setIsDragOver(true);
  };

  const handleDragLeave = (event: React.DragEvent) => {
    event.preventDefault();
    setIsDragOver(false);
  };

  const removeFile = (fileId: string) => {
    setUploadedFiles((prev) => prev.filter((file) => file.id !== fileId));
  };

  const handleEmailSubmit = (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();

    if (uploadedFiles.length === 0) {
      alert("Please select at least one file to send.");
      return;
    }

    const formData = new FormData(event.currentTarget);
    const email = formData.get("email") as string;
    const contacts = formData.get("contacts") as string;
    const title = formData.get("title") as string;

    alert(
      `Email would be sent to: ${contacts}\nFrom: ${email}\nSubject: ${
        title || "File Transfer"
      }\nFiles: ${
        uploadedFiles.length
      }\n\nThis is a demo - no actual email will be sent.`
    );
  };

  const handleLinkSubmit = (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();

    if (uploadedFiles.length === 0) {
      alert("Please select at least one file to create a link.");
      return;
    }

    const formData = new FormData(event.currentTarget);
    const title = formData.get("title") as string;
    const customUrl = formData.get("customUrl") as string;
    const requireEmail = formData.get("accessTracking") === "on";

    const linkUrl = customUrl
      ? `wurknest.com/${customUrl}`
      : `wurknest.com/${generateRandomId()}`;

    alert(
      `Link created successfully!\n\nURL: ${linkUrl}\nTitle: ${
        title || "Untitled"
      }\nFiles: ${uploadedFiles.length}\nRequire email: ${
        requireEmail ? "Yes" : "No"
      }\n\nThis is a demo - no actual link will be created.`
    );
  };

  const generateRandomId = () => {
    return (
      Math.random().toString(36).substring(2, 15) +
      Math.random().toString(36).substring(2, 15)
    );
  };

  const totalSize = uploadedFiles.reduce((sum, file) => sum + file.size, 0);

  return (
    <div
      className="prepare-subcontainer"
      style={{
        maxWidth: "800px",
        margin: "0 auto",
        display: "flex",
        gap: "20px",
        flexWrap: "wrap",
        padding: "20px",
      }}
    >
      <div
        className="upload-files-box"
        style={{
          flex: "1",
          minWidth: "300px",
          background: "white",
          borderRadius: "12px",
          boxShadow: "0 2px 12px rgba(0, 0, 0, 0.08)",
          overflow: "hidden",
        }}
      >
        <div
          className="upload-files-box-header"
          style={{
            display: "flex",
            justifyContent: "space-between",
            alignItems: "center",
            padding: "16px 20px",
            borderBottom: "1px solid #e9ecef",
            backgroundColor: "#fafbfc",
          }}
        >
          <div
            className="left-header"
            style={{ display: "flex", alignItems: "center", gap: "8px" }}
          >
            <span
              className="left-header-text-total-count"
              style={{ fontWeight: "600", color: "#2d3748" }}
            >
              {uploadedFiles.length} file{uploadedFiles.length !== 1 ? "s" : ""}
            </span>
            <span
              className="left-header-text-total-size"
              style={{ color: "#718096", fontSize: "14px" }}
            >
              ({formatFileSize(totalSize)})
            </span>
          </div>
          <div className="right-header">
            <button
              className="upload-files-box-add-button"
              onClick={() => fileInputRef.current?.click()}
              style={{
                width: "32px",
                height: "32px",
                borderRadius: "50%",
                backgroundColor: "#3b82f6",
                color: "white",
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
                fontSize: "18px",
                fontWeight: "bold",
                cursor: "pointer",
                border: "none",
                transition: "background-color 0.2s",
              }}
              onMouseEnter={(e) =>
                (e.currentTarget.style.backgroundColor = "#dc2626")
              }
              onMouseLeave={(e) =>
                (e.currentTarget.style.backgroundColor = "#ef4444")
              }
            >
              +
            </button>
          </div>
        </div>

        <div
          className="file-drop"
          onDrop={handleDrop}
          onDragOver={handleDragOver}
          onDragLeave={handleDragLeave}
          style={{
            backgroundColor: isDragOver ? "#f0f9ff" : "transparent",
            border: isDragOver ? "2px dashed #3b82f6" : "none",
            transition: "all 0.2s",
          }}
        >
          <div className="file-drop-container">
            <div
              className="file-drop-input-container"
              style={{ position: "relative" }}
            >
              <input
                type="file"
                ref={fileInputRef}
                className="file-drop-input"
                accept="*"
                multiple
                onChange={handleFileChange}
                style={{
                  position: "absolute",
                  top: 0,
                  left: 0,
                  width: "100%",
                  height: "100%",
                  opacity: 0,
                  cursor: "pointer",
                  zIndex: 10,
                }}
              />

              {uploadedFiles.length === 0 && (
                <div
                  style={{
                    textAlign: "center",
                    padding: "40px 20px",
                    color: "#718096",
                    fontSize: "14px",
                  }}
                >
                  Drag and drop files here or click the + button to select files
                </div>
              )}

              <div
                className="files-list-container"
                style={{ padding: "20px", minHeight: "120px" }}
              >
                <ul
                  className="files-list"
                  style={{ listStyle: "none", margin: 0, padding: 0 }}
                >
                  {uploadedFiles.map((file) => (
                    <li
                      key={file.id}
                      className="files-list-item"
                      style={{
                        display: "flex",
                        justifyContent: "space-between",
                        alignItems: "center",
                        padding: "12px 0",
                        borderBottom: "1px solid #e9ecef",
                      }}
                    >
                      <div className="meta-container" style={{ flex: 1 }}>
                        <div
                          className="title"
                          style={{
                            display: "flex",
                            alignItems: "center",
                            gap: "8px",
                            marginBottom: "4px",
                          }}
                        >
                          <div
                            className="title-icon"
                            style={{
                              width: "20px",
                              height: "20px",
                              backgroundColor: "#ef4444",
                              borderRadius: "4px",
                              display: "flex",
                              alignItems: "center",
                              justifyContent: "center",
                              fontSize: "12px",
                            }}
                          >
                            📄
                          </div>
                          <span
                            className="title-name"
                            style={{ fontWeight: "500", color: "#2d3748" }}
                          >
                            {file.name}
                          </span>
                        </div>
                        <div
                          className="description"
                          style={{ fontSize: "14px", color: "#718096" }}
                        >
                          <span className="size">
                            {formatFileSize(file.size)}
                          </span>
                          <span
                            className="separator"
                            style={{ margin: "0 8px" }}
                          >
                            -
                          </span>
                          <span className="extension">
                            {getFileExtension(file.name)}
                          </span>
                        </div>
                      </div>
                      <button
                        className="files-list-item-icon-delete"
                        onClick={() => removeFile(file.id)}
                        style={{
                          width: "24px",
                          height: "24px",
                          borderRadius: "50%",
                          backgroundColor: "#f1f3f4",
                          display: "flex",
                          alignItems: "center",
                          justifyContent: "center",
                          cursor: "pointer",
                          border: "none",
                          fontSize: "18px",
                          color: "#718096",
                          transition: "background-color 0.2s",
                        }}
                        onMouseEnter={(e) =>
                          (e.currentTarget.style.backgroundColor = "#e2e8f0")
                        }
                        onMouseLeave={(e) =>
                          (e.currentTarget.style.backgroundColor = "#f1f3f4")
                        }
                      >
                        ×
                      </button>
                    </li>
                  ))}
                </ul>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div
        className="upload-parameters-box"
        style={{
          flex: "1",
          minWidth: "350px",
          background: "white",
          borderRadius: "12px",
          boxShadow: "0 2px 12px rgba(0, 0, 0, 0.08)",
          overflow: "hidden",
        }}
      >
        <div
          className="upload-parameters-box-header"
          style={{ borderBottom: "1px solid #e9ecef" }}
        >
          <div className="tabs" style={{ display: "flex" }}>
            <button
              className={`tab ${tab === "email" ? "tab-selected" : ""}`}
              onClick={() => setTab("email")}
              style={{
                flex: "1",
                padding: "16px 20px",
                background: "none",
                border: "none",
                cursor: "pointer",
                fontSize: "16px",
                fontWeight: "500",
                color: tab === "email" ? "#2d3748" : "#718096",
                backgroundColor: tab === "email" ? "white" : "transparent",
                position: "relative",
                transition: "all 0.2s",
                borderBottom: tab === "email" ? "2px solid #ef4444" : "none",
              }}
            >
              Email
            </button>
            <button
              className={`tab ${tab === "link" ? "tab-selected" : ""}`}
              onClick={() => setTab("link")}
              style={{
                flex: "1",
                padding: "16px 20px",
                background: "none",
                border: "none",
                cursor: "pointer",
                fontSize: "16px",
                fontWeight: "500",
                color: tab === "link" ? "#2d3748" : "#718096",
                backgroundColor: tab === "link" ? "white" : "transparent",
                position: "relative",
                transition: "all 0.2s",
                borderBottom: tab === "link" ? "2px solid #ef4444" : "none",
              }}
            >
              Link
            </button>
          </div>
        </div>

        <div
          className="upload-parameters-box-content"
          style={{ padding: "24px" }}
        >
          {tab === "email" ? (
            <form
              className="email-form-container"
              onSubmit={handleEmailSubmit}
              style={{
                display: "flex",
                flexDirection: "column",
                gap: "16px",
              }}
            >
              <div className="input-container">
                <input
                  type="email"
                  name="email"
                  placeholder="Your email"
                  required
                  style={{
                    width: "100%",
                    padding: "12px 16px",
                    border: "1px solid #e2e8f0",
                    borderRadius: "8px",
                    fontSize: "14px",
                    backgroundColor: "#fafbfc",
                    transition: "border-color 0.2s, box-shadow 0.2s",
                  }}
                  onFocus={(e) => {
                    e.target.style.borderColor = "#ef4444";
                    e.target.style.boxShadow =
                      "0 0 0 3px rgba(239, 68, 68, 0.1)";
                    e.target.style.backgroundColor = "white";
                  }}
                  onBlur={(e) => {
                    e.target.style.borderColor = "#e2e8f0";
                    e.target.style.boxShadow = "none";
                    e.target.style.backgroundColor = "#fafbfc";
                  }}
                />
              </div>
              <div className="input-container">
                <input
                  type="text"
                  name="contacts"
                  placeholder="Your contacts' emails"
                  required
                  style={{
                    width: "100%",
                    padding: "12px 16px",
                    border: "1px solid #e2e8f0",
                    borderRadius: "8px",
                    fontSize: "14px",
                    backgroundColor: "#fafbfc",
                    transition: "border-color 0.2s, box-shadow 0.2s",
                  }}
                  onFocus={(e) => {
                    e.target.style.borderColor = "#ef4444";
                    e.target.style.boxShadow =
                      "0 0 0 3px rgba(239, 68, 68, 0.1)";
                    e.target.style.backgroundColor = "white";
                  }}
                  onBlur={(e) => {
                    e.target.style.borderColor = "#e2e8f0";
                    e.target.style.boxShadow = "none";
                    e.target.style.backgroundColor = "#fafbfc";
                  }}
                />
              </div>
              <div className="input-container">
                <input
                  type="text"
                  name="title"
                  placeholder="Subject"
                  style={{
                    width: "100%",
                    padding: "12px 16px",
                    border: "1px solid #e2e8f0",
                    borderRadius: "8px",
                    fontSize: "14px",
                    backgroundColor: "#fafbfc",
                    transition: "border-color 0.2s, box-shadow 0.2s",
                  }}
                  onFocus={(e) => {
                    e.target.style.borderColor = "#ef4444";
                    e.target.style.boxShadow =
                      "0 0 0 3px rgba(239, 68, 68, 0.1)";
                    e.target.style.backgroundColor = "white";
                  }}
                  onBlur={(e) => {
                    e.target.style.borderColor = "#e2e8f0";
                    e.target.style.boxShadow = "none";
                    e.target.style.backgroundColor = "#fafbfc";
                  }}
                />
              </div>
              <div className="input-container">
                <textarea
                  placeholder="Message"
                  name="message"
                  style={{
                    width: "100%",
                    padding: "12px 16px",
                    border: "1px solid #e2e8f0",
                    borderRadius: "8px",
                    fontSize: "14px",
                    backgroundColor: "#fafbfc",
                    minHeight: "80px",
                    resize: "vertical",
                    fontFamily: "inherit",
                    transition: "border-color 0.2s, box-shadow 0.2s",
                  }}
                  onFocus={(e) => {
                    e.target.style.borderColor = "#ef4444";
                    e.target.style.boxShadow =
                      "0 0 0 3px rgba(239, 68, 68, 0.1)";
                    e.target.style.backgroundColor = "white";
                  }}
                  onBlur={(e) => {
                    e.target.style.borderColor = "#e2e8f0";
                    e.target.style.boxShadow = "none";
                    e.target.style.backgroundColor = "#fafbfc";
                  }}
                />
              </div>
            </form>
          ) : (
            <form
              className="link-form-container"
              onSubmit={handleLinkSubmit}
              style={{
                display: "flex",
                flexDirection: "column",
                gap: "16px",
              }}
            >
              <div className="input-container">
                <input
                  type="text"
                  name="title"
                  placeholder="Title"
                  style={{
                    width: "100%",
                    padding: "12px 16px",
                    border: "1px solid #e2e8f0",
                    borderRadius: "8px",
                    fontSize: "14px",
                    backgroundColor: "#fafbfc",
                    transition: "border-color 0.2s, box-shadow 0.2s",
                  }}
                  onFocus={(e) => {
                    e.target.style.borderColor = "#ef4444";
                    e.target.style.boxShadow =
                      "0 0 0 3px rgba(239, 68, 68, 0.1)";
                    e.target.style.backgroundColor = "white";
                  }}
                  onBlur={(e) => {
                    e.target.style.borderColor = "#e2e8f0";
                    e.target.style.boxShadow = "none";
                    e.target.style.backgroundColor = "#fafbfc";
                  }}
                />
              </div>
              <div className="input-container" style={{ position: "relative" }}>
                <div
                  className="input-prefix"
                  style={{
                    position: "absolute",
                    left: "12px",
                    top: "50%",
                    transform: "translateY(-50%)",
                    color: "#718096",
                    fontSize: "14px",
                    pointerEvents: "none",
                    zIndex: 1,
                  }}
                >
                  wurknest.com/
                </div>
                <input
                  type="text"
                  name="customUrl"
                  placeholder=""
                  style={{
                    width: "100%",
                    padding: "12px 16px",
                    paddingLeft: "120px",
                    border: "1px solid #e2e8f0",
                    borderRadius: "8px",
                    fontSize: "14px",
                    backgroundColor: "#fafbfc",
                    transition: "border-color 0.2s, box-shadow 0.2s",
                  }}
                  onFocus={(e) => {
                    e.target.style.borderColor = "#ef4444";
                    e.target.style.boxShadow =
                      "0 0 0 3px rgba(239, 68, 68, 0.1)";
                    e.target.style.backgroundColor = "white";
                  }}
                  onBlur={(e) => {
                    e.target.style.borderColor = "#e2e8f0";
                    e.target.style.boxShadow = "none";
                    e.target.style.backgroundColor = "#fafbfc";
                  }}
                />
              </div>
              <div className="input-container">
                <input
                  type="email"
                  name="email"
                  placeholder="Your email"
                  style={{
                    width: "100%",
                    padding: "12px 16px",
                    border: "1px solid #e2e8f0",
                    borderRadius: "8px",
                    fontSize: "14px",
                    backgroundColor: "#fafbfc",
                    transition: "border-color 0.2s, box-shadow 0.2s",
                  }}
                  onFocus={(e) => {
                    e.target.style.borderColor = "#ef4444";
                    e.target.style.boxShadow =
                      "0 0 0 3px rgba(239, 68, 68, 0.1)";
                    e.target.style.backgroundColor = "white";
                  }}
                  onBlur={(e) => {
                    e.target.style.borderColor = "#e2e8f0";
                    e.target.style.boxShadow = "none";
                    e.target.style.backgroundColor = "#fafbfc";
                  }}
                />
              </div>
              <div
                className="input-container checkbox"
                style={{
                  display: "flex",
                  alignItems: "center",
                  gap: "8px",
                }}
              >
                <label
                  style={{
                    display: "flex",
                    alignItems: "center",
                    gap: "8px",
                    cursor: "pointer",
                    fontSize: "14px",
                    color: "#2d3748",
                  }}
                >
                  <input type="checkbox" name="accessTracking" />
                  Require email to download files
                </label>
              </div>
            </form>
          )}
        </div>

        <div
          className="upload-parameters-box-footer"
          style={{
            padding: "16px 24px",
            backgroundColor: "#fafbfc",
            borderTop: "1px solid #e9ecef",
            display: "flex",
            justifyContent: "space-between",
            alignItems: "center",
          }}
        >
          <div
            className="upload-parameters-box-footer-left"
            style={{
              display: "flex",
              alignItems: "center",
              gap: "12px",
            }}
          >
            <button
              className="icon-settings"
              onClick={() => setShowSettingsModal(true)}
              style={{
                width: "24px",
                height: "24px",
                backgroundColor: "#718096",
                borderRadius: "4px",
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
                fontSize: "14px",
                border: "none",
                cursor: "pointer",
                transition: "background-color 0.2s",
              }}
              onMouseEnter={(e) =>
                (e.currentTarget.style.backgroundColor = "#5a6c7d")
              }
              onMouseLeave={(e) =>
                (e.currentTarget.style.backgroundColor = "#718096")
              }
            >
              ⚙️
            </button>
            <div>
              <div
                className="info-expiration-date"
                style={{
                  fontSize: "14px",
                  fontWeight: "500",
                  color: "#2d3748",
                }}
              >
                Expires on{" "}
                {expirationDate.toLocaleDateString("en-US", {
                  month: "short",
                  day: "numeric",
                  year: "numeric",
                })}
              </div>
              <div
                className="info-password"
                style={{
                  fontSize: "12px",
                  color: "#718096",
                }}
              >
                {settings.password ? "Password protected" : "No password added"}
              </div>
            </div>
          </div>
          <div
            className="upload-parameters-box-footer-right"
            style={{
              display: "flex",
              alignItems: "center",
              gap: "12px",
            }}
          >
            <button
              type="button"
              style={{
                padding: "8px 16px",
                backgroundColor: "#3b82f6",
                color: "white",
                border: "none",
                borderRadius: "6px",
                fontSize: "14px",
                fontWeight: "500",
                cursor: "pointer",
                transition: "background-color 0.2s",
              }}
              onMouseEnter={(e) =>
                (e.currentTarget.style.backgroundColor = "#286efb")
              }
              onMouseLeave={(e) =>
                (e.currentTarget.style.backgroundColor = "#3b82f6")
              }
            >
              {tab === "email" ? "Send" : "Get a link"}
            </button>
          </div>
        </div>
      </div>

      {showSettingsModal && (
        <div
          style={{
            position: "fixed",
            top: 0,
            left: 0,
            right: 0,
            bottom: 0,
            backgroundColor: "rgba(0, 0, 0, 0.5)",
            display: "flex",
            alignItems: "center",
            justifyContent: "center",
            zIndex: 1000,
          }}
        >
          <div
            style={{
              backgroundColor: "white",
              borderRadius: "12px",
              width: "400px",
              maxHeight: "80vh",
              overflow: "auto",
              boxShadow: "0 4px 20px rgba(0, 0, 0, 0.15)",
            }}
          >
            <div
              style={{
                padding: "20px 24px 16px",
                borderBottom: "1px solid #e5e7eb",
              }}
            >
              <h3
                style={{
                  margin: 0,
                  fontSize: "18px",
                  fontWeight: "600",
                  color: "#1f2937",
                }}
              >
                Transfer settings
              </h3>
            </div>

            <div style={{ padding: "20px 24px" }}>
              <div
                style={{
                  display: "flex",
                  justifyContent: "space-between",
                  alignItems: "center",
                  marginBottom: "20px",
                }}
              >
                <span
                  style={{
                    fontSize: "14px",
                    color: "#374151",
                    fontWeight: "500",
                  }}
                >
                  Expiration
                </span>
                <div
                  style={{
                    display: "flex",
                    alignItems: "center",
                    gap: "8px",
                  }}
                >
                  <input
                    type="number"
                    value={settings.expiration}
                    onChange={(e) =>
                      setSettings({
                        ...settings,
                        expiration: parseInt(e.target.value) || 7,
                      })
                    }
                    style={{
                      width: "60px",
                      padding: "6px 8px",
                      border: "1px solid #d1d5db",
                      borderRadius: "6px",
                      fontSize: "14px",
                      textAlign: "center",
                    }}
                  />
                  <span
                    style={{
                      fontSize: "14px",
                      color: "#6b7280",
                    }}
                  >
                    days
                  </span>
                </div>
              </div>

              <div
                style={{
                  display: "flex",
                  justifyContent: "space-between",
                  alignItems: "center",
                  marginBottom: "20px",
                }}
              >
                <span
                  style={{
                    fontSize: "14px",
                    color: "#374151",
                    fontWeight: "500",
                  }}
                >
                  Password
                </span>
                <input
                  type="password"
                  placeholder="Add password"
                  value={settings.password}
                  onChange={(e) =>
                    setSettings({ ...settings, password: e.target.value })
                  }
                  style={{
                    width: "140px",
                    padding: "6px 12px",
                    border: "1px solid #d1d5db",
                    borderRadius: "6px",
                    fontSize: "14px",
                    color: "#9ca3af",
                  }}
                />
              </div>

              <div style={{ marginBottom: "16px" }}>
                <label
                  style={{
                    display: "flex",
                    alignItems: "center",
                    justifyContent: "space-between",
                    marginBottom: "12px",
                    cursor: "pointer",
                  }}
                >
                  <span
                    style={{
                      fontSize: "14px",
                      color: "#374151",
                    }}
                  >
                    Be notified of the creation of the transfer
                  </span>
                  <input
                    type="checkbox"
                    checked={settings.notifyCreation}
                    onChange={(e) =>
                      setSettings({
                        ...settings,
                        notifyCreation: e.target.checked,
                      })
                    }
                    style={{
                      width: "16px",
                      height: "16px",
                      accentColor: "#3b82f6",
                    }}
                  />
                </label>

                <label
                  style={{
                    display: "flex",
                    alignItems: "center",
                    justifyContent: "space-between",
                    marginBottom: "12px",
                    cursor: "pointer",
                  }}
                >
                  <span
                    style={{
                      fontSize: "14px",
                      color: "#374151",
                    }}
                  >
                    Be notified of downloads
                  </span>
                  <input
                    type="checkbox"
                    checked={settings.notifyDownloads}
                    onChange={(e) =>
                      setSettings({
                        ...settings,
                        notifyDownloads: e.target.checked,
                      })
                    }
                    style={{
                      width: "16px",
                      height: "16px",
                      accentColor: "#3b82f6",
                    }}
                  />
                </label>

                <label
                  style={{
                    display: "flex",
                    alignItems: "center",
                    justifyContent: "space-between",
                    marginBottom: "12px",
                    cursor: "pointer",
                  }}
                >
                  <span
                    style={{
                      fontSize: "14px",
                      color: "#374151",
                    }}
                  >
                    Be notified of non-download
                  </span>
                  <input
                    type="checkbox"
                    checked={settings.notifyNonDownload}
                    onChange={(e) =>
                      setSettings({
                        ...settings,
                        notifyNonDownload: e.target.checked,
                      })
                    }
                    style={{
                      width: "16px",
                      height: "16px",
                      accentColor: "#3b82f6",
                    }}
                  />
                </label>

                <label
                  style={{
                    display: "flex",
                    alignItems: "center",
                    justifyContent: "space-between",
                    marginBottom: "12px",
                    cursor: "pointer",
                  }}
                >
                  <span
                    style={{
                      fontSize: "14px",
                      color: "#374151",
                    }}
                  >
                    Preview files
                  </span>
                  <input
                    type="checkbox"
                    checked={settings.previewFiles}
                    onChange={(e) =>
                      setSettings({
                        ...settings,
                        previewFiles: e.target.checked,
                      })
                    }
                    style={{
                      width: "16px",
                      height: "16px",
                      accentColor: "#3b82f6",
                    }}
                  />
                </label>
              </div>

              <div
                style={{
                  display: "flex",
                  justifyContent: "space-between",
                  alignItems: "center",
                  marginBottom: "20px",
                  opacity: 0.5,
                }}
              >
                <span
                  style={{
                    fontSize: "14px",
                    color: "#6b7280",
                  }}
                >
                  Customise the design
                </span>
                <span
                  style={{
                    fontSize: "16px",
                    color: "#6b7280",
                  }}
                >
                  🔒
                </span>
              </div>
            </div>

            <div
              style={{
                padding: "16px 24px",
                borderTop: "1px solid #e5e7eb",
                display: "flex",
                justifyContent: "space-between",
                alignItems: "center",
              }}
            >
              <button
                onClick={() => setShowSettingsModal(false)}
                style={{
                  background: "none",
                  border: "none",
                  fontSize: "18px",
                  cursor: "pointer",
                  color: "#6b7280",
                  padding: "8px",
                }}
              >
                ✕
              </button>
              <button
                onClick={() => setShowSettingsModal(false)}
                style={{
                  backgroundColor: "#3b82f6",
                  color: "white",
                  border: "none",
                  borderRadius: "8px",
                  padding: "10px 24px",
                  fontSize: "14px",
                  fontWeight: "500",
                  cursor: "pointer",
                  transition: "background-color 0.2s",
                }}
                onMouseEnter={(e) =>
                  (e.currentTarget.style.backgroundColor = "#286efb")
                }
                onMouseLeave={(e) =>
                  (e.currentTarget.style.backgroundColor = "#3b82f6")
                }
              >
                Done
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
