'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { Search, FileText, FileSpreadsheet, Image as ImageIcon, File } from 'lucide-react'

interface SearchResult {
  id: number
  name: string
  type: 'document' | 'spreadsheet' | 'image' | 'file'
  size: string
  modified: string
  icon: React.ReactNode
}

const searchQueries = [
  'Q4 Report',
  'Contract',
  'Invoice'
]

const allResults: SearchResult[] = [
  {
    id: 1,
    name: 'Q4_Financial_Report.pdf',
    type: 'document',
    size: '2.4 MB',
    modified: '2 days ago',
    icon: <FileText className="w-4 h-4 md:w-5 md:h-5" />
  },
  {
    id: 2,
    name: 'Q4_Sales_Data.xlsx',
    type: 'spreadsheet',
    size: '856 KB',
    modified: '5 days ago',
    icon: <FileSpreadsheet className="w-4 h-4 md:w-5 md:h-5" />
  },
  {
    id: 3,
    name: 'Q4_Presentation.pptx',
    type: 'document',
    size: '5.2 MB',
    modified: '1 week ago',
    icon: <FileText className="w-4 h-4 md:w-5 md:h-5" />
  }
]

export default function FileSearchAnimationPanel() {
  const [searchText, setSearchText] = useState('')
  const [currentQueryIndex, setCurrentQueryIndex] = useState(0)
  const [results, setResults] = useState<SearchResult[]>([])
  const [isSearching, setIsSearching] = useState(false)
  const [isVisible, setIsVisible] = useState(false)
  const panelRef = useRef<HTMLDivElement>(null)
  const typingRef = useRef(0)

  // Intersection observer to detect when component is visible
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  // Animation sequence
  useEffect(() => {
    if (!isVisible) return

    const animateSearch = async () => {
      const currentQuery = searchQueries[currentQueryIndex]
      
      // Reset state
      setSearchText('')
      setResults([])
      setIsSearching(false)
      
      await new Promise(resolve => setTimeout(resolve, 800))
      
      // Type the search query
      for (let i = 0; i <= currentQuery.length; i++) {
        setSearchText(currentQuery.slice(0, i))
        await new Promise(resolve => setTimeout(resolve, 100))
      }
      
      // Show searching state
      setIsSearching(true)
      await new Promise(resolve => setTimeout(resolve, 600))
      
      // Show results one by one
      setIsSearching(false)
      for (let i = 0; i < allResults.length; i++) {
        setResults(prev => [...prev, allResults[i]])
        await new Promise(resolve => setTimeout(resolve, 200))
      }
      
      // Hold results
      await new Promise(resolve => setTimeout(resolve, 2000))
      
      // Move to next query
      setCurrentQueryIndex((prev) => (prev + 1) % searchQueries.length)
    }

    animateSearch()
  }, [currentQueryIndex, isVisible])

  return (
    <div 
      ref={panelRef}
      className={`w-[310px] md:w-[450px] mx-auto mb-8 md:mb-0 transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Search Interface */}
      <div className="w-full bg-white dark:bg-[#1a1a1a] backdrop-blur-md rounded-xl md:rounded-2xl shadow-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
        {/* Header */}
        <div className="bg-gray-50 dark:bg-[#1a1a1a] px-3 md:px-5 py-2.5 md:py-4 border-b border-gray-200 dark:border-gray-700">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-2 md:gap-3">
              <h3 className="text-gray-700 dark:text-white font-semibold text-xs md:text-base">File Search</h3>
            </div>
            
            {/* Results count with smooth transition */}
            <div className="flex items-center gap-1.5 md:gap-2 min-w-[80px] md:min-w-[100px] justify-end">
              <AnimatePresence mode="wait">
                <motion.span
                  key={results.length > 0 ? 'found' : 'ai'}
                  initial={{ opacity: 0, y: -10 }}
                  animate={{ opacity: 1, y: 0 }}
                  exit={{ opacity: 0, y: 10 }}
                  transition={{ duration: 0.3 }}
                  className="text-gray-600 dark:text-gray-300 text-[10px] md:text-sm"
                >
                  {results.length > 0 ? `${results.length} found` : 'AI-powered'}
                </motion.span>
              </AnimatePresence>
            </div>
          </div>
        </div>

        {/* Search Bar */}
        <div className="p-3 md:p-5 border-b border-gray-200 dark:border-gray-700">
          <div className="relative">
            <Search className="absolute left-2.5 md:left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 md:w-5 md:h-5 text-gray-500 dark:text-gray-400" />
            <div className="bg-gray-100 dark:bg-[#0a0a0a] border border-gray-200 dark:border-gray-700 rounded-lg md:rounded-xl pl-8 md:pl-10 pr-3 py-2 md:py-2.5 text-gray-800 dark:text-white text-xs md:text-sm min-h-[32px] md:min-h-[40px] flex items-center">
              {searchText ? (
                <span className="inline-flex items-center">
                  {searchText}
                  {searchText.length < searchQueries[currentQueryIndex].length && (
                    <motion.span
                      animate={{ opacity: [1, 0] }}
                      transition={{ duration: 0.5, repeat: Infinity }}
                      className="inline-block w-0.5 h-3 md:h-4 bg-gray-800 dark:bg-white ml-0.5"
                    />
                  )}
                </span>
              ) : (
                <span className="text-gray-500 dark:text-gray-400">Search files...</span>
              )}
            </div>
          </div>
        </div>

        {/* Results Container */}
        <div className="p-3 md:p-5 space-y-2 md:space-y-2.5 h-[200px] md:h-[280px] overflow-hidden">
          {/* Searching state */}
          {isSearching && (
            <motion.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              className="flex items-center justify-center h-full"
            >
              <div className="flex flex-col items-center gap-2">
                <div className="flex gap-1">
                  {[0, 1, 2].map((i) => (
                    <motion.div
                      key={i}
                      className="w-2 h-2 bg-gray-500 dark:bg-gray-400 rounded-full"
                      animate={{ 
                        scale: [1, 1.5, 1],
                        opacity: [0.5, 1, 0.5]
                      }}
                      transition={{
                        duration: 0.8,
                        repeat: Infinity,
                        delay: i * 0.2
                      }}
                    />
                  ))}
                </div>
                <p className="text-gray-500 dark:text-gray-400 text-xs md:text-sm">Searching...</p>
              </div>
            </motion.div>
          )}

          {/* Search Results */}
          {!isSearching && (
            <AnimatePresence mode="popLayout">
              {results.map((result, index) => (
                <motion.div
                  key={result.id}
                  initial={{ y: 20, opacity: 0, scale: 0.95 }}
                  animate={{ y: 0, opacity: 1, scale: 1 }}
                  exit={{ opacity: 0, scale: 0.95 }}
                  transition={{
                    type: 'spring',
                    stiffness: 400,
                    damping: 25
                  }}
                  className="bg-gray-100 dark:bg-transparent border border-gray-200 dark:border-gray-700 rounded-lg md:rounded-xl p-2.5 md:p-3 hover:bg-gray-200 dark:hover:bg-white/5 transition-colors cursor-pointer"
                >
                  <div className="flex items-center gap-2.5 md:gap-3">
                    {/* File Icon */}
                    <div className="w-8 h-8 md:w-10 md:h-10 rounded-lg bg-gray-200 dark:bg-[#1a1a1a] flex items-center justify-center flex-shrink-0 text-gray-700 dark:text-gray-300">
                      {result.icon}
                    </div>

                    {/* File Info */}
                    <div className="flex-1 min-w-0">
                      <p className="text-gray-700 dark:text-white font-medium text-[10px] md:text-sm truncate">
                        {result.name}
                      </p>
                      <div className="flex items-center gap-2 mt-0.5">
                        <span className="text-gray-500 dark:text-gray-400 text-[8px] md:text-xs">
                          {result.size}
                        </span>
                        <span className="text-gray-300 dark:text-gray-600">•</span>
                        <span className="text-gray-500 dark:text-gray-400 text-[8px] md:text-xs">
                          {result.modified}
                        </span>
                      </div>
                    </div>

                    {/* Match indicator */}
                    <motion.div
                      initial={{ scale: 0 }}
                      animate={{ scale: 1 }}
                      transition={{ delay: 0.1, type: 'spring' }}
                      className="w-1.5 h-1.5 md:w-2 md:h-2 bg-green-500 rounded-full flex-shrink-0"
                    />
                  </div>
                </motion.div>
              ))}
            </AnimatePresence>
          )}

          {/* Empty state */}
          {!isSearching && results.length === 0 && searchText === '' && (
            <div className="flex items-center justify-center h-full text-gray-400 dark:text-gray-600 text-xs md:text-sm">
              Type to search files...
            </div>
          )}
        </div>
      </div>
    </div>
  )
}
