import React from "react";
import Image from "next/image";
import { ArrowRightIcon } from "@heroicons/react/24/outline";
import { FaStar } from "react-icons/fa";

interface Feature {
  icon: React.ReactNode;
  title: string;
  description: string;
}

import type { StaticImageData } from "next/image";

interface Quote {
  text: string;
  userImg: string | StaticImageData;
  userName: string;
  userRole: string;
}

interface FeatureSectionProps {
  title: string;
  icon?: React.ReactNode;
  showIcon?: boolean;
  description: string;
  features: Feature[];
  cta1: {
    label: string;
    onClick: () => void;
    className?: string;
  };
  cta2: {
    label: string;
    onClick: () => void;
    className?: string;
    icon?: React.ReactNode;
    mobileLabel?: string;
  };
  quote?: Quote;
  theme?: string;
}

const FeatureSection: React.FC<FeatureSectionProps> = ({
  title,
  icon,
  showIcon = true,
  description,
  features,
  cta1,
  cta2,
  quote,
  theme,
}) => (
  <div className="container max-w-[1233px] max-[1300px]:px-14 max-[1100px]:px-0 mx-auto px-0">
    
      <div className="space-y-8 justify-self-start">
        <div className="space-y-4">
          <span className="hidden md:block text-[#253237] dark:text-white text-[14px] font-bold">
            Feature
          </span>
          <h2 className="text-[23px]  flex items-center gap-2  md:text-5xl text-gray-900 dark:text-white  font-inter font-[800] leading-[29px] md:leading-[1.2] mt-4 md:mt-0">
            {showIcon && icon && (
              <div className="w-6 h-6 md:w-9 md:h-9 rounded-md md:rounded-lg bg-[#253237] dark:bg-white flex items-center justify-center">
                <div className="w-6 h-6 md:w-9 md:h-9 rounded-md md:rounded-lg bg-[#253237] dark:bg-white flex items-center justify-center
                [&>svg]:w-4.5 [&>svg]:h-4.5 md:[&>svg]:w-5 md:[&>svg]:h-5 [&>svg]:text-white dark:[&>svg]:text-gray-900">
                {icon}
                </div>
              </div>
            )}
            <span>{title}</span>
          </h2>
          <p className="text-gray-600 dark:text-[#d7d7de] max-w-lg font-inter font-normal text-[16px] md:text-[18px]">
            {description}
          </p>
        </div>
        <div className="grid grid-cols-2 gap-2 md:gap-4 max-w-[600px] mb-4">
          {features.map((feature, index) => (
            <div
              key={index}
              className="group relative bg-white dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E] rounded-lg md:rounded-xl p-2 md:p-5 hover:border-gray-300 dark:hover:border-[#26282E] transition-all duration-300 hover:shadow-lg"
            >
              <div className="flex flex-col md:flex-row items-start gap-1.5 md:gap-3">
                <div className="flex-shrink-0 w-8 h-8 md:w-10 md:h-10 rounded-md md:rounded-xl bg-gray-100 dark:bg-[#16171B] flex items-center justify-center [&>svg]:w-4 [&>svg]:h-4 md:[&>svg]:w-5 md:[&>svg]:h-5">
                  {feature.icon}
                </div>
                <div className="flex-1">
                  <h3 className="text-gray-900 dark:text-white text-[14px] md:text-base font-bold mb-0.5 md:mb-1 leading-tight">
                    {feature.title}
                  </h3>
                  <p className="text-gray-600 dark:text-gray-400 text-[12px] md:text-sm leading-tight md:leading-relaxed">
                    {feature.description}
                  </p>
                </div>
              </div>
            </div>
          ))}
        </div>
        <div className="flex flex-col gap-3">
          <div className="grid grid-cols-2 gap-2 md:gap-3 max-w-[600px]">
            <button
              onClick={cta1.onClick}
              className={
                cta1.className ||
                "px-2 py-2 md:px-6 md:py-3 bg-white dark:bg-white/5 text-gray-900 dark:text-white text-[14px] md:text-base font-medium rounded-lg md:rounded-xl border-1 border-gray-200 dark:border-[#26282E] hover:border-gray-300 dark:hover:border-gray-200 transition-all duration-300 dark:shadow-[0_0_20px_rgba(255,255,255,0.05)] dark:hover:shadow-[0_0_30px_rgba(255,255,255,0.075)]"
              }
            >
              {cta1.label}
            </button>
            <button
              onClick={cta2.onClick}
              className={
                cta2.className ||
                "px-2 py-2 md:px-6 md:py-3 bg-[#253237] dark:bg-black hover:bg-black dark:hover:bg-gray-900 text-white text-[14px] md:text-base font-medium rounded-lg md:rounded-xl transition-all duration-300 hover:shadow-lg hover:scale-105 flex items-center justify-center gap-1 md:gap-2 dark:shadow-[0_0_20px_rgba(255,255,255,0.1)] dark:hover:shadow-[0_0_30px_rgba(255,255,255,0.15)]"
              }
            >
              <span className="hidden md:inline">{cta2.label}</span>
              <span className="md:hidden">{cta2.mobileLabel || cta2.label}</span>
              <ArrowRightIcon className="w-3.5 h-3.5 md:w-4 md:h-4" />
              {cta2.icon}
            </button>
          </div>

          {quote && (
            <div className="hidden md:flex flex-col items-end pl-4 rounded-lg text-gray-600 dark:text-[#94A3B8] font-light mt-2">
              <p className="text-sm leading-relaxed font-inter text-right">
                <span className="text-[#ff4b5c]">&rdquo;</span>
                {quote.text}
                <span className="text-[#ff4b5c]">&rdquo;</span>
              </p>
              <div className="flex items-center gap-3 mt-2">
                <div className="text-right">
                  <p className="font-semibold text-sm text-gray-900 dark:text-white">{quote.userName}</p>
                  <p className="text-xs text-gray-500 dark:text-gray-400">{quote.userRole}</p>
                </div>
                <Image
                  src={quote.userImg}
                  alt={quote.userName}
                  width={24}
                  height={24}
                  className="w-6 h-6 rounded-full"
                />
              </div>
            </div>
          )}
        </div>
      </div>
  </div>
);

export default FeatureSection;
