'use client'

import { useState, useEffect, useRef } from 'react'
import { Plus, Minus } from 'lucide-react'

interface FAQ {
  question: string
  answer: string
}

interface FAQSectionProps {
  faqs?: FAQ[]
  title?: string
  subtitle?: string
}

const defaultFaqs: FAQ[] = [
  {
    question: 'What features are included in the Starter plan?',
    answer: 'The Starter plan includes basic Campaigns features, Chat for up to 10 users, 5GB file storage, up to 3 seats, 10 monitors, and email support. Perfect for individuals and small projects getting started.'
  },
  {
    question: 'Can I upgrade or downgrade my plan at any time?',
    answer: 'Yes! You can upgrade or downgrade your plan anytime. When upgrading, you\'ll be charged the prorated difference. When downgrading, the changes will take effect at the start of your next billing cycle.'
  },
  {
    question: 'What payment methods do you accept?',
    answer: 'We accept all major credit cards (Visa, Mastercard, American Express), debit cards, and PayPal. All payments are processed securely through our encrypted payment gateway.'
  },
  {
    question: 'Is there a free trial available?',
    answer: 'Yes! All plans come with a 14-day free trial. No credit card required to start. You can explore all features and decide which plan works best for your team before committing.'
  },
  {
    question: 'What happens to my data if I cancel my subscription?',
    answer: 'Your data remains accessible for 30 days after cancellation. During this period, you can export all your files, campaigns, and documents. After 30 days, data is permanently deleted from our servers.'
  },
  {
    question: 'Do you offer discounts for annual subscriptions?',
    answer: 'Yes! When you choose annual billing, you save 17% compared to monthly payments. This applies to all plan tiers and you receive unlimited credits when paying yearly.'
  },
  {
    question: 'Can I add more team members to my plan?',
    answer: 'Absolutely! You can add team members at any time. The Starter plan supports up to 3 seats, Professional up to 8 seats, and Enterprise up to 50 team members. Additional seats can be purchased if needed.'
  },
  {
    question: 'What kind of support do you provide?',
    answer: 'Support varies by plan: Starter includes email support with 24-hour response time, Professional gets priority support with 12-hour response, and Enterprise receives dedicated support with a 4-hour response time and a dedicated account manager.'
  }
]

export default function FAQSection({ faqs = defaultFaqs, title = "Frequently Asked Questions", subtitle = "Our friendly team would love to answer your questions." }: FAQSectionProps) {
  const [openIndex, setOpenIndex] = useState<number | null>(0)
  const [isVisible, setIsVisible] = useState(false)
  const sectionRef = useRef<HTMLElement>(null)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const toggleFAQ = (index: number) => {
    setOpenIndex(openIndex === index ? null : index)
  }

  return (
    <section 
      ref={sectionRef}
      className="relative bg-white dark:bg-transparent px-6 md:px-12 lg:px-20 py-20 md:py-28 transition-colors duration-300"
    >
      {/* Background decoration */}
      <div className="absolute inset-0 -z-10 overflow-hidden pointer-events-none">
        <div className="absolute top-0 left-1/4 w-96 h-96 bg-gray-100/50 dark:bg-transparent rounded-full blur-3xl" />
        <div className="absolute bottom-0 right-1/4 w-96 h-96 bg-gray-100/50 dark:bg-transparent rounded-full blur-3xl" />
      </div>

      <div className="max-w-[1233px] mx-auto">
        {/* Header */}
        <div 
          className={`text-center mb-16 transition-all duration-1000 ${
            isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
          }`}
        >
          <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-gray-100 dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E] mb-6">
            <span className="text-sm font-semibold text-gray-900 dark:text-white tracking-wider uppercase">
              Ask Us Anything
            </span>
          </div>
          <h2 className="text-4xl md:text-5xl lg:text-6xl font-bold mb-6 text-gray-900 dark:text-white">
            {title}
          </h2>
          <p className="text-lg text-gray-600 dark:text-[#94A3B8] max-w-2xl mx-auto">
            {subtitle}
          </p>
        </div>

        {/* FAQ Items */}
        <div className="space-y-4">
          {faqs.map((faq, index) => (
            <div
              key={index}
              className={`transition-all duration-700 ${
                isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
              }`}
              style={{ transitionDelay: `${index * 50}ms` }}
            >
              <div
                className={`group rounded-2xl border-2 transition-all duration-300 overflow-hidden ${
                  openIndex === index
                    ? 'bg-[#253237] dark:bg-white border-[#253237] dark:border-white shadow-xl'
                    : 'bg-white dark:bg-[#16171B] border-gray-200 dark:border-[#26282E] hover:border-gray-300 dark:hover:border-[#3a3b42] shadow-sm hover:shadow-lg hover:-translate-y-1'
                }`}
              >
                <button
                  onClick={() => toggleFAQ(index)}
                  className="w-full px-6 md:px-8 py-6 flex items-center justify-between text-left gap-4"
                >
                  <span
                    className={`text-base md:text-lg font-bold transition-colors duration-300 ${
                      openIndex === index
                        ? 'text-white dark:text-[#253237]'
                        : 'text-gray-900 dark:text-white'
                    }`}
                  >
                    {faq.question}
                  </span>
                  <div
                    className={`flex-shrink-0 w-10 h-10 rounded-full flex items-center justify-center transition-all duration-300 ${
                      openIndex === index
                        ? 'bg-white/20 dark:bg-[#253237]/20 scale-110'
                        : 'bg-gray-100 dark:bg-[#0B0C0E] group-hover:bg-gray-200 dark:group-hover:bg-[#1a1d22]'
                    }`}
                  >
                    {openIndex === index ? (
                      <Minus className={`w-5 h-5 ${openIndex === index ? 'text-white dark:text-[#253237]' : 'text-gray-900 dark:text-white'}`} />
                    ) : (
                      <Plus className="w-5 h-5 text-gray-900 dark:text-white" />
                    )}
                  </div>
                </button>
                
                <div
                  className={`transition-all duration-300 ease-in-out ${
                    openIndex === index
                      ? 'max-h-96 opacity-100'
                      : 'max-h-0 opacity-0'
                  }`}
                >
                  <div className="px-6 md:px-8 pb-6">
                    <p
                      className={`text-sm md:text-base leading-relaxed ${
                        openIndex === index
                          ? 'text-white/90 dark:text-[#253237]/90'
                          : 'text-gray-600 dark:text-[#94A3B8]'
                      }`}
                    >
                      {faq.answer}
                    </p>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>

        {/* Contact CTA */}
        <div 
          className={`text-center mt-16 transition-all duration-1000 delay-500 ${
            isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
          }`}
        >
          <p className="text-gray-600 dark:text-[#94A3B8] mb-4">
            Still have questions? We&apos;re here to help!
          </p>
          <button className="bg-[#253237] text-white dark:bg-white dark:text-[#253237] px-8 py-3 rounded-xl font-semibold hover:bg-[#1a2329] dark:hover:bg-gray-100 transition-all duration-300 shadow-lg hover:shadow-xl hover:scale-105">
            Contact Support
          </button>
        </div>
      </div>
    </section>
  )
}
