'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { FileText, UserCheck, CheckCircle, Signature, Sparkles } from 'lucide-react'

type AnimationStage = 'idle' | 'template' | 'fields' | 'signing' | 'complete'

interface SignatureField {
  id: number
  position: { x: number; y: number }
  merged: boolean
}

const fieldPositions = [
  { x: -80, y: -60 },
  { x: 80, y: -60 },
  { x: 0, y: 60 }
]

export default function EsignWorkflowAnimationPanel() {
  const [stage, setStage] = useState<AnimationStage>('idle')
  const [fields, setFields] = useState<SignatureField[]>([])
  const [signatures, setSignatures] = useState<number[]>([])
  const [isVisible, setIsVisible] = useState(false)
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  useEffect(() => {
    if (!isVisible) return

    const runAnimation = async () => {
      // Reset
      setStage('idle')
      setFields([])
      setSignatures([])
      await new Promise(resolve => setTimeout(resolve, 800))

      // Create template (document appears)
      setStage('template')
      await new Promise(resolve => setTimeout(resolve, 1000))

      // Add signature fields flying in
      setStage('fields')
      for (let i = 0; i < fieldPositions.length; i++) {
        await new Promise(resolve => setTimeout(resolve, 600))
        setFields(prev => [...prev, {
          id: i,
          position: fieldPositions[i],
          merged: false
        }])
      }

      await new Promise(resolve => setTimeout(resolve, 800))

      // Merge fields to document
      for (let i = 0; i < fieldPositions.length; i++) {
        await new Promise(resolve => setTimeout(resolve, 400))
        setFields(prev => prev.map(f => 
          f.id === i ? { ...f, merged: true } : f
        ))
      }

      await new Promise(resolve => setTimeout(resolve, 600))

      // Signing stage - signatures appear
      setStage('signing')
      for (let i = 0; i < fieldPositions.length; i++) {
        await new Promise(resolve => setTimeout(resolve, 500))
        setSignatures(prev => [...prev, i])
      }

      await new Promise(resolve => setTimeout(resolve, 800))

      // Complete
      setStage('complete')
      await new Promise(resolve => setTimeout(resolve, 2000))

      // Restart
      runAnimation()
    }

    runAnimation()
  }, [isVisible])

  return (
    <div 
      ref={panelRef}
      className={`relative w-full max-w-2xl h-[400px] md:h-[550px] mx-auto flex items-center justify-center overflow-visible transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Sparkles for template creation */}
      <AnimatePresence>
        {stage === 'template' && (
          <>
            {[...Array(6)].map((_, i) => (
              <motion.div
                key={`sparkle-${i}`}
                className="absolute"
                initial={{ 
                  opacity: 0,
                  scale: 0,
                  x: 0,
                  y: 0
                }}
                animate={{ 
                  opacity: [0, 1, 0],
                  scale: [0, 1.5, 0],
                  x: Math.cos(i * 60 * Math.PI / 180) * 120,
                  y: Math.sin(i * 60 * Math.PI / 180) * 120
                }}
                exit={{ opacity: 0 }}
                transition={{ duration: 0.8, delay: i * 0.1 }}
              >
                <Sparkles className="w-6 h-6 md:w-8 md:h-8 text-[#286efb] dark:text-[#A9C5FF]" />
              </motion.div>
            ))}
          </>
        )}
      </AnimatePresence>

      {/* Complete status badge */}
      <AnimatePresence>
        {stage === 'complete' && (
          <motion.div
            className="absolute top-0 md:top-4 left-1/2 -translate-x-1/2 z-30"
            initial={{ opacity: 0, y: -30, scale: 0.8 }}
            animate={{ opacity: 1, y: 0, scale: 1 }}
            exit={{ opacity: 0, y: -30, scale: 0.8 }}
            transition={{ type: 'spring', stiffness: 300, damping: 20 }}
          >
            <div className="flex items-center gap-2 bg-green-100 dark:bg-green-900/30 border border-green-500/50 px-3 md:px-4 py-1.5 md:py-2 rounded-full">
              <CheckCircle className="w-4 h-4 md:w-5 md:h-5 text-green-600 dark:text-green-400" />
              <span className="text-xs md:text-sm font-semibold text-green-600 dark:text-green-400">
                Ready to Send
              </span>
            </div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Central Document */}
      <AnimatePresence mode="wait">
        {stage !== 'idle' && (
          <motion.div
            className="relative w-48 h-60 md:w-64 md:h-80 bg-white dark:bg-[#1a1a1a] rounded-lg md:rounded-xl border-2 border-gray-200 dark:border-gray-700 shadow-2xl z-10"
            initial={{ opacity: 0, scale: 0.5, rotateY: -90 }}
            animate={{ 
              opacity: 1, 
              scale: 1, 
              rotateY: 0,
              rotateZ: stage === 'complete' ? [0, -2, 2, 0] : 0
            }}
            exit={{ opacity: 0, scale: 0.5 }}
            transition={{ 
              type: 'spring', 
              stiffness: 300, 
              damping: 20,
              rotateZ: { duration: 0.5, delay: 0.5 }
            }}
            style={{
              filter: 'drop-shadow(0 10px 30px rgba(0, 0, 0, 0.2))'
            }}
          >
            {/* Document header */}
            <div className="p-2 md:p-3 border-b border-gray-200 dark:border-gray-700">
              <div className="flex items-center gap-2">
                <FileText className="w-4 h-4 md:w-5 md:h-5 text-gray-400" />
                <div className="flex-1 space-y-1">
                  <div className="h-1.5 md:h-2 bg-gray-200 dark:bg-gray-700 rounded w-3/4"></div>
                  <div className="h-1 md:h-1.5 bg-gray-200 dark:bg-gray-700 rounded w-1/2"></div>
                </div>
              </div>
            </div>

            {/* Document content lines */}
            <div className="p-2 md:p-3 space-y-1.5 md:space-y-2">
              {[...Array(6)].map((_, i) => (
                <div
                  key={`line-${i}`}
                  className="h-1 md:h-1.5 bg-gray-100 dark:bg-gray-800 rounded"
                  style={{ width: `${Math.random() * 30 + 60}%` }}
                />
              ))}
            </div>

            {/* Signature fields on document */}
            <AnimatePresence>
              {fields.map((field) => field.merged && (
                <motion.div
                  key={`merged-${field.id}`}
                  className="absolute"
                  initial={false}
                  animate={{
                    left: field.id === 0 ? '10%' : field.id === 1 ? '55%' : '10%',
                    top: field.id === 0 ? '45%' : field.id === 1 ? '45%' : '70%',
                  }}
                  style={{
                    width: '35%',
                  }}
                >
                  <div className={`border-2 ${
                    signatures.includes(field.id) 
                      ? 'border-green-500 bg-green-50 dark:bg-green-900/20' 
                      : 'border-dashed border-[#286efb] dark:border-[#A9C5FF] bg-blue-50 dark:bg-[#286efb]/10'
                  } rounded p-1 md:p-1.5 transition-all duration-300`}>
                    {signatures.includes(field.id) ? (
                      <motion.div
                        initial={{ scale: 0, rotate: -20 }}
                        animate={{ scale: 1, rotate: 0 }}
                        transition={{ type: 'spring', stiffness: 300 }}
                        className="text-center"
                      >
                        <div className="text-[#286efb] dark:text-[#A9C5FF] font-script text-xs md:text-sm italic">
                          {field.id === 0 ? 'J. Smith' : field.id === 1 ? 'A. Lee' : 'M. Chen'}
                        </div>
                      </motion.div>
                    ) : (
                      <div className="h-3 md:h-4 flex items-center justify-center">
                        <Signature className="w-2.5 h-2.5 md:w-3 md:h-3 text-[#286efb] dark:text-[#A9C5FF]" />
                      </div>
                    )}
                  </div>
                </motion.div>
              ))}
            </AnimatePresence>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Flying signature field icons */}
      <AnimatePresence>
        {fields.map((field) => !field.merged && (
          <motion.div
            key={`field-${field.id}`}
            className="absolute z-20"
            initial={{ 
              opacity: 0,
              scale: 0,
              x: field.position.x * 2,
              y: field.position.y * 2,
              rotate: Math.random() * 360
            }}
            animate={{ 
              opacity: 1,
              scale: 1,
              x: field.position.x,
              y: field.position.y,
              rotate: 0
            }}
            exit={{ 
              opacity: 0,
              scale: 0.5,
              x: 0,
              y: 0
            }}
            transition={{ 
              type: 'spring',
              stiffness: 200,
              damping: 15
            }}
          >
            <div className="w-16 h-16 md:w-20 md:h-20 bg-[#286efb] dark:bg-[#A9C5FF] rounded-xl md:rounded-2xl flex items-center justify-center shadow-lg">
              {field.id === 0 ? (
                <Signature className="w-8 h-8 md:w-10 md:h-10 text-white dark:text-[#1a1a1a]" />
              ) : field.id === 1 ? (
                <UserCheck className="w-8 h-8 md:w-10 md:h-10 text-white dark:text-[#1a1a1a]" />
              ) : (
                <FileText className="w-8 h-8 md:w-10 md:h-10 text-white dark:text-[#1a1a1a]" />
              )}
            </div>
          </motion.div>
        ))}
      </AnimatePresence>

      {/* Single centered checkmark for complete document */}
      <AnimatePresence>
        {stage === 'complete' && (
          <motion.div
            className="absolute z-30 bottom-0 md:bottom-4"
            initial={{ 
              opacity: 0,
              scale: 0,
              y: 50
            }}
            animate={{ 
              opacity: 1,
              scale: 1,
              y: 0,
              rotate: [0, -10, 10, 0]
            }}
            exit={{ opacity: 0, scale: 0 }}
            transition={{ 
              type: 'spring',
              stiffness: 300,
              damping: 20,
              rotate: { duration: 0.6, delay: 0.3 }
            }}
          >
            <div className="w-20 h-20 md:w-24 md:h-24 bg-green-500 rounded-full flex items-center justify-center shadow-xl border-4 border-white dark:border-[#1a1a1a]">
              <CheckCircle className="w-12 h-12 md:w-14 md:h-14 text-white" strokeWidth={3} />
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  )
}
