'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { Clock, Calendar, CheckCircle2, User, FileCheck, TrendingUp } from 'lucide-react'
import Image from 'next/image'

type AnimationStage = 'idle' | 'employees' | 'tracking' | 'approval' | 'complete'

interface Employee {
  id: number
  name: string
  role: string
  avatarUrl: string
  status: 'active' | 'pending' | 'approved'
  position: { x: number; y: number }
}

const employeesData: Employee[] = [
  {
    id: 1,
    name: 'Alex Turner',
    role: 'Developer',
    avatarUrl: 'https://images.unsplash.com/photo-1500648767791-00dcc994a43e?w=100&h=100&fit=crop&crop=faces',
    status: 'active',
    position: { x: -100, y: -80 }
  },
  {
    id: 2,
    name: 'Emma Wilson',
    role: 'Designer',
    avatarUrl: 'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=100&h=100&fit=crop&crop=faces',
    status: 'active',
    position: { x: 100, y: -80 }
  },
  {
    id: 3,
    name: 'James Chen',
    role: 'Manager',
    avatarUrl: 'https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=100&h=100&fit=crop&crop=faces',
    status: 'active',
    position: { x: 0, y: 80 }
  }
]

export default function DigitalHRAnimationPanel() {
  const [stage, setStage] = useState<AnimationStage>('idle')
  const [employees, setEmployees] = useState<Employee[]>([])
  const [showTimeTracking, setShowTimeTracking] = useState(false)
  const [showLeaveRequest, setShowLeaveRequest] = useState(false)
  const [isVisible, setIsVisible] = useState(false)
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  useEffect(() => {
    if (!isVisible) return

    const runAnimation = async () => {
      // Reset
      setStage('idle')
      setEmployees([])
      setShowTimeTracking(false)
      setShowLeaveRequest(false)
      await new Promise(resolve => setTimeout(resolve, 500))

      // Employees appear
      setStage('employees')
      for (let i = 0; i < employeesData.length; i++) {
        await new Promise(resolve => setTimeout(resolve, 400))
        setEmployees(prev => [...prev, employeesData[i]])
      }

      await new Promise(resolve => setTimeout(resolve, 800))

      // Time tracking icons appear
      setStage('tracking')
      setShowTimeTracking(true)
      await new Promise(resolve => setTimeout(resolve, 1000))

      // Leave request appears and gets approved
      setStage('approval')
      setShowLeaveRequest(true)
      await new Promise(resolve => setTimeout(resolve, 1200))

      // Approve the request
      setEmployees(prev => prev.map(emp => 
        emp.id === 2 ? { ...emp, status: 'approved' } : emp
      ))

      await new Promise(resolve => setTimeout(resolve, 800))

      // Complete state with success metrics
      setStage('complete')
      await new Promise(resolve => setTimeout(resolve, 2000))

      // Restart
      runAnimation()
    }

    runAnimation()
  }, [isVisible])

  return (
    <div 
      ref={panelRef}
      className={`relative w-full max-w-2xl h-[400px] md:h-[550px] mx-auto flex items-center justify-center overflow-visible transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Central Hub/Platform */}
      <AnimatePresence mode="wait">
        {stage !== 'idle' && (
          <motion.div
            className="relative w-56 h-56 md:w-72 md:h-72 bg-gradient-to-br from-gray-50 to-gray-100 dark:from-[#1a1a1a] dark:to-[#0f0f0f] rounded-3xl md:rounded-[2rem] border-2 border-gray-200 dark:border-gray-700 shadow-2xl z-10"
            initial={{ opacity: 0, scale: 0.3, rotateX: -90 }}
            animate={{ 
              opacity: 1, 
              scale: 1, 
              rotateX: 0,
            }}
            exit={{ opacity: 0, scale: 0.3 }}
            transition={{ 
              type: 'spring', 
              stiffness: 200, 
              damping: 20
            }}
            style={{
              filter: 'drop-shadow(0 20px 40px rgba(0, 0, 0, 0.15))'
            }}
          >
            {/* Central User Icon - Fixed and Smaller */}
            <div className="absolute inset-0 flex items-center justify-center z-0">
              <motion.div
                initial={{ scale: 0, opacity: 0 }}
                animate={{ scale: 1, opacity: 1 }}
                transition={{ type: 'spring', stiffness: 300, delay: 0.3 }}
              >
                <div className="w-8 h-8 md:w-10 md:h-10 bg-gray-200 dark:bg-gray-800 rounded-full flex items-center justify-center">
                  <User className="w-4 h-4 md:w-5 md:h-5 text-gray-600 dark:text-gray-300" />
                </div>
              </motion.div>
            </div>

            {/* Connection lines - Behind user icon */}
            <svg className="absolute inset-0 w-full h-full pointer-events-none -z-10" style={{ overflow: 'visible' }}>
              <AnimatePresence>
                {employees.map((emp, idx) => (
                  <motion.line
                    key={`line-${emp.id}`}
                    x1="50%"
                    y1="50%"
                    x2={`calc(50% + ${emp.position.x}px)`}
                    y2={`calc(50% + ${emp.position.y}px)`}
                    stroke="currentColor"
                    strokeWidth="2"
                    strokeDasharray="4 4"
                    className="text-gray-300 dark:text-gray-700"
                    initial={{ pathLength: 0, opacity: 0 }}
                    animate={{ pathLength: 1, opacity: 1 }}
                    exit={{ pathLength: 0, opacity: 0 }}
                    transition={{ duration: 0.6, delay: idx * 0.2 }}
                  />
                ))}
              </AnimatePresence>
            </svg>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Employee Cards floating around */}
      <AnimatePresence>
        {employees.map((employee, idx) => (
          <motion.div
            key={`employee-${employee.id}`}
            className="absolute z-20"
            initial={{ 
              opacity: 0,
              scale: 0,
              x: 0,
              y: 0,
              rotate: Math.random() * 360 - 180
            }}
            animate={{ 
              opacity: 1,
              scale: 1,
              x: employee.position.x,
              y: employee.position.y,
              rotate: 0
            }}
            exit={{ 
              opacity: 0,
              scale: 0,
              x: 0,
              y: 0
            }}
            transition={{ 
              type: 'spring',
              stiffness: 200,
              damping: 20,
              delay: idx * 0.2
            }}
          >
            <motion.div 
              className={`relative bg-white dark:bg-[#1a1a1a] rounded-xl md:rounded-2xl p-2.5 md:p-3 shadow-xl border-2 ${
                employee.status === 'approved' 
                  ? 'border-green-500 dark:border-green-400' 
                  : 'border-gray-200 dark:border-gray-700'
              }`}
              animate={employee.status === 'approved' ? {
                scale: [1, 1.1, 1],
                boxShadow: [
                  '0 10px 30px rgba(0, 0, 0, 0.1)',
                  '0 10px 40px rgba(34, 197, 94, 0.4)',
                  '0 10px 30px rgba(0, 0, 0, 0.1)'
                ]
              } : {}}
              transition={{ duration: 0.6 }}
            >
              <div className="flex items-center gap-2 md:gap-3">
                <div className="relative w-10 h-10 md:w-12 md:h-12 rounded-full overflow-hidden border-2 border-gray-200 dark:border-gray-700 flex-shrink-0">
                  <Image
                    src={employee.avatarUrl}
                    alt={employee.name}
                    fill
                    className="object-cover"
                    sizes="48px"
                  />
                </div>
                <div className="min-w-0">
                  <p className="text-xs md:text-sm font-semibold text-gray-800 dark:text-white truncate">
                    {employee.name}
                  </p>
                  <p className="text-[10px] md:text-xs text-gray-500 dark:text-gray-400 truncate">
                    {employee.role}
                  </p>
                </div>
              </div>
              {employee.status === 'approved' && (
                <motion.div
                  initial={{ scale: 0 }}
                  animate={{ scale: 1 }}
                  className="absolute -top-2 -right-2 w-6 h-6 md:w-7 md:h-7 bg-green-500 rounded-full flex items-center justify-center border-2 border-white dark:border-[#1a1a1a]"
                >
                  <CheckCircle2 className="w-3.5 h-3.5 md:w-4 md:h-4 text-white" />
                </motion.div>
              )}
            </motion.div>
          </motion.div>
        ))}
      </AnimatePresence>

      {/* Floating Time Tracking Badge */}
      <AnimatePresence>
        {showTimeTracking && (
          <motion.div
            className="absolute z-30 top-0 md:top-8 left-1/2 -translate-x-1/2"
            initial={{ opacity: 0, y: -50, scale: 0.5 }}
            animate={{ opacity: 1, y: 0, scale: 1 }}
            exit={{ opacity: 0, y: -50, scale: 0.5 }}
            transition={{ type: 'spring', stiffness: 300, damping: 20 }}
          >
            <div className="bg-gradient-to-r from-gray-100 to-gray-50 dark:from-gray-800 dark:to-gray-900 border-2 border-gray-200 dark:border-gray-700 rounded-xl md:rounded-2xl px-3 md:px-4 py-2 md:py-2.5 shadow-lg flex items-center gap-2 md:gap-3">
              <Clock className="w-4 h-4 md:w-5 md:h-5 text-gray-700 dark:text-gray-300" />
              <div>
                <p className="text-[10px] md:text-xs font-semibold text-gray-800 dark:text-white">
                  Active Tracking
                </p>
                <p className="text-[9px] md:text-[10px] text-gray-500 dark:text-gray-400">
                  3 employees clocked in
                </p>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Leave Request Notification */}
      <AnimatePresence>
        {showLeaveRequest && (
          <motion.div
            className="absolute z-30 bottom-0 md:bottom-8 left-1/2 -translate-x-1/2"
            initial={{ opacity: 0, y: 50, scale: 0.5 }}
            animate={{ opacity: 1, y: 0, scale: 1 }}
            exit={{ opacity: 0, y: 50, scale: 0.5 }}
            transition={{ type: 'spring', stiffness: 300, damping: 20 }}
          >
            <motion.div 
              className="bg-gradient-to-r from-gray-100 to-gray-50 dark:from-gray-800 dark:to-gray-900 border-2 border-gray-200 dark:border-gray-700 rounded-xl md:rounded-2xl px-3 md:px-4 py-2 md:py-2.5 shadow-lg"
              animate={employees.find(e => e.id === 2)?.status === 'approved' ? {
                borderColor: ['#d1d5db', '#22c55e', '#22c55e'],
                backgroundColor: ['#f9fafb', '#f0fdf4', '#f9fafb']
              } : {}}
              transition={{ duration: 0.6 }}
            >
              <div className="flex items-center gap-2 md:gap-3">
                <Calendar className="w-4 h-4 md:w-5 md:h-5 text-gray-700 dark:text-gray-300" />
                <div>
                  <p className="text-[10px] md:text-xs font-semibold text-gray-800 dark:text-white flex items-center gap-1.5">
                    Leave Request
                    {employees.find(e => e.id === 2)?.status === 'approved' && (
                      <motion.span
                        initial={{ scale: 0 }}
                        animate={{ scale: 1 }}
                        className="text-green-600 dark:text-green-400"
                      >
                        <CheckCircle2 className="w-3 h-3 md:w-3.5 md:h-3.5" />
                      </motion.span>
                    )}
                  </p>
                  <p className="text-[9px] md:text-[10px] text-gray-500 dark:text-gray-400">
                    {employees.find(e => e.id === 2)?.status === 'approved' ? 'Approved ✓' : 'Pending approval'}
                  </p>
                </div>
              </div>
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Success Metrics - Complete Stage */}
      <AnimatePresence>
        {stage === 'complete' && (
          <motion.div
            className="absolute z-30 right-0 md:right-8 top-1/2 -translate-y-1/2"
            initial={{ opacity: 0, x: 50, scale: 0.8 }}
            animate={{ opacity: 1, x: 0, scale: 1 }}
            exit={{ opacity: 0, x: 50, scale: 0.8 }}
            transition={{ type: 'spring', stiffness: 300, damping: 20 }}
          >
            <div className="bg-green-50 dark:bg-green-900/20 border-2 border-green-500/50 rounded-xl md:rounded-2xl p-2.5 md:p-3 shadow-lg">
              <div className="flex items-center gap-2">
                <TrendingUp className="w-5 h-5 md:w-6 md:h-6 text-green-600 dark:text-green-400" />
                <div>
                  <p className="text-xs md:text-sm font-bold text-green-700 dark:text-green-400">
                    100% Digital
                  </p>
                  <p className="text-[10px] md:text-xs text-green-600 dark:text-green-500">
                    Paperless HR
                  </p>
                </div>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  )
}
