'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { TrendingUp, Users, DollarSign, BarChart3, Activity } from 'lucide-react'

interface MetricCard {
  id: number
  icon: React.ReactNode
  label: string
  value: string
  trend: string
  position: { x: number; y: number }
  gridPosition: { row: number; col: number }
}

const metricsData: MetricCard[] = [
  {
    id: 1,
    icon: <DollarSign className="w-5 h-5 md:w-6 md:h-6" />,
    label: 'Revenue',
    value: '$45.2K',
    trend: '+12%',
    position: { x: -150, y: -100 },
    gridPosition: { row: 1, col: 1 }
  },
  {
    id: 2,
    icon: <Users className="w-5 h-5 md:w-6 md:h-6" />,
    label: 'Customers',
    value: '1,234',
    trend: '+8%',
    position: { x: 150, y: -100 },
    gridPosition: { row: 1, col: 2 }
  },
  {
    id: 3,
    icon: <TrendingUp className="w-5 h-5 md:w-6 md:h-6" />,
    label: 'Growth',
    value: '23%',
    trend: '+5%',
    position: { x: -150, y: 100 },
    gridPosition: { row: 2, col: 1 }
  },
  {
    id: 4,
    icon: <Activity className="w-5 h-5 md:w-6 md:h-6" />,
    label: 'Activity',
    value: '892',
    trend: '+15%',
    position: { x: 150, y: 100 },
    gridPosition: { row: 2, col: 2 }
  }
]

export default function DashboardVisualizationPanel() {
  const [stage, setStage] = useState<'scattered' | 'building' | 'complete'>('scattered')
  const [builtCards, setBuiltCards] = useState<number[]>([])
  const [showChart, setShowChart] = useState(false)
  const [isVisible, setIsVisible] = useState(false)
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  useEffect(() => {
    if (!isVisible) return

    const runAnimation = async () => {
      // Reset
      setStage('scattered')
      setBuiltCards([])
      setShowChart(false)
      await new Promise(resolve => setTimeout(resolve, 1000))

      // Build dashboard - cards snap into grid
      setStage('building')
      for (let i = 0; i < metricsData.length; i++) {
        await new Promise(resolve => setTimeout(resolve, 600))
        setBuiltCards(prev => [...prev, i])
      }

      await new Promise(resolve => setTimeout(resolve, 800))

      // Show complete with chart
      setStage('complete')
      setShowChart(true)
      await new Promise(resolve => setTimeout(resolve, 2500))

      // Restart
      runAnimation()
    }

    runAnimation()
  }, [isVisible])

  return (
    <div 
      ref={panelRef}
      className={`relative w-full max-w-2xl h-[400px] md:h-[550px] mx-auto flex items-center justify-center overflow-visible transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Dashboard Title */}
      <AnimatePresence>
        {stage === 'complete' && (
          <motion.div
            className="absolute top-0 md:top-8 left-1/2 -translate-x-1/2 z-30"
            initial={{ opacity: 0, y: -30 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: -30 }}
            transition={{ type: 'spring', stiffness: 300 }}
          >
            <div className="bg-[#253237] dark:bg-white px-4 md:px-6 py-2 md:py-3 rounded-xl shadow-lg">
              <div className="flex items-center gap-2">
                <BarChart3 className="w-4 h-4 md:w-5 md:h-5 text-white dark:text-[#253237]" />
                <span className="text-sm md:text-base font-bold text-white dark:text-[#253237]">
                  Live Dashboard
                </span>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Central Dashboard Grid */}
      <div className="relative w-72 md:w-96">
        {/* Background Grid */}
        {stage !== 'scattered' && (
          <motion.div
            initial={{ opacity: 0, scale: 0.8 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.5 }}
            className="absolute inset-0 -m-4 md:-m-6 bg-white/50 dark:bg-[#1a1a1a]/50 backdrop-blur-sm rounded-2xl border-2 border-dashed border-gray-300 dark:border-gray-600"
          />
        )}

        {/* Metric Cards */}
        {metricsData.map((metric, index) => {
          const isBuilt = builtCards.includes(index)
          
          return (
            <motion.div
              key={metric.id}
              className="absolute"
              initial={false}
              animate={
                isBuilt || stage === 'complete'
                  ? {
                      x: (metric.gridPosition.col - 1) * 160 - 70,
                      y: (metric.gridPosition.row - 1) * 120 - 50,
                      rotate: 0,
                      scale: 1
                    }
                  : {
                      x: metric.position.x,
                      y: metric.position.y,
                      rotate: Math.random() * 40 - 20,
                      scale: 0.9
                    }
              }
              transition={{
                type: 'spring',
                stiffness: 200,
                damping: 20
              }}
            >
              <motion.div
                className="bg-white dark:bg-[#1a1a1a] border-2 border-gray-200 dark:border-gray-700 rounded-xl p-3 md:p-4 shadow-xl w-32 md:w-40"
                animate={
                  stage === 'complete'
                    ? {
                        boxShadow: [
                          '0 10px 30px rgba(0,0,0,0.1)',
                          '0 15px 40px rgba(37,50,55,0.3)',
                          '0 10px 30px rgba(0,0,0,0.1)'
                        ]
                      }
                    : {}
                }
                transition={{ duration: 2, repeat: Infinity }}
              >
                <div className="flex items-center gap-2 mb-2">
                  <div className="w-8 h-8 md:w-10 md:h-10 rounded-lg bg-[#253237] dark:bg-white flex items-center justify-center text-white dark:text-[#253237]">
                    {metric.icon}
                  </div>
                  {isBuilt && (
                    <motion.div
                      initial={{ scale: 0, rotate: -180 }}
                      animate={{ scale: 1, rotate: 0 }}
                      transition={{ type: 'spring', delay: 0.2 }}
                      className="ml-auto"
                    >
                      <div className="text-[9px] md:text-xs font-bold text-green-600 dark:text-green-400">
                        {metric.trend}
                      </div>
                    </motion.div>
                  )}
                </div>
                <p className="text-[10px] md:text-xs text-gray-600 dark:text-gray-400 font-medium mb-1">
                  {metric.label}
                </p>
                <p className="text-base md:text-xl font-bold text-[#253237] dark:text-white">
                  {metric.value}
                </p>
              </motion.div>
            </motion.div>
          )
        })}
      </div>

      {/* Chart Visualization - Center Bottom */}
      <AnimatePresence>
        {showChart && (
          <motion.div
            className="absolute bottom-0 md:bottom-8 left-1/2 -translate-x-1/2 z-30"
            initial={{ opacity: 0, y: 50, scale: 0.8 }}
            animate={{ opacity: 1, y: 0, scale: 1 }}
            exit={{ opacity: 0, y: 50, scale: 0.8 }}
            transition={{ type: 'spring', stiffness: 300, damping: 20 }}
          >
            <div className="bg-white dark:bg-[#1a1a1a] border-2 border-gray-200 dark:border-gray-700 rounded-xl p-3 md:p-4 shadow-xl">
              <div className="flex items-end gap-1.5 md:gap-2 h-16 md:h-20">
                {[40, 65, 45, 80, 55, 70].map((height, i) => (
                  <motion.div
                    key={i}
                    className="w-4 md:w-6 bg-[#253237] dark:bg-white rounded-t"
                    initial={{ height: 0 }}
                    animate={{ height: `${height}%` }}
                    transition={{ delay: i * 0.1, duration: 0.5, ease: 'easeOut' }}
                  />
                ))}
              </div>
              <div className="mt-2 flex items-center justify-center gap-1.5">
                <div className="w-1.5 h-1.5 rounded-full bg-[#253237] dark:bg-white animate-pulse" />
                <p className="text-[9px] md:text-xs font-medium text-gray-600 dark:text-gray-400">
                  Real-time Analytics
                </p>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  )
}
