'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { Calendar, Edit, Send, BarChart2, CheckCircle2 } from 'lucide-react'

type WorkflowStage = 'idle' | 'planning' | 'creating' | 'publishing' | 'complete'

interface ContentPost {
  id: number
  platform: string
  icon: string
  published: boolean
}

const platforms: ContentPost[] = [
  { id: 1, platform: 'Twitter', icon: '𝕏', published: false },
  { id: 2, platform: 'LinkedIn', icon: 'in', published: false },
  { id: 3, platform: 'Instagram', icon: '📷', published: false }
]

export default function ContentWorkflowAnimationPanel() {
  const [stage, setStage] = useState<WorkflowStage>('idle')
  const [publishedPlatforms, setPublishedPlatforms] = useState<number[]>([])
  const [showAnalytics, setShowAnalytics] = useState(false)
  const [isVisible, setIsVisible] = useState(false)
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  useEffect(() => {
    if (!isVisible) return

    const runWorkflow = async () => {
      // Reset
      setStage('idle')
      setPublishedPlatforms([])
      setShowAnalytics(false)
      await new Promise(resolve => setTimeout(resolve, 800))

      // Planning stage
      setStage('planning')
      await new Promise(resolve => setTimeout(resolve, 1500))

      // Creating stage
      setStage('creating')
      await new Promise(resolve => setTimeout(resolve, 1500))

      // Publishing stage - publish to each platform
      setStage('publishing')
      for (let i = 0; i < platforms.length; i++) {
        await new Promise(resolve => setTimeout(resolve, 700))
        setPublishedPlatforms(prev => [...prev, i])
      }

      await new Promise(resolve => setTimeout(resolve, 800))

      // Complete with analytics
      setStage('complete')
      setShowAnalytics(true)
      await new Promise(resolve => setTimeout(resolve, 2500))

      // Restart
      runWorkflow()
    }

    runWorkflow()
  }, [isVisible])

  return (
    <div 
      ref={panelRef}
      className={`relative w-full max-w-2xl h-[400px] md:h-[550px] mx-auto flex items-center justify-center overflow-visible transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Workflow Stages - Vertical Flow */}
      <div className="relative flex flex-col items-center gap-8 md:gap-12">
        
        {/* Stage 1: Planning */}
        <motion.div
          className="relative z-10"
          animate={stage === 'planning' ? { scale: [1, 1.1, 1] } : {}}
          transition={{ duration: 0.6 }}
        >
          <div className={`w-24 h-24 md:w-32 md:h-32 rounded-2xl flex items-center justify-center transition-all duration-500 ${
            stage !== 'idle'
              ? 'bg-gray-800 dark:bg-gray-200 shadow-xl'
              : 'bg-gray-200 dark:bg-gray-800 shadow-lg'
          }`}>
            <Calendar className={`w-10 h-10 md:w-14 md:h-14 transition-colors ${
              stage !== 'idle'
                ? 'text-white dark:text-gray-900'
                : 'text-gray-600 dark:text-gray-400'
            }`} />
          </div>
          <p className="text-center mt-2 text-xs md:text-sm font-semibold text-gray-700 dark:text-gray-300">
            Plan
          </p>
          {stage === 'planning' && (
            <motion.div
              className="absolute -right-12 md:-right-16 top-1/2 -translate-y-1/2"
              initial={{ opacity: 0, scale: 0 }}
              animate={{ opacity: 1, scale: 1 }}
              transition={{ delay: 0.3 }}
            >
              <div className="bg-white dark:bg-[#1a1a1a] border-2 border-gray-300 dark:border-gray-700 rounded-lg p-2 shadow-lg text-[10px] md:text-xs whitespace-nowrap">
                📅 Scheduling...
              </div>
            </motion.div>
          )}
        </motion.div>

        {/* Connector Line 1 */}
        <motion.div
          className="absolute top-28 md:top-36 w-0.5 h-8 md:h-12 bg-gray-400 dark:bg-gray-600"
          initial={{ scaleY: 0 }}
          animate={{ scaleY: stage !== 'idle' && stage !== 'planning' ? 1 : 0 }}
          transition={{ duration: 0.5 }}
          style={{ transformOrigin: 'top' }}
        />

        {/* Stage 2: Creating */}
        <motion.div
          className="relative z-10"
          animate={stage === 'creating' ? { scale: [1, 1.1, 1] } : {}}
          transition={{ duration: 0.6 }}
        >
          <div className={`w-24 h-24 md:w-32 md:h-32 rounded-2xl flex items-center justify-center transition-all duration-500 ${
            stage === 'creating' || (stage === 'publishing' || stage === 'complete')
              ? 'bg-gray-800 dark:bg-gray-200 shadow-xl'
              : 'bg-gray-200 dark:bg-gray-800 shadow-lg'
          }`}>
            <Edit className={`w-10 h-10 md:w-14 md:h-14 transition-colors ${
              stage === 'creating' || (stage === 'publishing' || stage === 'complete')
                ? 'text-white dark:text-gray-900'
                : 'text-gray-600 dark:text-gray-400'
            }`} />
          </div>
          <p className="text-center mt-2 text-xs md:text-sm font-semibold text-gray-700 dark:text-gray-300">
            Create
          </p>
          {stage === 'creating' && (
            <motion.div
              className="absolute -right-12 md:-right-16 top-1/2 -translate-y-1/2"
              initial={{ opacity: 0, scale: 0 }}
              animate={{ opacity: 1, scale: 1 }}
              transition={{ delay: 0.3 }}
            >
              <div className="bg-white dark:bg-[#1a1a1a] border-2 border-gray-300 dark:border-gray-700 rounded-lg p-2 shadow-lg text-[10px] md:text-xs whitespace-nowrap">
                ✍️ Writing...
              </div>
            </motion.div>
          )}
        </motion.div>

        {/* Connector Line 2 */}
        <motion.div
          className="absolute top-[14.5rem] md:top-[19rem] w-0.5 h-8 md:h-12 bg-gray-400 dark:bg-gray-600"
          initial={{ scaleY: 0 }}
          animate={{ scaleY: stage === 'publishing' || stage === 'complete' ? 1 : 0 }}
          transition={{ duration: 0.5 }}
          style={{ transformOrigin: 'top' }}
        />

        {/* Stage 3: Publishing */}
        <motion.div
          className="relative z-10"
          animate={stage === 'publishing' ? { scale: [1, 1.1, 1] } : {}}
          transition={{ duration: 0.6 }}
        >
          <div className={`w-24 h-24 md:w-32 md:h-32 rounded-2xl flex items-center justify-center transition-all duration-500 ${
            stage === 'publishing' || stage === 'complete'
              ? 'bg-gray-800 dark:bg-gray-200 shadow-xl'
              : 'bg-gray-200 dark:bg-gray-800 shadow-lg'
          }`}>
            <Send className={`w-10 h-10 md:w-14 md:h-14 transition-colors ${
              stage === 'publishing' || stage === 'complete'
                ? 'text-white dark:text-gray-900'
                : 'text-gray-600 dark:text-gray-400'
            }`} />
          </div>
          <p className="text-center mt-2 text-xs md:text-sm font-semibold text-gray-700 dark:text-gray-300">
            Publish
          </p>
        </motion.div>

        {/* Platform Icons - Fly to right side */}
        <AnimatePresence>
          {stage === 'publishing' || stage === 'complete' ? (
            <div className="absolute left-32 md:left-40 top-[16rem] md:top-[21rem] flex flex-col gap-2 md:gap-3 z-20">
              {platforms.map((platform, index) => {
                const isPublished = publishedPlatforms.includes(index)
                
                return (
                  <motion.div
                    key={platform.id}
                    initial={{ opacity: 0, x: -100, scale: 0.5 }}
                    animate={isPublished ? { opacity: 1, x: 0, scale: 1 } : {}}
                    transition={{ type: 'spring', stiffness: 200, damping: 20 }}
                    className="relative"
                  >
                    <div className="bg-white dark:bg-[#1a1a1a] border-2 border-gray-300 dark:border-gray-700 rounded-lg p-2 md:p-3 shadow-lg flex items-center gap-2 md:gap-3 min-w-[100px] md:min-w-[140px]">
                      <div className="text-lg md:text-2xl">{platform.icon}</div>
                      <div className="flex-1">
                        <p className="text-[10px] md:text-xs font-semibold text-gray-800 dark:text-white">
                          {platform.platform}
                        </p>
                        <p className="text-[9px] md:text-[10px] text-gray-600 dark:text-gray-400">
                          Published
                        </p>
                      </div>
                      {isPublished && (
                        <motion.div
                          initial={{ scale: 0, rotate: -180 }}
                          animate={{ scale: 1, rotate: 0 }}
                          transition={{ delay: 0.2 }}
                        >
                          <CheckCircle2 className="w-4 h-4 md:w-5 md:h-5 text-green-600 dark:text-green-400" />
                        </motion.div>
                      )}
                    </div>
                  </motion.div>
                )
              })}
            </div>
          ) : null}
        </AnimatePresence>
      </div>

      {/* Analytics Panel - Bottom */}
      <AnimatePresence>
        {showAnalytics && (
          <motion.div
            className="absolute bottom-0 md:bottom-8 left-1/2 -translate-x-1/2 z-30"
            initial={{ opacity: 0, y: 50, scale: 0.8 }}
            animate={{ opacity: 1, y: 0, scale: 1 }}
            exit={{ opacity: 0, y: 50, scale: 0.8 }}
            transition={{ type: 'spring', stiffness: 300, damping: 20 }}
          >
            <div className="bg-white dark:bg-[#1a1a1a] border-2 border-gray-300 dark:border-gray-700 rounded-xl p-3 md:p-4 shadow-2xl">
              <div className="flex items-center gap-2 md:gap-3 mb-2">
                <BarChart2 className="w-4 h-4 md:w-5 md:h-5 text-gray-700 dark:text-gray-300" />
                <span className="text-xs md:text-sm font-bold text-gray-800 dark:text-white">
                  Performance
                </span>
              </div>
              <div className="flex gap-4 md:gap-6">
                <div>
                  <p className="text-lg md:text-2xl font-bold text-gray-800 dark:text-white">1.2K</p>
                  <p className="text-[9px] md:text-xs text-gray-600 dark:text-gray-400">Reach</p>
                </div>
                <div>
                  <p className="text-lg md:text-2xl font-bold text-gray-800 dark:text-white">234</p>
                  <p className="text-[9px] md:text-xs text-gray-600 dark:text-gray-400">Engagement</p>
                </div>
                <div>
                  <p className="text-lg md:text-2xl font-bold text-green-600 dark:text-green-400">+15%</p>
                  <p className="text-[9px] md:text-xs text-gray-600 dark:text-gray-400">Growth</p>
                </div>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Success Badge - Top */}
      <AnimatePresence>
        {stage === 'complete' && (
          <motion.div
            className="absolute top-0 md:top-8 left-1/2 -translate-x-1/2 z-30"
            initial={{ opacity: 0, y: -30, scale: 0.8 }}
            animate={{ opacity: 1, y: 0, scale: 1 }}
            exit={{ opacity: 0, y: -30, scale: 0.8 }}
            transition={{ type: 'spring', stiffness: 300, damping: 20 }}
          >
            <div className="bg-gray-800 dark:bg-gray-200 px-4 md:px-6 py-2 md:py-3 rounded-xl shadow-xl">
              <div className="flex items-center gap-2 md:gap-3">
                <CheckCircle2 className="w-4 h-4 md:w-5 md:h-5 text-white dark:text-gray-900" />
                <span className="text-xs md:text-sm font-bold text-white dark:text-gray-900">
                  Workflow Complete!
                </span>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  )
}
