"use client";

import { useState } from "react";
import { X, Sparkles, TrendingDown, Check } from "lucide-react";
import accountingIcon from "../../public/images/quickbooks.svg";
import ecommerceIcon from "../../public/images/shopify.svg";
import crmIcon from "../../public/images/salesforce.svg";
import chatIcon from "../../public/images/slack.svg";
import hrIcon from "../../public/images/bamboohr.svg";
import signIcon from "../../public/images/docusign.svg";
import inventoryIcon from "../../public/images/inflow.svg";
import documentsIcon from "../../public/images/sharepoint.svg";
import posIcon from "../../public/images/lightspeed.svg";
import expensesIcon from "../../public/images/expensify.svg";
import appointmentIcon from "../../public/images/calendly.svg";
import purchaseIcon from "../../public/images/coupa.svg";
import socialIcon from "../../public/images/hootsuite.svg";
import planningIcon from "../../public/images/asana.svg";
import emailingIcon from "../../public/images/google.svg";
import timesheetIcon from "../../public/images/harvest.svg";
import helpdeskIcon from "../../public/images/zendesk.svg";
import eventsIcon from "../../public/images/eventbrite.svg";
import mrpIcon from "../../public/images/katanamrp.svg";
import Image from "next/image";

interface App {
  id: number;
  name: string;
  icon: string;
  price: number;
}

const apps: App[] = [
  {
    id: 2,
    name: "Accounting",
    icon: accountingIcon,
    price: 20,
  },
  {
    id: 3,
    name: "eCommerce",
    icon: ecommerceIcon,
    price: 20,
  },
  {
    id: 4,
    name: "CRM",
    icon: crmIcon,
    price: 20,
  },
  {
    id: 5,
    name: "Discuss",
    icon: chatIcon,
    price: 20,
  },
  {
    id: 7,
    name: "HR",
    icon: hrIcon,
    price: 20,
  },
  {
    id: 8,
    name: "Sign",
    icon: signIcon,
    price: 20,
  },
  {
    id: 9,
    name: "Inventory",
    icon: inventoryIcon,
    price: 20,
  },
  {
    id: 11,
    name: "Documents",
    icon: documentsIcon,
    price: 20,
  },
  {
    id: 12,
    name: "PoS",
    icon: posIcon,
    price: 20,
  },
  {
    id: 13,
    name: "Expenses",
    icon: expensesIcon,
    price: 20,
  },
  {
    id: 14,
    name: "Appointment",
    icon: appointmentIcon,
    price: 20,
  },
  {
    id: 15,
    name: "Purchase",
    icon: purchaseIcon,
    price: 20,
  },
  {
    id: 16,
    name: "Social",
    icon: socialIcon,
    price: 20,
  },
  {
    id: 17,
    name: "Planning",
    icon: planningIcon,
    price: 20,
  },
  {
    id: 18,
    name: "Emailing",
    icon: emailingIcon,
    price: 20,
  },
  {
    id: 19,
    name: "Timesheet",
    icon: timesheetIcon,
    price: 20,
  },
  {
    id: 20,
    name: "Helpdesk",
    icon: helpdeskIcon,
    price: 20,
  },
  {
    id: 21,
    name: "Events",
    icon: eventsIcon,
    price: 20,
  },
  {
    id: 22,
    name: "MRP",
    icon: mrpIcon,
    price: 20,
  },
];

const ODOO_MONTHLY_PER_USER = 2;

export default function PricingCalculator() {
  const [selectedApps, setSelectedApps] = useState<Record<number, boolean>>(
    () => {
      const defaultSelectedApps: Record<number, boolean> = {};
      for (let i = 1; i <= 7; i++) {
        defaultSelectedApps[i] = true;
      }
      return defaultSelectedApps;
    }
  );
  const [users, setUsers] = useState<number>(10);

  const toggleApp = (id: number) => {
    setSelectedApps((prev) => ({
      ...prev,
      [id]: !prev[id],
    }));
  };

  const removeApp = (id: number) => {
    setSelectedApps((prev) => {
      const next = { ...prev };
      delete next[id];
      return next;
    });
  };

  const replacedMonthlyPerUser = apps
    .filter((app) => selectedApps[app.id])
    .reduce((sum, app) => sum + app.price, 0);

  const replacedMonthlyTotal = replacedMonthlyPerUser * users;
  const replacedYearlyTotal = replacedMonthlyTotal * 12;

  const odooMonthlyTotal = ODOO_MONTHLY_PER_USER * users;
  const odooYearlyTotal = odooMonthlyTotal * 12;

  const savingsYearly = Math.max(0, replacedYearlyTotal - odooYearlyTotal);

  return (
    <section className="py-6 md:py-10 bg-white dark:bg-transparent relative overflow-hidden">
      {/* Background decoration */}
      <div className="absolute inset-0 -z-10 overflow-hidden pointer-events-none">
        <div className="absolute top-0 left-1/4 w-96 h-96 bg-gray-100/50 dark:bg-transparent rounded-full blur-3xl" />
        <div className="absolute bottom-0 right-1/4 w-96 h-96 bg-gray-100/50 dark:bg-transparent rounded-full blur-3xl" />
      </div>

      <div className="container max-w-7xl mx-auto px-4 md:px-6">
        {/* Header */}
        <div className="text-center mb-4 md:mb-6">
          <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-gray-100 dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E] mb-4">
            <TrendingDown className="w-4 h-4 text-[#253237] dark:text-white" />
            <span className="text-sm font-semibold text-gray-900 dark:text-white">
              Cost Savings Calculator
            </span>
          </div>
          
          <h2 className="text-2xl md:text-3xl font-bold mb-2 text-gray-900 dark:text-white">
            See How Much You Can <span className="text-[#253237] dark:text-white">Save</span>
          </h2>
          
          <p className="text-sm md:text-base text-gray-600 dark:text-[#94A3B8] max-w-2xl mx-auto">
            Calculate your potential savings by replacing multiple subscriptions with Clikkle&apos;s all-in-one platform
          </p>
        </div>

        <div className="flex flex-col lg:flex-row gap-3 md:gap-4">
          {/* Left Panel - App Selection */}
          <div className="flex-1 bg-white dark:bg-[#16171B] rounded-2xl p-3 md:p-5 border border-gray-200 dark:border-[#26282E] hover:shadow-xl dark:hover:shadow-2xl dark:hover:shadow-white/5 transition-all duration-300">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-bold text-gray-900 dark:text-white">
                Select Your Current Apps
              </h3>
              <div className="text-sm text-gray-600 dark:text-[#94A3B8]">
                {Object.keys(selectedApps).length} selected
              </div>
            </div>
            
            <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-2 md:gap-3">
              {apps.map((app) => (
                <div
                  key={app.id}
                  role="button"
                  onClick={() => toggleApp(app.id)}
                  className={`group relative cursor-pointer border rounded-xl p-3 flex flex-col items-center justify-center transition-all duration-300 hover:-translate-y-1 ${
                    selectedApps[app.id]
                      ? "border-[#253237] dark:border-white bg-gray-50 dark:bg-white/5 shadow-lg"
                      : "border-gray-200 dark:border-[#26282E] bg-white dark:bg-[#0B0C0E] hover:border-gray-300 dark:hover:border-[#3a3b42] hover:shadow-md"
                  }`}
                >
                  {selectedApps[app.id] && (
                    <>
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          removeApp(app.id);
                        }}
                        className="absolute top-2 right-2 p-1 rounded-md bg-white dark:bg-[#253237] hover:bg-gray-100 dark:hover:bg-[#1a1d22] transition-colors shadow-sm"
                        aria-label={`Remove ${app.name}`}
                      >
                        <X className="w-3 h-3 text-gray-600 dark:text-white" />
                      </button>
                      <div className="absolute top-2 left-2 w-5 h-5 rounded-full bg-[#253237] dark:bg-white flex items-center justify-center">
                        <Check className="w-3 h-3 text-white dark:text-[#253237]" />
                      </div>
                    </>
                  )}
                  <div className="mb-2 w-8 h-8 flex items-center justify-center">
                    <Image
                      src={app.icon}
                      alt={`${app.name} icon`}
                      className="w-6 h-6 object-contain"
                      width={24}
                      height={24}
                    />
                  </div>
                  <span className="font-semibold text-xs text-gray-900 dark:text-white text-center">
                    {app.name}
                  </span>
                  <div className={`text-[10px] mt-1.5 font-medium transition-colors ${
                    selectedApps[app.id]
                      ? "text-[#253237] dark:text-white"
                      : "text-gray-500 dark:text-[#94A3B8]"
                  }`}>
                    {selectedApps[app.id]
                      ? `$${app.price}/user`
                      : "Tap to add"}
                  </div>
                </div>
              ))}
            </div>
            
            {/* User Count Selector */}
            <div className="mt-8 pt-6 border-t border-gray-200 dark:border-[#26282E]">
              <div className="flex items-center justify-between gap-4 flex-wrap">
                <div className="text-sm font-medium text-gray-900 dark:text-white">
                  Number of users
                </div>
                <div className="flex items-center gap-3 bg-gray-50 dark:bg-[#0B0C0E] rounded-xl p-2 border border-gray-200 dark:border-[#26282E]">
                  <button
                    onClick={() => setUsers((u) => Math.max(1, u - 1))}
                    className="w-10 h-10 rounded-lg bg-white dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E] text-gray-900 dark:text-white hover:bg-[#253237] hover:text-white dark:hover:bg-white dark:hover:text-[#253237] transition-all font-bold"
                  >
                    —
                  </button>
                  <div className="min-w-[60px] text-center font-bold text-lg text-gray-900 dark:text-white">
                    {users}
                  </div>
                  <button
                    onClick={() => setUsers((u) => Math.min(999, u + 1))}
                    className="w-10 h-10 rounded-lg bg-white dark:bg-[#16171B] border border-gray-200 dark:border-[#26282E] text-gray-900 dark:text-white hover:bg-[#253237] hover:text-white dark:hover:bg-white dark:hover:text-[#253237] transition-all font-bold"
                  >
                    +
                  </button>
                </div>
              </div>
            </div>
          </div>

          {/* Right Panel - Cost Breakdown */}
          <aside className="w-full lg:w-[360px] space-y-3">
            {/* Selected Apps Summary */}
            <div className="bg-white dark:bg-[#16171B] rounded-2xl p-3 border border-gray-200 dark:border-[#26282E] hover:shadow-xl dark:hover:shadow-2xl dark:hover:shadow-white/5 transition-all duration-300">
              <h3 className="text-base font-bold mb-3 text-gray-900 dark:text-white">
                Cost Breakdown for {users} {users === 1 ? 'user' : 'users'}
              </h3>
              
              <div className="space-y-1.5 max-h-[200px] overflow-y-auto scrollbar-thin">
                {Object.keys(selectedApps).length === 0 ? (
                  <div className="py-6 text-center">
                    <Sparkles className="w-10 h-10 mx-auto mb-3 text-gray-300 dark:text-[#3a3b42]" />
                    <p className="text-sm text-gray-500 dark:text-[#94A3B8]">
                      Select apps above to see your cost breakdown
                    </p>
                  </div>
                ) : (
                  apps
                    .filter((app) => selectedApps[app.id])
                    .map((app) => (
                      <div
                        key={app.id}
                        className="flex items-center justify-between py-2"
                      >
                        <div className="flex items-center gap-3">
                          <div className="w-8 h-8 rounded-lg bg-gray-50 dark:bg-[#0B0C0E] border border-gray-200 dark:border-[#26282E] flex items-center justify-center">
                            <Image
                              src={app.icon}
                              alt={`${app.name} icon`}
                              className="w-5 h-5 object-contain"
                              width={20}
                              height={20}
                            />
                          </div>
                          <div className="text-sm font-medium text-gray-900 dark:text-white">
                            {app.name}
                          </div>
                        </div>
                        <div className="text-sm text-gray-600 dark:text-[#94A3B8]">
                          ${app.price}/user
                        </div>
                      </div>
                    ))
                )}
              </div>
              
              {Object.keys(selectedApps).length > 0 && (
                <div className="mt-4 pt-4 border-t border-gray-200 dark:border-[#26282E]">
                  <div className="flex items-center justify-between mb-2">
                    <span className="text-sm font-semibold text-gray-900 dark:text-white">
                      Current Total (Yearly)
                    </span>
                    <span className="text-lg font-bold text-gray-900 dark:text-white">
                      ${replacedYearlyTotal.toFixed(2)}
                    </span>
                  </div>
                  <div className="text-xs text-gray-500 dark:text-[#94A3B8]">
                    ${replacedMonthlyTotal.toFixed(2)}/month × 12 months
                  </div>
                </div>
              )}
            </div>

            {/* Clikkle Pricing */}
            <div className="bg-gradient-to-br from-gray-50 to-gray-100 dark:from-[#16171B] dark:to-[#0B0C0E] rounded-2xl p-3 border border-gray-200 dark:border-[#26282E]">
              <div className="flex items-start justify-between mb-4">
                <div>
                  <h4 className="text-base font-bold text-gray-900 dark:text-white mb-1">
                    Clikkle All-in-One
                  </h4>
                  <p className="text-xs text-gray-600 dark:text-[#94A3B8]">
                    All apps included, no hidden fees
                  </p>
                </div>
                <Sparkles className="w-5 h-5 text-[#253237] dark:text-white" />
              </div>
              <div className="text-2xl font-bold text-gray-900 dark:text-white mb-1">
                ${odooYearlyTotal.toFixed(2)}
                <span className="text-sm font-normal text-gray-600 dark:text-[#94A3B8]">/year</span>
              </div>
              <div className="text-xs text-gray-500 dark:text-[#94A3B8]">
                Only ${ODOO_MONTHLY_PER_USER}/user/month
              </div>
            </div>

            {/* Savings Card */}
            {savingsYearly > 0 && (
              <div className="bg-[#253237] dark:bg-white rounded-2xl p-3 shadow-xl relative overflow-hidden">
                {/* Decorative element */}
                <div className="absolute top-0 right-0 w-32 h-32 bg-white/10 dark:bg-[#253237]/10 rounded-full -mr-16 -mt-16" />
                
                <div className="relative">
                  <div className="flex items-center gap-2 mb-3">
                    <TrendingDown className="w-5 h-5 text-white dark:text-[#253237]" />
                    <span className="text-sm font-semibold text-white/90 dark:text-[#253237]/90">
                      Your Annual Savings
                    </span>
                  </div>
                  <div className="text-4xl font-bold text-white dark:text-[#253237] mb-2">
                    ${savingsYearly.toFixed(2)}
                  </div>
                  <p className="text-sm text-white/80 dark:text-[#253237]/80">
                    Save {((savingsYearly / replacedYearlyTotal) * 100).toFixed(0)}% with Clikkle&apos;s integrated platform
                  </p>
                </div>
              </div>
            )}
          </aside>
        </div>
      </div>
    </section>
  );
}
