'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import Image from 'next/image'

interface Message {
  id: number
  sender: string
  role: string
  text: string
  avatar: string
  avatarUrl: string
  color: string
}

const messagesData: Message[] = [
  {
    id: 1,
    sender: 'Sarah Chen',
    role: 'CEO',
    text: 'Status on the Q1 Hiring Plan?',
    avatar: 'SC',
    avatarUrl: 'https://images.unsplash.com/photo-1573496359142-b8d87734a5a2?w=100&h=100&fit=crop&crop=faces',
    color: 'bg-gray-300 dark:bg-gray-600'
  },
  {
    id: 2,
    sender: 'Michael Davis',
    role: 'HR Director',
    text: 'Finalizing the candidates now.',
    avatar: 'MD',
    avatarUrl: 'https://images.unsplash.com/photo-1556157382-97eda2d62296?w=100&h=100&fit=crop&crop=faces',
    color: 'bg-gray-200 dark:bg-gray-500'
  },
  {
    id: 3,
    sender: 'Emily Wilson',
    role: 'Operations',
    text: '👍 Budget approved!',
    avatar: 'EW',
    avatarUrl: 'https://images.unsplash.com/photo-1551836022-d5d88e9218df?w=100&h=100&fit=crop&crop=faces',
    color: 'bg-gray-400 dark:bg-gray-700'
  }
]

export default function ChatAnimationPanel() {
  const [messages, setMessages] = useState<Message[]>([])
  const [activeCount, setActiveCount] = useState(3)
  const [isVisible, setIsVisible] = useState(false)
  const messageCounterRef = useRef(0)
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer to detect when component is visible
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  // Animation logic - only starts when visible
  useEffect(() => {
    if (!isVisible) return

    let messageIndex = 0
    let timerId: NodeJS.Timeout

    const addMessage = () => {
      if (messageIndex < messagesData.length) {
        const messageWithUniqueId = {
          ...messagesData[messageIndex],
          id: messageCounterRef.current++
        }
        setMessages((prev) => [...prev, messageWithUniqueId])
        messageIndex++
        timerId = setTimeout(addMessage, 800)
      } else {
        // Hold for 1.5 seconds then reset
        timerId = setTimeout(() => {
          setMessages([])
          messageIndex = 0
          timerId = setTimeout(addMessage, 500)
        }, 1500)
      }
    }

    timerId = setTimeout(addMessage, 500)

    return () => clearTimeout(timerId)
  }, [isVisible])

  return (
    <div 
      ref={panelRef}
      className={`w-[310px] md:w-[450px] mx-auto mb-8 md:mb-0 transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Chat Window */}
      <div className="w-full bg-white dark:bg-[#1a1a1a] backdrop-blur-md rounded-xl md:rounded-2xl shadow-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
        {/* Header */}
        <div className="bg-gray-50 dark:bg-[#1a1a1a] px-3 md:px-5 py-2.5 md:py-4 border-b border-gray-200 dark:border-gray-700">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-2 md:gap-3">
              <h3 className="text-gray-700 dark:text-white font-semibold text-xs md:text-base">Management Sync</h3>
            </div>
            
            {/* Active indicator with pulse */}
            <div className="flex items-center gap-1.5 md:gap-2">
              <div className="relative">
                <div className="w-1.5 h-1.5 md:w-2 md:h-2 bg-green-500 rounded-full"></div>
                <div className="absolute inset-0 w-1.5 h-1.5 md:w-2 md:h-2 bg-green-500 rounded-full animate-ping"></div>
              </div>
              <span className="text-gray-600 dark:text-gray-300 text-[10px] md:text-sm">{activeCount} active</span>
            </div>
          </div>
        </div>

        {/* Messages Container */}
        <div className="p-3 md:p-5 space-y-2.5 md:space-y-4 h-[255px] md:h-[340px] overflow-hidden">
          <AnimatePresence mode="popLayout">
            {messages.map((message, index) => message ? (
              <motion.div
                key={message.id}
                initial={{ x: -100, opacity: 0, scale: 0.8 }}
                animate={{ x: 0, opacity: 1, scale: 1 }}
                exit={{ opacity: 0, scale: 0.8 }}
                transition={{
                  type: 'spring',
                  stiffness: 400,
                  damping: 25
                }}
                className="flex items-start gap-3"
              >
                {/* Avatar */}
                <div className={`w-7 h-7 md:w-10 md:h-10 rounded-full ${message.color} flex items-center justify-center flex-shrink-0 overflow-hidden shadow-md relative`}>
                  <Image
                    src={message.avatarUrl}
                    alt={message.sender}
                    fill
                    className="object-cover"
                    sizes="40px"
                  />
                </div>

                {/* Message Bubble */}
                <div className="flex-1">
                  <div className="flex items-baseline gap-1.5 md:gap-2 mb-0.5 md:mb-1">
                    <span className="text-gray-700 dark:text-white font-medium text-[10px] md:text-sm">{message.sender}</span>
                    <span className="text-gray-500 dark:text-gray-400 text-[8px] md:text-xs">{message.role}</span>
                  </div>
                  <div className="bg-gray-200 dark:bg-white/10 text-gray-800 dark:text-white px-2.5 md:px-4 py-1.5 md:py-2.5 rounded-xl md:rounded-2xl rounded-tl-sm shadow-sm">
                    <p className="text-[10px] md:text-sm leading-relaxed">
                      {message.text}
                    </p>
                  </div>
                </div>
              </motion.div>
            ) : null)}
          </AnimatePresence>
        </div>
      </div>
    </div>
  )
}
