'use client'

import { useState, useEffect, useRef } from 'react'
import { Mail, Send, Users, TrendingUp } from 'lucide-react'
import { motion, AnimatePresence, useAnimation } from 'framer-motion'

type AnimationStage = 'leadsEntering' | 'processing' | 'launching' | 'results' | 'complete'

function CampaignWorkflowAnimation() {
  const [stage, setStage] = useState<AnimationStage>('leadsEntering')
  const [processedLeads, setProcessedLeads] = useState<number[]>([])
  const [showEnvelopes, setShowEnvelopes] = useState(false)
  const [chartHeight, setChartHeight] = useState(0)
  const [isVisible, setIsVisible] = useState(false)
  const cursorControls = useAnimation()
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  useEffect(() => {
    if (!isVisible) return

    const runAnimation = async () => {
      // Reset
      setProcessedLeads([])
      setShowEnvelopes(false)
      setChartHeight(0)
      
      // Stage 1: Leads entering
      setStage('leadsEntering')
      await new Promise(resolve => setTimeout(resolve, 2000))
      
      // Stage 2: Processing
      setStage('processing')
      for (let i = 0; i < 3; i++) {
        await new Promise(resolve => setTimeout(resolve, 400))
        setProcessedLeads(prev => [...prev, i])
      }
      await new Promise(resolve => setTimeout(resolve, 500))
      
      // Stage 3: Cursor moves and clicks
      setStage('launching')
      
      // Adjust cursor position based on screen size
      const isMobile = window.innerWidth < 768
      await cursorControls.start({
        x: isMobile ? -50 : 0,
        y: isMobile ? 15 : 40,
        opacity: 1,
        transition: { type: 'spring', stiffness: 100, damping: 15 }
      })
      await new Promise(resolve => setTimeout(resolve, 300))
      
      // Click animation
      await cursorControls.start({
        scale: 0.9,
        transition: { duration: 0.1 }
      })
      await cursorControls.start({
        scale: 1,
        transition: { duration: 0.1 }
      })
      
      // Stage 4: Send envelopes
      setShowEnvelopes(true)
      await new Promise(resolve => setTimeout(resolve, 1000))
      
      // Stage 5: Show results
      setStage('results')
      setChartHeight(100)
      await cursorControls.start({
        opacity: 0,
        transition: { duration: 0.3 }
      })
      
      // Stage 6: Complete - hold for 2 seconds
      setStage('complete')
      await new Promise(resolve => setTimeout(resolve, 2000))
      
      // Reset and loop
      await cursorControls.set({ x: -100, y: 0, opacity: 0, scale: 1 })
      runAnimation()
    }

    runAnimation()
  }, [cursorControls, isVisible])

  const leadsVariants = {
    hidden: { x: -200, opacity: 0 },
    visible: (i: number) => ({
      x: 0,
      opacity: 1,
      transition: {
        delay: i * 0.3,
        type: 'spring' as const,
        stiffness: 100,
        damping: 15
      }
    })
  }

  return (
    <div 
      ref={panelRef}
      className={`relative w-full max-w-2xl h-[350px] md:h-[450px] mx-auto transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Main Container */}
      <div className="relative h-full flex items-center justify-between gap-3 md:gap-8">
        
        {/* Left: Leads */}
        <div className="flex flex-col gap-2 md:gap-4">
          {[0, 1, 2].map((index) => (
            <motion.div
              key={index}
              custom={index}
              variants={leadsVariants}
              initial="hidden"
              animate={stage !== 'leadsEntering' || index === 0 || index === 1 || index === 2 ? "visible" : "hidden"}
              className={`w-8 h-8 md:w-12 md:h-12 rounded-full flex items-center justify-center border-2 transition-colors duration-500 ${
                processedLeads.includes(index)
                  ? 'bg-[#253237] dark:bg-white border-[#253237] dark:border-white'
                  : 'bg-gray-200 dark:bg-gray-900 border-gray-300 dark:border-gray-800'
              }`}
            >
              <Users className={`w-4 h-4 md:w-6 md:h-6 ${
                processedLeads.includes(index) ? 'text-white dark:text-[#253237]' : 'text-gray-600 dark:text-gray-400'
              }`} />
            </motion.div>
          ))}
        </div>

        {/* Center: Campaign Engine */}
        <div className="flex-1 flex flex-col items-center gap-2 md:gap-4">
          <div className="relative w-full max-w-[160px] md:max-w-xs h-36 md:h-48 bg-white/80 dark:bg-[#1a1a1a]/80 border border-gray-300 dark:border-gray-800 rounded-xl md:rounded-2xl backdrop-blur-sm p-3 md:p-6 flex flex-col items-center justify-center shadow-lg">
            <div className="mb-2 md:mb-4">
              <div className="flex items-center justify-center gap-1 md:gap-2 mb-1 md:mb-2">
                {/* Processing indicator */}
                {stage === 'processing' && (
                  <motion.div
                    className="w-3 h-3 md:w-4 md:h-4"
                    animate={{ rotate: 360 }}
                    transition={{ duration: 1, repeat: Infinity, ease: 'linear' }}
                  >
                    <div className="w-full h-full border-2 border-gray-300 dark:border-gray-800 border-t-[#253237] dark:border-t-white rounded-full" />
                  </motion.div>
                )}
                <h4 className="text-[10px] md:text-sm font-semibold text-[#253237] dark:text-white">Campaign Engine</h4>
              </div>
            </div>

            {/* Launch Button */}
            <motion.button
              className="relative px-3 py-1.5 md:px-6 md:py-2.5 bg-[#253237] hover:bg-[#253237]/90 dark:bg-white dark:hover:bg-gray-200 text-white dark:text-[#253237] text-[10px] md:text-sm font-medium rounded-md md:rounded-lg shadow-lg transition-colors"
              whileHover={{ scale: 1.05 }}
              whileTap={{ scale: 0.95 }}
            >
              <Send className="inline-block w-3 h-3 md:w-4 md:h-4 mr-1 md:mr-2" />
              <span className="hidden md:inline">Launch Campaign</span>
              <span className="md:hidden">Launch</span>
            </motion.button>
          </div>
        </div>

        {/* Flying Envelopes */}
        <AnimatePresence>
          {showEnvelopes && [0, 1, 2].map((index) => (
            <motion.div
              key={`envelope-${index}`}
              className="absolute left-1/2 top-1/2"
              initial={{ x: 0, y: 0, opacity: 0, scale: 0 }}
              animate={{
                x: [0, 120, 120],
                y: -20 + index * 30,
                opacity: [0, 1, 0],
                scale: 1,
              }}
              transition={{
                delay: index * 0.2,
                duration: 1.2,
                ease: 'easeOut'
              }}
            >
              <Mail className="w-6 h-6 md:w-8 md:h-8 text-[#253237] dark:text-white" />
            </motion.div>
          ))}
        </AnimatePresence>

        {/* Right: Results Panel */}
        <div className="w-32 h-44 md:w-48 md:h-64 bg-white/80 dark:bg-[#1a1a1a]/80 border border-gray-300 dark:border-gray-800 rounded-xl md:rounded-2xl backdrop-blur-sm p-2 md:p-4 shadow-lg">
          <div className="flex items-center gap-1 md:gap-2 mb-2 md:mb-4">
            <TrendingUp className="w-4 h-4 md:w-5 md:h-5 text-[#253237] dark:text-white" />
            <h4 className="text-[10px] md:text-xs font-semibold text-[#253237] dark:text-white">Results</h4>
          </div>
          
          {/* Bar Chart */}
          <div className="h-28 md:h-40 flex items-end justify-around gap-1 md:gap-2">
            {[70, 85, 100].map((targetHeight, index) => (
              <motion.div
                key={index}
                className="flex-1 bg-gradient-to-t from-[#253237] to-gray-600 dark:from-white dark:to-gray-300 rounded-t-lg"
                initial={{ height: 0 }}
                animate={{
                  height: stage === 'results' || stage === 'complete' ? `${(targetHeight / 100) * chartHeight}%` : 0
                }}
                transition={{
                  delay: index * 0.15,
                  type: 'spring' as const,
                  stiffness: 200,
                  damping: 15
                }}
              />
            ))}
          </div>
        </div>
      </div>

      {/* Cursor */}
      <motion.div
        className="absolute pointer-events-none z-50"
        style={{ left: '50%', top: '50%' }}
        animate={cursorControls}
        initial={{ x: -100, y: 0, opacity: 0 }}
      >
        <svg
          width="24"
          height="24"
          viewBox="0 0 24 24"
          fill="none"
          className="drop-shadow-lg"
        >
          <path
            d="M3 3L10.07 19.97L12.58 12.58L19.97 10.07L3 3Z"
            fill="white"
            stroke="black"
            strokeWidth="1"
          />
        </svg>
      </motion.div>
    </div>
  )
}

export default function CampaignsAnimationPanel() {
  return <CampaignWorkflowAnimation />
}
