'use client'

import { useState, useEffect, useRef } from 'react'
import { Users, Mail, Phone, Calendar, MessageSquare } from 'lucide-react'
import { motion, AnimatePresence } from 'framer-motion'

type AnimationStage = 'idle' | 'merging' | 'complete' | 'exploding'

const iconData = [
  { Icon: Mail, position: { x: -120, y: -120 } },
  { Icon: Phone, position: { x: 120, y: -120 } },
  { Icon: Calendar, position: { x: -120, y: 120 } },
  { Icon: MessageSquare, position: { x: 120, y: 120 } }
]

function CustomerInsightsAnimation() {
  const [stage, setStage] = useState<AnimationStage>('idle')
  const [mergedIcons, setMergedIcons] = useState<number[]>([])
  const [leadScore, setLeadScore] = useState(0)
  const [showPulse, setShowPulse] = useState(false)
  const [isVisible, setIsVisible] = useState(false)
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  useEffect(() => {
    if (!isVisible) return

    const runAnimation = async () => {
      // Reset
      setStage('idle')
      setMergedIcons([])
      setLeadScore(0)
      await new Promise(resolve => setTimeout(resolve, 1000))

      // Merge icons one by one
      setStage('merging')
      for (let i = 0; i < iconData.length; i++) {
        await new Promise(resolve => setTimeout(resolve, 800))
        setMergedIcons(prev => [...prev, i])
        setLeadScore((i + 1) * 25)
        setShowPulse(true)
        await new Promise(resolve => setTimeout(resolve, 100))
        setShowPulse(false)
      }

      // Complete state
      await new Promise(resolve => setTimeout(resolve, 500))
      setStage('complete')
      await new Promise(resolve => setTimeout(resolve, 2500))

      // Explode back
      setStage('exploding')
      setLeadScore(0) // Clear score during explosion
      await new Promise(resolve => setTimeout(resolve, 1000))

      // Restart
      runAnimation()
    }

    runAnimation()
  }, [isVisible])

  return (
    <div 
      ref={panelRef}
      className={`relative w-full max-w-lg h-[350px] md:h-[450px] mx-auto flex items-center justify-center overflow-hidden transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Lead Score Text - Top Middle */}
      <AnimatePresence mode="wait">
        {stage === 'complete' && (
          <motion.div
            className="absolute top-4 md:top-8 left-1/2 -translate-x-1/2 z-30"
            initial={{ opacity: 0, y: -20 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: -20 }}
            transition={{ duration: 0.3 }}
          >
            <div className="text-center">
              <p className="text-[10px] md:text-xs text-gray-600 dark:text-gray-400 font-medium">Lead Score</p>
              <p className="text-2xl md:text-3xl font-bold text-[#253237] dark:text-white">{leadScore}%</p>
            </div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Central Avatar */}
      <motion.div
        className="relative w-24 h-24 md:w-32 md:h-32 rounded-full bg-[#253237] dark:bg-white flex items-center justify-center z-20"
        style={{
          filter: 'drop-shadow(0 0 20px rgba(0, 0, 0, 0.3)) drop-shadow(0 0 10px rgba(255, 255, 255, 0.1))'
        }}
        animate={
          mergedIcons.length > 0
            ? { scale: [1, 1.1, 1] }
            : { scale: 1 }
        }
        transition={{ duration: 0.3 }}
      >
        <Users className="w-12 h-12 md:w-16 md:h-16 text-white dark:text-[#253237]" />

        {/* Lead Score Ring */}
        <svg className="absolute inset-0 w-full h-full -rotate-90" viewBox="0 0 100 100">
          <circle
            cx="50"
            cy="50"
            r="48"
            fill="none"
            stroke="rgba(255,255,255,0.2)"
            strokeWidth="2"
          />
          <motion.circle
            cx="50"
            cy="50"
            r="48"
            fill="none"
            stroke="white"
            className="dark:stroke-[#253237]"
            strokeWidth="3"
            strokeLinecap="round"
            strokeDasharray={`${2 * Math.PI * 48}`}
            initial={{ strokeDashoffset: 2 * Math.PI * 48 }}
            animate={{ 
              strokeDashoffset: 2 * Math.PI * 48 * (1 - leadScore / 100)
            }}
            transition={{ duration: 0.5, ease: 'easeOut' }}
          />
        </svg>
      </motion.div>

      {/* Pulse Effect */}
      <AnimatePresence>
        {showPulse && (
          <motion.div
            className="absolute w-24 h-24 md:w-32 md:h-32 rounded-full border-4 border-[#253237]/50 dark:border-white/50"
            initial={{ scale: 1, opacity: 1 }}
            animate={{ scale: 2.5, opacity: 0 }}
            exit={{ opacity: 0 }}
            transition={{ duration: 0.6, ease: 'easeOut' }}
          />
        )}
      </AnimatePresence>

      {/* Orbiting Icons */}
      {iconData.map((item, index) => {
        const isMerged = mergedIcons.includes(index)
        const isExploding = stage === 'exploding'

        // Adjust positions for mobile
        const mobilePosition = {
          x: item.position.x * 0.6,
          y: item.position.y * 0.6
        }

        return (
          <motion.div
            key={index}
            className="absolute w-10 h-10 md:w-12 md:h-12 rounded-xl bg-white/80 dark:bg-[#1a1a1a]/80 backdrop-blur-sm border border-gray-300 dark:border-gray-800 flex items-center justify-center"
            initial={window.innerWidth < 768 ? mobilePosition : item.position}
            animate={
              isExploding
                ? (window.innerWidth < 768 ? mobilePosition : item.position)
                : isMerged
                ? { x: 0, y: 0, scale: 0, opacity: 0 }
                : (window.innerWidth < 768 ? mobilePosition : item.position)
            }
            transition={{
              duration: isExploding ? 0.6 : 1.2,
              ease: isExploding ? 'easeOut' : 'circIn',
              delay: isExploding ? 0 : 0
            }}
            style={{
              filter: 'drop-shadow(0 0 10px rgba(0, 0, 0, 0.2))'
            }}
          >
            <item.Icon className="w-5 h-5 md:w-6 md:h-6 text-[#253237] dark:text-white" />
          </motion.div>
        )
      })}
    </div>
  )
}

export default function CRMAnimationPanel() {
  return <CustomerInsightsAnimation />
}
