'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { Bug, AlertTriangle, CheckCircle2, Code, Zap } from 'lucide-react'

interface BugIssue {
  id: number
  type: 'critical' | 'warning' | 'info'
  label: string
  stage: 'pending' | 'analyzing' | 'fixed'
}

const bugsData: BugIssue[] = [
  { id: 1, type: 'critical', label: 'Memory Leak', stage: 'pending' },
  { id: 2, type: 'warning', label: 'UI Glitch', stage: 'pending' },
  { id: 3, type: 'info', label: 'Slow Query', stage: 'pending' }
]

export default function BugTrackingAnimationPanel() {
  const [bugs, setBugs] = useState<BugIssue[]>([])
  const [isComplete, setIsComplete] = useState(false)
  const [isVisible, setIsVisible] = useState(false)
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  useEffect(() => {
    if (!isVisible) return

    const runPipeline = async () => {
      // Reset
      setBugs([])
      setIsComplete(false)
      await new Promise(resolve => setTimeout(resolve, 500))

      // Add bugs one by one to pending
      for (let i = 0; i < bugsData.length; i++) {
        await new Promise(resolve => setTimeout(resolve, 600))
        setBugs(prev => [...prev, bugsData[i]])
      }

      await new Promise(resolve => setTimeout(resolve, 800))

      // Process each bug through the pipeline
      for (let i = 0; i < bugsData.length; i++) {
        // Move to analyzing
        await new Promise(resolve => setTimeout(resolve, 700))
        setBugs(prev => prev.map((bug, idx) => 
          idx === i ? { ...bug, stage: 'analyzing' } : bug
        ))

        // Analyze time
        await new Promise(resolve => setTimeout(resolve, 1200))

        // Move to fixed
        setBugs(prev => prev.map((bug, idx) => 
          idx === i ? { ...bug, stage: 'fixed' } : bug
        ))

        await new Promise(resolve => setTimeout(resolve, 500))
      }

      // Complete
      await new Promise(resolve => setTimeout(resolve, 800))
      setIsComplete(true)
      await new Promise(resolve => setTimeout(resolve, 2500))

      // Restart
      runPipeline()
    }

    runPipeline()
  }, [isVisible])

  const getBugIcon = (type: string, size: 'sm' | 'md' = 'md') => {
    const sizeClass = size === 'sm' ? 'w-4 h-4 md:w-5 md:h-5' : 'w-5 h-5 md:w-6 md:h-6'
    
    switch (type) {
      case 'critical':
        return <AlertTriangle className={sizeClass} />
      case 'warning':
        return <Bug className={sizeClass} />
      case 'info':
        return <Code className={sizeClass} />
      default:
        return <Bug className={sizeClass} />
    }
  }

  const getPendingBugs = () => bugs.filter(b => b.stage === 'pending')
  const getAnalyzingBugs = () => bugs.filter(b => b.stage === 'analyzing')
  const getFixedBugs = () => bugs.filter(b => b.stage === 'fixed')

  return (
    <div 
      ref={panelRef}
      className={`relative w-full max-w-2xl h-[400px] md:h-[550px] mx-auto flex items-center justify-center transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* 3-Column Pipeline */}
      <div className="flex gap-4 md:gap-6 items-start">
        {/* Column 1: Pending */}
        <div className="flex flex-col items-center gap-3 w-28 md:w-36">
          <div className="text-center mb-2">
            <div className="w-10 h-10 md:w-12 md:h-12 mx-auto mb-2 rounded-full bg-gray-200 dark:bg-gray-800 flex items-center justify-center">
              <Bug className="w-5 h-5 md:w-6 md:h-6 text-gray-600 dark:text-gray-400" strokeWidth={2} />
            </div>
            <p className="text-xs md:text-sm font-semibold text-gray-700 dark:text-gray-300">Pending</p>
            <p className="text-[10px] md:text-xs text-gray-500 dark:text-gray-400">{getPendingBugs().length}</p>
          </div>

          <div className="space-y-2 md:space-y-3 min-h-[200px] md:min-h-[280px]">
            <AnimatePresence mode="popLayout">
              {getPendingBugs().map((bug) => (
                <motion.div
                  key={`pending-${bug.id}`}
                  layout
                  initial={{ opacity: 0, x: -50, scale: 0.8 }}
                  animate={{ opacity: 1, x: 0, scale: 1 }}
                  exit={{ opacity: 0, x: 50, scale: 0.8 }}
                  transition={{ type: 'spring', stiffness: 300, damping: 25 }}
                  className="bg-white/90 dark:bg-[#1a1a1a]/80 border border-gray-300 dark:border-gray-700 rounded-lg p-2 md:p-3 shadow-md"
                >
                  <div className="text-gray-700 dark:text-gray-300 mb-1">
                    {getBugIcon(bug.type, 'sm')}
                  </div>
                  <p className="text-[10px] md:text-xs font-medium text-gray-800 dark:text-white truncate">{bug.label}</p>
                </motion.div>
              ))}
            </AnimatePresence>
          </div>
        </div>

        {/* Column 2: Analyzing */}
        <div className="flex flex-col items-center gap-3 w-28 md:w-36">
          <div className="text-center mb-2">
            <motion.div 
              className="w-10 h-10 md:w-12 md:h-12 mx-auto mb-2 rounded-full bg-gray-200 dark:bg-gray-800 flex items-center justify-center"
              animate={getAnalyzingBugs().length > 0 ? {
                boxShadow: [
                  '0 0 0 rgba(0,0,0,0.2)',
                  '0 0 20px rgba(156,163,175,0.6)',
                  '0 0 0 rgba(0,0,0,0.2)'
                ]
              } : {}}
              transition={{ duration: 1, repeat: Infinity }}
            >
              <Zap className="w-5 h-5 md:w-6 md:h-6 text-gray-600 dark:text-gray-400" strokeWidth={2} />
            </motion.div>
            <p className="text-xs md:text-sm font-semibold text-gray-700 dark:text-gray-300">Analyzing</p>
            <p className="text-[10px] md:text-xs text-gray-500 dark:text-gray-400">{getAnalyzingBugs().length}</p>
          </div>

          <div className="space-y-2 md:space-y-3 min-h-[200px] md:min-h-[280px]">
            <AnimatePresence mode="popLayout">
              {getAnalyzingBugs().map((bug) => (
                <motion.div
                  key={`analyzing-${bug.id}`}
                  layout
                  initial={{ opacity: 0, x: -50, scale: 0.8 }}
                  animate={{ 
                    opacity: 1, 
                    x: 0, 
                    scale: 1,
                  }}
                  exit={{ opacity: 0, x: 50, scale: 0.8 }}
                  transition={{ type: 'spring', stiffness: 300, damping: 25 }}
                  className="bg-white/90 dark:bg-[#1a1a1a]/80 border-2 border-gray-400 dark:border-gray-500 rounded-lg p-2 md:p-3 shadow-lg"
                >
                  <motion.div 
                    className="text-gray-700 dark:text-gray-300 mb-1"
                    animate={{ rotate: [0, 360] }}
                    transition={{ duration: 2, repeat: Infinity, ease: 'linear' }}
                  >
                    {getBugIcon(bug.type, 'sm')}
                  </motion.div>
                  <p className="text-[10px] md:text-xs font-medium text-gray-800 dark:text-white truncate">{bug.label}</p>
                  <p className="text-[9px] md:text-[10px] text-gray-500 dark:text-gray-400">Processing...</p>
                </motion.div>
              ))}
            </AnimatePresence>
          </div>
        </div>

        {/* Column 3: Fixed */}
        <div className="flex flex-col items-center gap-3 w-28 md:w-36">
          <div className="text-center mb-2">
            <div className="w-10 h-10 md:w-12 md:h-12 mx-auto mb-2 rounded-full bg-gray-800 dark:bg-gray-200 flex items-center justify-center">
              <CheckCircle2 className="w-5 h-5 md:w-6 md:h-6 text-white dark:text-gray-900" strokeWidth={2} />
            </div>
            <p className="text-xs md:text-sm font-semibold text-gray-700 dark:text-gray-300">Fixed</p>
            <p className="text-[10px] md:text-xs text-gray-500 dark:text-gray-400">{getFixedBugs().length}</p>
          </div>

          <div className="space-y-2 md:space-y-3 min-h-[200px] md:min-h-[280px]">
            <AnimatePresence mode="popLayout">
              {getFixedBugs().map((bug) => (
                <motion.div
                  key={`fixed-${bug.id}`}
                  layout
                  initial={{ opacity: 0, x: -50, scale: 0.8 }}
                  animate={{ opacity: 1, x: 0, scale: 1 }}
                  exit={{ opacity: 0, scale: 0 }}
                  transition={{ type: 'spring', stiffness: 300, damping: 25 }}
                  className="bg-gray-100 dark:bg-gray-800/50 border border-gray-300 dark:border-gray-600 rounded-lg p-2 md:p-3 shadow-md opacity-60"
                >
                  <div className="text-gray-600 dark:text-gray-400 mb-1">
                    {getBugIcon(bug.type, 'sm')}
                  </div>
                  <p className="text-[10px] md:text-xs font-medium text-gray-700 dark:text-gray-300 truncate line-through">{bug.label}</p>
                </motion.div>
              ))}
            </AnimatePresence>
          </div>
        </div>
      </div>

      {/* Completion Overlay */}
      <AnimatePresence>
        {isComplete && (
          <motion.div
            className="absolute inset-0 flex items-center justify-center bg-transparent dark:bg-transparent backdrop-blur-sm rounded-2xl"
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
          >
            <motion.div
              initial={{ scale: 0, rotate: -180 }}
              animate={{ scale: 1, rotate: 0 }}
              transition={{ type: 'spring', stiffness: 200, damping: 15 }}
              className="text-center"
            >
              <div className="w-20 h-20 md:w-24 md:h-24 mx-auto mb-4 bg-gray-800 dark:bg-gray-200 rounded-full flex items-center justify-center">
                <CheckCircle2 className="w-12 h-12 md:w-14 md:h-14 text-white dark:text-gray-900" strokeWidth={3} />
              </div>
              <p className="text-base md:text-lg font-bold text-gray-800 dark:text-white">All Bugs Fixed!</p>
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  )
}
