'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { Sparkles, Lightbulb, TrendingUp, Zap, Brain, CheckCircle2 } from 'lucide-react'

type Stage = 'idle' | 'thinking' | 'generating' | 'analyzing' | 'optimizing' | 'complete'

interface Idea {
  id: number
  text: string
  score: number
}

const ideasData: Idea[] = [
  { id: 1, text: '10 Tips for Productivity', score: 85 },
  { id: 2, text: 'Behind the Scenes Vlog', score: 92 },
  { id: 3, text: 'Industry Trends 2024', score: 78 }
]

export default function AIContentIntelligencePanel() {
  const [stage, setStage] = useState<Stage>('idle')
  const [ideas, setIdeas] = useState<Idea[]>([])
  const [selectedIdea, setSelectedIdea] = useState<number | null>(null)
  const [trendScore, setTrendScore] = useState(0)
  const [isVisible, setIsVisible] = useState(false)
  const panelRef = useRef<HTMLDivElement>(null)

  // Intersection observer
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.2 }
    )

    if (panelRef.current) {
      observer.observe(panelRef.current)
    }

    return () => {
      if (panelRef.current) {
        observer.unobserve(panelRef.current)
      }
    }
  }, [])

  useEffect(() => {
    if (!isVisible) return

    const runAI = async () => {
      // Reset
      setStage('idle')
      setIdeas([])
      setSelectedIdea(null)
      setTrendScore(0)
      await new Promise(resolve => setTimeout(resolve, 800))

      // AI Thinking
      setStage('thinking')
      await new Promise(resolve => setTimeout(resolve, 1500))

      // Generating Ideas
      setStage('generating')
      for (let i = 0; i < ideasData.length; i++) {
        await new Promise(resolve => setTimeout(resolve, 600))
        setIdeas(prev => [...prev, ideasData[i]])
      }

      await new Promise(resolve => setTimeout(resolve, 800))

      // Analyzing best idea
      setStage('analyzing')
      setSelectedIdea(1) // Select the highest scoring idea
      
      // Animate trend score
      for (let i = 0; i <= 92; i += 4) {
        await new Promise(resolve => setTimeout(resolve, 20))
        setTrendScore(Math.min(i, 92))
      }

      await new Promise(resolve => setTimeout(resolve, 1000))

      // Optimizing
      setStage('optimizing')
      await new Promise(resolve => setTimeout(resolve, 1500))

      // Complete
      setStage('complete')
      await new Promise(resolve => setTimeout(resolve, 2500))

      // Restart
      runAI()
    }

    runAI()
  }, [isVisible])

  return (
    <div 
      ref={panelRef}
      className={`relative w-full max-w-2xl h-[400px] md:h-[550px] mx-auto flex items-center justify-center overflow-visible transition-all duration-700 ${
        isVisible ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10'
      }`}
    >
      {/* Central AI Brain */}
      <motion.div
        className={`relative w-32 h-32 md:w-40 md:h-40 rounded-full flex items-center justify-center transition-all duration-500 ${
          stage === 'thinking' || stage === 'generating'
            ? 'bg-gray-800 dark:bg-gray-200'
            : stage === 'analyzing' || stage === 'optimizing'
            ? 'bg-gray-700 dark:bg-gray-300'
            : stage === 'complete'
            ? 'bg-gray-900 dark:bg-gray-100'
            : 'bg-gray-200 dark:bg-gray-800'
        }`}
        animate={
          stage === 'thinking' || stage === 'generating'
            ? {
                scale: [1, 1.1, 1],
                rotate: [0, 5, -5, 0]
              }
            : stage === 'analyzing'
            ? { rotate: 360 }
            : {}
        }
        transition={
          stage === 'thinking' || stage === 'generating'
            ? { duration: 2, repeat: Infinity }
            : stage === 'analyzing'
            ? { duration: 2, ease: 'linear' }
            : {}
        }
        style={{
          filter: 'drop-shadow(0 10px 30px rgba(0, 0, 0, 0.2))'
        }}
      >
        <Brain className={`w-16 h-16 md:w-20 md:h-20 transition-colors ${
          stage !== 'idle' ? 'text-white dark:text-gray-900' : 'text-gray-600 dark:text-gray-400'
        }`} />

        {/* Thinking particles */}
        {stage === 'thinking' && (
          <>
            {[...Array(6)].map((_, i) => (
              <motion.div
                key={i}
                className="absolute"
                animate={{
                  opacity: [0, 1, 0],
                  scale: [0, 1.5, 0],
                  x: Math.cos((i * 60) * Math.PI / 180) * 60,
                  y: Math.sin((i * 60) * Math.PI / 180) * 60
                }}
                transition={{
                  duration: 1.5,
                  repeat: Infinity,
                  delay: i * 0.2
                }}
              >
                <Sparkles className="w-4 h-4 md:w-5 md:h-5 text-gray-700 dark:text-gray-300" />
              </motion.div>
            ))}
          </>
        )}
      </motion.div>

      {/* Generated Ideas - Left Side */}
      <AnimatePresence>
        {stage === 'generating' || stage === 'analyzing' || stage === 'optimizing' || stage === 'complete' ? (
          <div className="absolute left-0 md:left-8 top-1/2 -translate-y-1/2 space-y-2 md:space-y-3 w-48 md:w-56">
            {ideas.map((idea, index) => (
              <motion.div
                key={idea.id}
                initial={{ opacity: 0, x: -100, scale: 0.8 }}
                animate={{ 
                  opacity: 1, 
                  x: 0, 
                  scale: selectedIdea === index ? 1.05 : 1
                }}
                transition={{ type: 'spring', stiffness: 200, damping: 20 }}
                className={`bg-white dark:bg-[#1a1a1a] border-2 rounded-lg p-2 md:p-3 shadow-lg transition-all ${
                  selectedIdea === index
                    ? 'border-gray-800 dark:border-gray-200'
                    : 'border-gray-300 dark:border-gray-700'
                }`}
              >
                <div className="flex items-start gap-2">
                  <Lightbulb className={`w-4 h-4 md:w-5 md:h-5 flex-shrink-0 mt-0.5 ${
                    selectedIdea === index
                      ? 'text-gray-800 dark:text-gray-200'
                      : 'text-gray-600 dark:text-gray-400'
                  }`} />
                  <div className="flex-1 min-w-0">
                    <p className="text-[10px] md:text-xs font-semibold text-gray-800 dark:text-white truncate">
                      {idea.text}
                    </p>
                    {selectedIdea === index && (
                      <motion.div
                        initial={{ opacity: 0, height: 0 }}
                        animate={{ opacity: 1, height: 'auto' }}
                        className="mt-1"
                      >
                        <p className="text-[9px] md:text-[10px] text-gray-600 dark:text-gray-400">
                          Score: {idea.score}%
                        </p>
                      </motion.div>
                    )}
                  </div>
                  {selectedIdea === index && (
                    <motion.div
                      initial={{ scale: 0 }}
                      animate={{ scale: 1 }}
                      transition={{ delay: 0.2 }}
                    >
                      <CheckCircle2 className="w-4 h-4 md:w-5 md:h-5 text-green-600 dark:text-green-400 flex-shrink-0" />
                    </motion.div>
                  )}
                </div>
              </motion.div>
            ))}
          </div>
        ) : null}
      </AnimatePresence>

      {/* Trend Analysis - Right Side */}
      <AnimatePresence>
        {stage === 'analyzing' || stage === 'optimizing' || stage === 'complete' ? (
          <motion.div
            className="absolute right-0 md:right-8 top-1/2 -translate-y-1/2"
            initial={{ opacity: 0, x: 100, scale: 0.8 }}
            animate={{ opacity: 1, x: 0, scale: 1 }}
            exit={{ opacity: 0, x: 100, scale: 0.8 }}
            transition={{ type: 'spring', stiffness: 200, damping: 20 }}
          >
            <div className="bg-white dark:bg-[#1a1a1a] border-2 border-gray-300 dark:border-gray-700 rounded-xl p-3 md:p-4 shadow-xl w-40 md:w-48">
              <div className="flex items-center gap-2 mb-3">
                <TrendingUp className="w-4 h-4 md:w-5 md:h-5 text-gray-700 dark:text-gray-300" />
                <span className="text-xs md:text-sm font-bold text-gray-800 dark:text-white">
                  Trend Score
                </span>
              </div>
              
              {/* Circular Progress */}
              <div className="relative w-20 h-20 md:w-24 md:h-24 mx-auto">
                <svg className="w-full h-full transform -rotate-90">
                  <circle
                    cx="50%"
                    cy="50%"
                    r="35%"
                    stroke="currentColor"
                    strokeWidth="6"
                    fill="none"
                    className="text-gray-200 dark:text-gray-700"
                  />
                  <motion.circle
                    cx="50%"
                    cy="50%"
                    r="35%"
                    stroke="currentColor"
                    strokeWidth="6"
                    fill="none"
                    strokeLinecap="round"
                    className="text-gray-800 dark:text-gray-200"
                    initial={{ pathLength: 0 }}
                    animate={{ pathLength: trendScore / 100 }}
                    transition={{ duration: 0.5 }}
                    style={{
                      strokeDasharray: '1 1'
                    }}
                  />
                </svg>
                <div className="absolute inset-0 flex items-center justify-center">
                  <span className="text-xl md:text-2xl font-bold text-gray-800 dark:text-white">
                    {trendScore}%
                  </span>
                </div>
              </div>
              
              <p className="text-center text-[10px] md:text-xs text-gray-600 dark:text-gray-400 mt-2">
                Trending Topic
              </p>
            </div>
          </motion.div>
        ) : null}
      </AnimatePresence>

      {/* Optimizing Badge - Bottom */}
      <AnimatePresence>
        {stage === 'optimizing' && (
          <motion.div
            className="absolute bottom-0 md:bottom-8 left-1/2 -translate-x-1/2"
            initial={{ opacity: 0, y: 50 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: 50 }}
            transition={{ type: 'spring', stiffness: 300 }}
          >
            <div className="bg-white dark:bg-[#1a1a1a] border-2 border-gray-300 dark:border-gray-700 rounded-xl px-4 md:px-5 py-2 md:py-3 shadow-xl">
              <div className="flex items-center gap-2 md:gap-3">
                <motion.div
                  animate={{ rotate: 360 }}
                  transition={{ duration: 2, repeat: Infinity, ease: 'linear' }}
                >
                  <Zap className="w-4 h-4 md:w-5 md:h-5 text-gray-700 dark:text-gray-300" />
                </motion.div>
                <span className="text-xs md:text-sm font-semibold text-gray-800 dark:text-white">
                  Auto-Optimizing...
                </span>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Complete Badge - Top */}
      <AnimatePresence>
        {stage === 'complete' && (
          <motion.div
            className="absolute top-0 md:top-8 left-1/2 -translate-x-1/2 z-30"
            initial={{ opacity: 0, y: -30, scale: 0.8 }}
            animate={{ opacity: 1, y: 0, scale: 1 }}
            exit={{ opacity: 0, y: -30, scale: 0.8 }}
            transition={{ type: 'spring', stiffness: 300, damping: 20 }}
          >
            <div className="bg-gray-800 dark:bg-gray-200 px-4 md:px-6 py-2 md:py-3 rounded-xl shadow-xl">
              <div className="flex items-center gap-2 md:gap-3">
                <CheckCircle2 className="w-4 h-4 md:w-5 md:h-5 text-white dark:text-gray-900" />
                <span className="text-xs md:text-sm font-bold text-white dark:text-gray-900">
                  Content Ready!
                </span>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  )
}
