"use client";
import React, { useState } from "react";
import Link from "next/link";
import Image from "next/image";
import { Video, KeyboardIcon, Users, Shield, Monitor, Globe } from "lucide-react";
import { useTheme } from "../atoms/theme";
import MeetHeader from "../components/MeetHeader";
import Footer from "../components/Footer";
import PreFooterCTA from "../components/PreFooterCTA";
import SecuritySection from "@/components/SecuritySection";
import SidebarDrawer from "../components/sidebarDrawer";
import HeroSection from "../components/HeroSection";
import AppsSection from "../components/appsSection";
import { apps } from "../constants/products";

import user1 from "../../public/images/user1.jpeg";
import user2 from "../../public/images/user2.jpeg";
import user3 from "../../public/images/user3.jpeg";
import user4 from "../../public/images/user4.jpeg";

const CrewHome: React.FC = () => {
  const { theme } = useTheme();
  const [leftDrawerOpen, setLeftDrawerOpen] = useState(false);
  const toggleLeftDrawer = () => setLeftDrawerOpen((prev) => !prev);
  const [meetingCode, setMeetingCode] = useState("");

  const features = [
    {
      icon: Monitor,
      title: "Meet on Any Device",
      description: "Join from your computer, phone, or tablet. No downloads required - just click and connect from any modern web browser."
    },
    {
      icon: Video,
      title: "Crystal Clear Quality",
      description: "AI-powered video and audio optimization ensures high-quality calls that adapt to your network speed for the best experience."
    },
    {
      icon: Shield,
      title: "Enterprise-Grade Security",
      description: "Your meetings are encrypted in transit with the same protections Wurknest uses to secure your data and safeguard your privacy."
    },
    {
      icon: Users,
      title: "Collaborate Seamlessly",
      description: "Present proposals, collaborate on projects, or catch up face-to-face. Screen sharing and real-time collaboration built in."
    },
    {
      icon: Globe,
      title: "Accessible to Everyone",
      description: "Live captions powered by AI make meetings accessible for non-native speakers and hearing-impaired participants."
    }
  ];

  const handleJoinMeeting = () => {
    if (meetingCode.trim()) {
      window.location.href = `https://crew.clikkle.com/join/${meetingCode}`;
    }
  };

  return (
    <div className="min-h-screen bg-white dark:bg-[#0B0C0E]">
      <MeetHeader />
      
      <div className="w-full">
        {/* Hero Section */}
        <HeroSection
          badge={{
            text: "Video Conferencing",
            showPulse: true
          }}
          title={{
            beforeHighlight: "Secure Video Calls and Meetings for",
            highlight: "Everyone",
            animated: false
          }}
          description={{
            desktop: "High-quality video meetings and calls available on any device. Connect, collaborate, and communicate seamlessly with Wurknest Meet.",
            mobile: "High-quality video meetings available on any device."
          }}
          cta={{
            primary: {
              text: "Start a Meeting",
              href: `https://hub.clikkle.com/signup?referralApp=Wurknest%20Website&redirectto=https://crew.clikkle.com/&theme=${theme}`
            },
            secondary: {
              text: "Learn More",
              href: "#features"
            }
          }}
          trustIndicators={[
            { text: "No downloads required" },
            { text: "Free to start" },
            { text: "Works on any device" }
          ]}
        />

        {/* Meeting Code Section */}
        <section className="w-full py-12 bg-gray-50 dark:bg-[#16171B] border-y border-gray-200 dark:border-white/10">
          <div className="container max-w-[1233px] mx-auto px-4 md:px-8">
            <div className="flex flex-col md:flex-row items-center justify-center gap-4">
              <div className="flex items-center gap-3 flex-1 max-w-md">
                <div className="flex items-center gap-2 flex-1 px-4 py-3 bg-white dark:bg-[#0B0C0E] border border-gray-300 dark:border-white/20 rounded-xl">
                  <KeyboardIcon className="w-5 h-5 text-gray-600 dark:text-gray-400" />
                  <input
                    type="text"
                    placeholder="Enter meeting code"
                    value={meetingCode}
                    onChange={(e) => setMeetingCode(e.target.value)}
                    onKeyPress={(e) => e.key === 'Enter' && handleJoinMeeting()}
                    className="flex-1 bg-transparent outline-none text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400"
                  />
                </div>
                <button
                  onClick={handleJoinMeeting}
                  disabled={!meetingCode.trim()}
                  className="px-6 py-3 bg-gray-900 dark:bg-white text-white dark:text-black rounded-xl font-semibold hover:scale-105 transition-all disabled:opacity-50 disabled:cursor-not-allowed disabled:hover:scale-100"
                >
                  Join
                </button>
              </div>
              <div className="flex items-center gap-2 text-sm text-gray-600 dark:text-gray-400">
                <span>or</span>
                <Link 
                  href={`https://hub.clikkle.com/signup?referralApp=Wurknest%20Website&redirectto=https://crew.clikkle.com/&theme=${theme}`}
                  className="text-gray-900 dark:text-white font-semibold hover:underline"
                >
                  Get Started Now
                </Link>
              </div>
            </div>
          </div>
        </section>

        {/* Features Section */}
        <section id="features" className="w-full py-20 md:py-32">
          <div className="container max-w-[1233px] mx-auto">
            <div className="text-center mb-16 md:mb-20">
              <h2 className="text-3xl md:text-5xl font-bold text-gray-900 dark:text-white mb-4">
                Everything You Need to{" "}
                <span className="relative inline-block after:content-[''] after:absolute after:bottom-0 after:left-0 after:right-0 after:h-3 after:bg-gray-300 dark:after:bg-gray-600 after:-z-10 after:rounded-sm">
                  Connect
                </span>
              </h2>
              <p className="text-lg text-gray-600 dark:text-gray-300 max-w-2xl mx-auto">
                Powerful features designed to make your video meetings productive and enjoyable
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8">
              {features.map((feature, index) => {
                const Icon = feature.icon;
                return (
                  <div
                    key={index}
                    className="group p-6 md:p-8 bg-white dark:bg-[#16171B] border border-gray-200 dark:border-white/10 rounded-2xl hover:shadow-2xl hover:scale-105 transition-all duration-300"
                  >
                    <div className="w-12 h-12 md:w-14 md:h-14 bg-gray-100 dark:bg-[#0B0C0E] rounded-xl flex items-center justify-center mb-4 group-hover:scale-110 transition-transform">
                      <Icon className="w-6 h-6 md:w-7 md:h-7 text-gray-900 dark:text-white" />
                    </div>
                    <h3 className="text-lg md:text-xl font-bold text-gray-900 dark:text-white mb-2">
                      {feature.title}
                    </h3>
                    <p className="text-sm md:text-base text-gray-600 dark:text-gray-300">
                      {feature.description}
                    </p>
                  </div>
                );
              })}
            </div>
          </div>
        </section>

        {/* Visual Feature Sections */}
        <section className="w-full py-20 md:py-32 bg-gray-50 dark:bg-[#16171B]">
          <div className="container max-w-[1233px] mx-auto space-y-32">
            {/* Feature 1 */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 lg:gap-16 items-center">
              <div className="order-2 lg:order-1">
                <h3 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
                  Meet on Any Device, Anywhere
                </h3>
                <p className="text-base md:text-lg text-gray-600 dark:text-gray-300 mb-6">
                  Guests can join from their computer using any modern web browser - no software to install. On mobile devices, they can join from the Wurknest Meet app.
                </p>
                <div className="flex flex-col sm:flex-row gap-4">
                  <Link
                    href="https://crew.clikkle.com"
                    className="px-6 py-3 bg-gray-900 dark:bg-white text-white dark:text-black rounded-xl font-semibold hover:scale-105 transition-all text-center"
                  >
                    Get Started
                  </Link>
                  <Link
                    href="https://crew.clikkle.com"
                    className="px-6 py-3 border-2 border-gray-900 dark:border-white text-gray-900 dark:text-white rounded-xl font-semibold hover:scale-105 transition-all text-center"
                  >
                    Learn More
                  </Link>
                </div>
              </div>
              <div className="order-1 lg:order-2">
                <Image
                  src="https://cdn.clikkle.com/images/crew/Clikkle%20crew%20LANDING%20PAGE%20UI/IMG_2737.JPG"
                  alt="Meet on any device"
                  className="rounded-2xl shadow-2xl w-full"
                  width={600}
                  height={400}
                />
              </div>
            </div>

            {/* Feature 2 */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 lg:gap-16 items-center">
              <div>
                <Image
                  src="https://cdn.clikkle.com/images/crew/Clikkle%20crew%20LANDING%20PAGE%20UI/Rectangle%209%20(1).png"
                  alt="Meet clearly"
                  className="rounded-2xl shadow-2xl w-full"
                  width={600}
                  height={400}
                />
              </div>
              <div>
                <h3 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
                  Meet Clearly
                </h3>
                <p className="text-base md:text-lg text-gray-600 dark:text-gray-300 mb-6">
                  Wurknest Meet adjusts to your network speed, ensuring high quality video calls wherever you are. New AI enhancements keep your calls clear even when your surroundings aren't.
                </p>
                <div className="flex flex-col sm:flex-row gap-4">
                  <Link
                    href="https://crew.clikkle.com"
                    className="px-6 py-3 bg-gray-900 dark:bg-white text-white dark:text-black rounded-xl font-semibold hover:scale-105 transition-all text-center"
                  >
                    Get Started
                  </Link>
                  <Link
                    href="https://crew.clikkle.com"
                    className="px-6 py-3 border-2 border-gray-900 dark:border-white text-gray-900 dark:text-white rounded-xl font-semibold hover:scale-105 transition-all text-center"
                  >
                    Learn More
                  </Link>
                </div>
              </div>
            </div>

            {/* Feature 3 */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 lg:gap-16 items-center">
              <div className="order-2 lg:order-1">
                <h3 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
                  Meet Safely
                </h3>
                <p className="text-base md:text-lg text-gray-600 dark:text-gray-300 mb-6">
                  Meet uses the same protections that Wurknest uses to secure your information and safeguard your privacy. Video conferences are encrypted in transit with continuous security updates.
                </p>
                <div className="flex flex-col sm:flex-row gap-4">
                  <Link
                    href="https://crew.clikkle.com"
                    className="px-6 py-3 bg-gray-900 dark:bg-white text-white dark:text-black rounded-xl font-semibold hover:scale-105 transition-all text-center"
                  >
                    Get Started
                  </Link>
                  <Link
                    href="https://crew.clikkle.com"
                    className="px-6 py-3 border-2 border-gray-900 dark:border-white text-gray-900 dark:text-white rounded-xl font-semibold hover:scale-105 transition-all text-center"
                  >
                    Learn More
                  </Link>
                </div>
              </div>
              <div className="order-1 lg:order-2">
                <Image
                  src="https://cdn.clikkle.com/images/crew/Clikkle%20crew%20LANDING%20PAGE%20UI/Rectangle%208%20(1).png"
                  alt="Meet safely"
                  className="rounded-2xl shadow-2xl w-full"
                  width={600}
                  height={400}
                />
              </div>
            </div>
          </div>
        </section>

        {/* Apps Section */}
        <AppsSection />

        {/* Pre-Footer CTA */}
        <SecuritySection />
        <PreFooterCTA
          badge="Ready to Connect?"
          mobileHeadline="Start Your First Meeting"
          desktopHeadline={`<span class="relative inline-block after:content-[''] after:absolute after:bottom-0 after:left-0 after:right-0 after:h-3 after:bg-gray-300 dark:after:bg-gray-600 after:-z-10 after:rounded-sm">Start Your First Meeting</span> Today.`}
          description='Experience high-quality video calls with enterprise-grade security.<span class="hidden md:inline"> No credit card required.</span>'
          buttonText="Try Wurknest Wurknest Free"
          buttonUrl={`https://hub.clikkle.com/signup?referralApp=Wurknest%20Website&redirectto=https://crew.clikkle.com/&theme=${theme}`}
          userImages={[user1, user2, user3, user4]}
          userCount="50,000+ users"
          rating="4.8/5"
          theme={theme}
        />
      </div>

      <Footer />
      <SidebarDrawer open={leftDrawerOpen} toggle={toggleLeftDrawer} />
    </div>
  );
};

export default CrewHome;
