"use client";
import React, { useEffect, useRef, useState } from "react";
import {
  Mail,
  Phone,
  MapPin,
  ArrowRight,
  MessageSquare,
  Users,
  Newspaper,
  ArrowRightIcon,
  Logs,
  Ellipsis,
  Send,
} from "lucide-react";
import { useTheme } from "../atoms/theme";
import AnnouncementBar from "../components/AnnouncementBar";
import Link from "next/link";
import Image from "next/image";
import SidebarDrawer from "../components/sidebarDrawer";
import { footerColumns } from "../constants/footerText";
import {
  FaArrowRight,
  FaLinkedin,
  FaInstagram,
  FaFacebook,
  FaYoutube,
} from "react-icons/fa";
import { getCookie } from "../utils/cookies";
import { apps } from "../constants/products";
import { Button } from "../components/button";
import illustration from "../../public/images/banner.png";
import ThemeSwitch from "../components/themeSwitch";
import { motion, AnimatePresence } from "framer-motion";
import Footer from "@/components/Footer";
import ClikkleLogo from "@/components/ClikkleLogo";
import MainHeader from "@/components/MainHeader";

const Contact: React.FC = () => {
  const [leftDrawerOpen, setLeftDrawerOpen] = useState(false);
  const toggleLeftDrawer = () => setLeftDrawerOpen((prev) => !prev);
  const { theme } = useTheme();

  const [productsOpen, setProductsOpen] = useState(false);
  const [dotsOpen, setDotsOpen] = useState(false);
  const productsRef = useRef<HTMLDivElement>(null);
  const dotsRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (
        productsRef.current &&
        !productsRef.current.contains(event.target as Node)
      ) {
        setProductsOpen(false);
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (dotsRef.current && !dotsRef.current.contains(event.target as Node)) {
        setDotsOpen(false);
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);
  const [scrolled, setScrolled] = useState(false);
  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 0);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);
  const [announcementOpen, setAnnouncementOpen] = useState(false);

  useEffect(() => {
    const checkHeight = () => {
      const el = document.querySelector("[data-announcement]") as HTMLElement;
      if (!el) return;
      if (el.offsetHeight === 0) {
        setAnnouncementOpen(false);
      }
    };
    checkHeight();
    const observer = new MutationObserver(checkHeight);
    observer.observe(document.body, { childList: true, subtree: true });
    return () => observer.disconnect();
  }, []);
  const [subject, setSubject] = useState<string>("General Inquiry");
  const [fullName, setFullName] = useState<string>("");
  const [email, setEmail] = useState<string>("");
  const [phone, setPhone] = useState<string>("");
  const [message, setMessage] = useState<string>("");
  const [submitting, setSubmitting] = useState<boolean>(false);
  const [submitError, setSubmitError] = useState<string>("");

  const messages = [
    "How can we help you?",
    "Request a demo for Hub",
    "Ask about pricing for your team",
    "Report an issue or bug",
    "Media & press inquiry",

    "I can't log into my Hub dashboard",
    "How do I reset my account password?",
    "My team is not receiving email notifications",
    "How can I upgrade my subscription plan?",
    "How do I add more users to my organization?",

    "How can I export reports from Hub?",
    "Error uploading files in Projects",
    "How do I customize company branding settings?",
    "Where can I download the mobile app?",
    "My invoices are not generating properly",

    "How can I integrate Wurknest Chat with other tools?",
    "Why am I not receiving verification emails?",
    "How do I set up payment gateways in Store?",
    "How do I invite new team members?",
    "How can I submit feedback or feature requests?",
  ];

  const [msgIndex, setMsgIndex] = useState(0);
  const [msg, setMsg] = useState(messages[0]);
  const [placeholderPaused, setPlaceholderPaused] = useState(false);
  const [messageValue, setMessageValue] = useState("");
  const [focused, setFocused] = useState(false);
  useEffect(() => {
    if (placeholderPaused) return;
    const id = setInterval(() => {
      setMsgIndex((i) => (i + 1) % messages.length);
    }, 2000);
    return () => clearInterval(id);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [placeholderPaused]);
  useEffect(() => {
    setMsg(messages[msgIndex]);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [msgIndex]);

  useEffect(() => {
    const checkDismissedAnnouncements = () => {
      const dismissed = getCookie("dismissedAnnouncements");
      if (dismissed) {
        try {
          const dismissedIds = JSON.parse(dismissed);
          if (dismissedIds.length > 0) {
            setAnnouncementOpen(false);
          }
        } catch (error) {
          console.error("Error parsing dismissed announcements:", error);
        }
      }
    };

    checkDismissedAnnouncements();

    const interval = setInterval(checkDismissedAnnouncements, 1000);

    return () => clearInterval(interval);
  }, []);
  const tryOpenChat = () => {
    const chatBtn = document.querySelector(
      'button[title="Open chat"]'
    ) as HTMLButtonElement;
    if (chatBtn) {
      chatBtn.click();
    } else {
      setTimeout(tryOpenChat, 500);
    }
  };
  return (
    <div className="min-h-screen bg-white dark:bg-[#0B0C0E] transition-colors duration-300">
      <section className="w-full">
        <section className="relative w-full h-28 overflow-hidden">
          <div
            className={`fixed top-0 inset-x-0 z-50 ${
              scrolled
                ? "backdrop-blur-md bg-white/90 dark:bg-[#1a1925]/90"
                : productsOpen || dotsOpen
                ? "backdrop-blur-md bg-white/90 dark:bg-[#1a1925]/90"
                : ""
            }`}
          >
            <div
              ref={(el) => {
                if (el) {
                  const height = el.getBoundingClientRect().height;

                  if (height !== 0) {
                    setAnnouncementOpen(true);
                  }
                }
              }}
            >
              <AnnouncementBar onClose={() => setAnnouncementOpen(false)} />
            </div>
            <MainHeader/>
          </div>
        </section>
      </section>
      <div className="max-w-7xl mx-auto px-6 pt-20 pb-16 md:pb-20">
        <div className="mx-auto">
          <div
            className={`max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 ${
              announcementOpen ? "pt-6" : ""
            } py-16 pt-4`}
          >
            <div className="text-center">
              <h1 className="text-4xl md:text-5xl font-bold text-[#111827] dark:text-white mb-4">
                Contact us
              </h1>
              <p className="text-xl text-[#4b5563] dark:text-[#d1d5db] max-w-3xl mx-auto">
                Have a question or need assistance? Start a chat, open a support
                ticket, or send us a message and our team will be happy to help.
              </p>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-16">
            <div className="group relative h-full">
              <div className="relative h-full flex flex-col p-6 bg-white/70 dark:bg-white/[0.03] backdrop-blur-sm border border-gray-300/80 dark:border-white/20 rounded-3xl overflow-hidden shadow-md shadow-gray-900/10 dark:shadow-white/5 transition-all duration-500 cursor-pointer hover:bg-white dark:hover:bg-white/10 hover:border-gray-400 dark:hover:border-white/30 hover:shadow-2xl hover:shadow-gray-900/20 dark:hover:shadow-white/10 hover:-translate-y-2">
                <div className="absolute inset-0 bg-gradient-to-br from-transparent via-transparent to-gray-50/50 dark:to-white/5 opacity-0 group-hover:opacity-100 transition-opacity duration-500" />
                <div className="relative z-10 flex flex-col h-full items-center justify-center text-center">
                  <div className="bg-[#e5e7eb] dark:bg-[#1f2937]/80 rounded-2xl p-3 w-fit mb-4 group-hover:scale-110 transition-transform duration-300 shadow-sm">
                    <Users className="w-7 h-7 text-[#111827] dark:text-white" />
                  </div>
                  <h3 className="text-lg font-bold text-[#111827] dark:text-white mb-2">Help & Support</h3>
                  <p className="text-sm text-[#4b5563] dark:text-[#d1d5db] leading-relaxed flex-grow">
                    Get assistance with setup, features, or troubleshooting.
                  </p>
                  <Button
                    onClick={() => {
                      tryOpenChat();
                    }}
                    className="mt-5 inline-flex bg-[#111827] dark:bg-white text-white dark:text-[#111827] hover:bg-[#374151] dark:hover:bg-[#e5e7eb] items-center gap-2 rounded-xl px-4 py-2 text-sm font-semibold transition-all duration-300"
                  >
                    Contact support <ArrowRight className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            </div>
            <div className="group relative h-full">
              <div className="relative h-full flex flex-col p-6 bg-white/70 dark:bg-white/[0.03] backdrop-blur-sm border border-gray-300/80 dark:border-white/20 rounded-3xl overflow-hidden shadow-md shadow-gray-900/10 dark:shadow-white/5 transition-all duration-500 cursor-pointer hover:bg-white dark:hover:bg-white/10 hover:border-gray-400 dark:hover:border-white/30 hover:shadow-2xl hover:shadow-gray-900/20 dark:hover:shadow-white/10 hover:-translate-y-2">
                <div className="absolute inset-0 bg-gradient-to-br from-transparent via-transparent to-gray-50/50 dark:to-white/5 opacity-0 group-hover:opacity-100 transition-opacity duration-500" />
                <div className="relative z-10 flex flex-col h-full items-center justify-center text-center">
                  <div className="bg-[#e5e7eb] dark:bg-[#1f2937]/80 rounded-2xl p-3 w-fit mb-4 group-hover:scale-110 transition-transform duration-300 shadow-sm">
                    <MessageSquare className="w-7 h-7 text-[#111827] dark:text-white" />
                  </div>
                  <h3 className="text-lg font-bold text-[#111827] dark:text-white mb-2">
                    Create or Manage Tickets
                  </h3>
                  <p className="text-sm text-[#4b5563] dark:text-[#d1d5db] leading-relaxed flex-grow">
                    Submit a new support request or track the status of your
                    existing tickets.
                  </p>
                  <Button
                    onClick={() => {
                      window.location.replace(
                        "https://support.clikkle.com/?category=Tickets"
                      );
                    }}
                    className="mt-5 inline-flex bg-[#111827] dark:bg-white text-white dark:text-[#111827] hover:bg-[#374151] dark:hover:bg-[#e5e7eb] items-center gap-2 rounded-xl px-4 py-2 text-sm font-semibold transition-all duration-300"
                  >
                    Open Support Portal <ArrowRight className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            </div>

            <div className="group relative h-full">
              <div className="relative h-full flex flex-col p-6 bg-white/70 dark:bg-white/[0.03] backdrop-blur-sm border border-gray-300/80 dark:border-white/20 rounded-3xl overflow-hidden shadow-md shadow-gray-900/10 dark:shadow-white/5 transition-all duration-500 cursor-pointer hover:bg-white dark:hover:bg-white/10 hover:border-gray-400 dark:hover:border-white/30 hover:shadow-2xl hover:shadow-gray-900/20 dark:hover:shadow-white/10 hover:-translate-y-2">
                <div className="absolute inset-0 bg-gradient-to-br from-transparent via-transparent to-gray-50/50 dark:to-white/5 opacity-0 group-hover:opacity-100 transition-opacity duration-500" />
                <div className="relative z-10 flex flex-col h-full items-center justify-center text-center">
                  <div className="bg-[#e5e7eb] dark:bg-[#1f2937]/80 rounded-2xl p-3 w-fit mb-4 group-hover:scale-110 transition-transform duration-300 shadow-sm">
                    <Newspaper className="w-7 h-7 text-[#111827] dark:text-white" />
                  </div>
                  <h3 className="text-lg font-bold text-[#111827] dark:text-white mb-2">Knowledge Base</h3>
                  <p className="text-sm text-[#4b5563] dark:text-[#d1d5db] leading-relaxed flex-grow">
                    Find answers to common questions and learn how to use Wurknest.
                  </p>
                  <Button
                    onClick={() => {
                      window.location.replace(
                        "https://support.clikkle.com/?category=Knowledge"
                      );
                    }}
                    className="mt-5 inline-flex bg-[#111827] dark:bg-white text-white dark:text-[#111827] hover:bg-[#374151] dark:hover:bg-[#e5e7eb] items-center gap-2 rounded-xl px-4 py-2 text-sm font-semibold transition-all duration-300"
                  >
                    Knowledge base <ArrowRight className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-8 items-start">
            <div className="group relative h-full">
              <div className="relative h-full flex flex-col justify-between p-8 bg-white/70 dark:bg-white/[0.03] backdrop-blur-sm border border-gray-300/80 dark:border-white/20 rounded-3xl overflow-hidden shadow-md shadow-gray-900/10 dark:shadow-white/5 transition-all duration-500">
                <div className="absolute inset-0 bg-gradient-to-br from-transparent via-transparent to-gray-50/50 dark:to-white/5" />
                <div className="relative z-10">
                  <h2 className="text-2xl font-bold text-[#111827] dark:text-white">
                    Contact Information
                  </h2>
                  <p className="mt-2 text-sm text-[#4b5563] dark:text-[#d1d5db]">
                    Help when you need it — our support portal and team are here.
                  </p>
                  <div className="mt-10 space-y-4">
                    <p
                      onClick={() => {
                        tryOpenChat();
                      }}
                      className="flex items-center gap-3 cursor-pointer text-[#111827] dark:text-white hover:text-[#374151] dark:hover:text-[#e5e7eb] transition-colors"
                    >
                      <Phone className="h-5 w-5" />
                      <span className="text-sm">
                        Live chat with our support team
                      </span>
                    </p>
                    <a
                      href="mailto:support@Wurknest.com"
                      className="flex items-center gap-3 text-[#111827] dark:text-white hover:text-[#374151] dark:hover:text-[#e5e7eb] transition-colors"
                    >
                      <Mail className="h-5 w-5" />
                      <span className="text-sm">support@Wurknest.com</span>
                    </a>
                    <div className="flex items-center gap-3 text-[#111827] dark:text-white">
                      <MapPin className="h-5 w-5" />
                      <span className="text-sm">
                        6-470 king st west suite 255 Oshawa ON L1J 2K9 Canada
                      </span>
                    </div>
                  </div>
                </div>

                <div className="relative z-10 mt-8 flex items-center gap-3">
                  <a
                    href="https://www.linkedin.com/company/clikkle"
                    target="_blank"
                    rel="noopener noreferrer"
                    className="h-12 w-12 rounded-full bg-[#e5e7eb] dark:bg-[#1f2937]/80 flex items-center justify-center hover:scale-110 transition-transform duration-300 shadow-sm"
                    aria-label="LinkedIn"
                  >
                    <FaLinkedin className="w-5 h-5 text-[#111827] dark:text-white" />
                  </a>
                  <a
                    href="https://instagram.com/myclikkle"
                    target="_blank"
                    rel="noopener noreferrer"
                    className="h-12 w-12 rounded-full bg-[#e5e7eb] dark:bg-[#1f2937]/80 flex items-center justify-center hover:scale-110 transition-transform duration-300 shadow-sm"
                    aria-label="Instagram"
                  >
                    <FaInstagram className="w-5 h-5 text-[#111827] dark:text-white" />
                  </a>
                  <a
                    href="https://facebook.com/clikkle"
                    target="_blank"
                    rel="noopener noreferrer"
                    className="h-12 w-12 rounded-full bg-[#e5e7eb] dark:bg-[#1f2937]/80 flex items-center justify-center hover:scale-110 transition-transform duration-300 shadow-sm"
                    aria-label="Facebook"
                  >
                    <FaFacebook className="w-5 h-5 text-[#111827] dark:text-white" />
                  </a>
                  <a
                    href="https://youtube.com/clikkle"
                    target="_blank"
                    rel="noopener noreferrer"
                    className="h-12 w-12 rounded-full bg-[#e5e7eb] dark:bg-[#1f2937]/80 flex items-center justify-center hover:scale-110 transition-transform duration-300 shadow-sm"
                    aria-label="YouTube"
                  >
                    <FaYoutube className="w-5 h-5 text-[#111827] dark:text-white" />
                  </a>
                </div>
              </div>
            </div>

            <div className="group relative h-full">
              <div className="relative h-full p-8 bg-white/70 dark:bg-white/[0.03] backdrop-blur-sm border border-gray-300/80 dark:border-white/20 rounded-3xl overflow-hidden shadow-md shadow-gray-900/10 dark:shadow-white/5 transition-all duration-500">
                <div className="absolute inset-0 bg-gradient-to-br from-transparent via-transparent to-gray-50/50 dark:to-white/5" />
                <div className="relative z-10">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-[#111827] dark:text-white pl-3">Full Name</label>
                      <input
                        className="w-full bg-white dark:bg-[#1a1925]/50 rounded-xl border border-gray-300/80 dark:border-white/20 px-4 py-3 text-sm outline-none focus:border-gray-400 dark:focus:border-white/40 transition-colors text-[#111827] dark:text-white"
                        placeholder="John Doe"
                        value={fullName}
                        onChange={(e) => setFullName(e.target.value)}
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-[#111827] dark:text-white pl-3">Email</label>
                      <input
                        className="w-full bg-white dark:bg-[#1a1925]/50 rounded-xl border border-gray-300/80 dark:border-white/20 px-4 py-3 text-sm outline-none focus:border-gray-400 dark:focus:border-white/40 transition-colors text-[#111827] dark:text-white"
                        placeholder="johndoe@example.com"
                        value={email}
                        onChange={(e) => setEmail(e.target.value)}
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-[#111827] dark:text-white pl-3">Phone Number</label>
                      <input
                        className="w-full bg-white dark:bg-[#1a1925]/50 rounded-xl border border-gray-300/80 dark:border-white/20 px-4 py-3 text-sm outline-none focus:border-gray-400 dark:focus:border-white/40 transition-colors text-[#111827] dark:text-white"
                        placeholder="+1 012 3456 789"
                        value={phone}
                        onChange={(e) => setPhone(e.target.value)}
                      />
                    </div>
                  </div>
                  <div className="mt-6">
                    <p className="text-sm font-medium text-[#111827] dark:text-white">Select Subject</p>
                    <div className="mt-3 flex flex-wrap items-center gap-4">
                      <label className="flex items-center gap-2 text-xs text-[#111827] dark:text-white cursor-pointer">
                        <input
                          type="radio"
                          name="subject"
                          className="accent-[#111827] dark:accent-white"
                          checked={subject === "General Inquiry"}
                          onChange={() => setSubject("General Inquiry")}
                        />
                        General Inquiry
                      </label>
                      <label className="flex items-center gap-2 text-xs text-[#111827] dark:text-white cursor-pointer">
                        <input
                          type="radio"
                          name="subject"
                          className="accent-[#111827] dark:accent-white"
                          checked={subject === "Sales & Partnerships"}
                          onChange={() => setSubject("Sales & Partnerships")}
                        />
                        Sales & Partnerships
                      </label>
                      <label className="flex items-center gap-2 text-xs text-[#111827] dark:text-white cursor-pointer">
                        <input
                          type="radio"
                          name="subject"
                          className="accent-[#111827] dark:accent-white"
                          checked={subject === "Technical Support"}
                          onChange={() => setSubject("Technical Support")}
                        />
                        Technical Support
                      </label>
                      <label className="flex items-center gap-2 text-xs text-[#111827] dark:text-white cursor-pointer">
                        <input
                          type="radio"
                          name="subject"
                          className="accent-[#111827] dark:accent-white"
                          checked={subject === "Billing & Account"}
                          onChange={() => setSubject("Billing & Account")}
                        />
                        Billing & Account
                      </label>
                      <label className="flex items-center gap-2 text-xs text-[#111827] dark:text-white cursor-pointer">
                        <input
                          type="radio"
                          name="subject"
                          className="accent-[#111827] dark:accent-white"
                          checked={subject === "Product Demo"}
                          onChange={() => setSubject("Product Demo")}
                        />
                        Product Demo
                      </label>
                    </div>
                  </div>
                  <div className="mt-6 space-y-2">
                    <label className="text-sm font-medium text-[#111827] dark:text-white">Message</label>
                    <div className="relative">
                      <textarea
                        className="min-h-[140px] w-full bg-white dark:bg-[#1a1925]/50 rounded-xl border border-gray-300/80 dark:border-white/20 px-4 py-3 text-sm outline-none focus:border-gray-400 dark:focus:border-white/40 transition-colors text-[#111827] dark:text-white resize-none"
                        placeholder=""
                        onFocus={() => {
                          setPlaceholderPaused(true);
                          setFocused(true);
                        }}
                        onBlur={() => {
                          setPlaceholderPaused(false);
                          setFocused(false);
                        }}
                        value={message}
                        onChange={(e) => {
                          setMessageValue(e.target.value);
                          setMessage(e.target.value);
                        }}
                      />
                      {!focused && messageValue.length === 0 && (
                        <AnimatePresence mode="wait">
                          <motion.span
                            key={msg}
                            className="pointer-events-none absolute left-4 top-3 text-sm text-[#9ca3af] dark:text-[#6b7280]"
                            initial={{ opacity: 0, y: 2 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -2 }}
                            transition={{ duration: 0.25 }}
                          >
                            {msg}
                          </motion.span>
                        </AnimatePresence>
                      )}
                    </div>
                  </div>
                  <div className="mt-6 flex flex-col gap-3">
                    {submitError && (
                      <div className="text-[#dc2626] dark:text-[#f87171] text-sm font-medium">{submitError}</div>
                    )}
                    <div className="flex flex-col gap-3">
                      <div className="flex justify-end">
                        <Button
                          className="inline-flex items-center gap-2 rounded-xl bg-[#111827] dark:bg-white text-white dark:text-[#111827] hover:bg-[#374151] dark:hover:bg-[#e5e7eb] px-6 py-3 text-sm font-semibold transition-all duration-300 shadow-md hover:shadow-lg"
                          onClick={async () => {
                            if (submitting) return;
                            setSubmitError("");
                            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                            if (!fullName.trim()) {
                              setSubmitError("Please enter your full name.");
                              return;
                            }
                            if (!emailRegex.test(email)) {
                              setSubmitError(
                                "Please enter a valid email address."
                              );
                              return;
                            }
                            if (!message.trim() || message.trim().length < 10) {
                              setSubmitError(
                                "Message should be at least 10 characters."
                              );
                              return;
                            }

                            setSubmitting(true);

                            try {
                              let recipientEmail = "support@clikkle.com";
                              switch (subject) {
                                case "General Inquiry":
                                  recipientEmail = "hi@clikkle.com";
                                  break;
                                case "Billing & Account":
                                  recipientEmail = "billing@clikkle.com";
                                  break;
                                case "Technical Support":
                                  recipientEmail = "support@clikkle.com";
                                  break;
                                case "Sales & Partnerships":
                                case "Product Demo":
                                default:
                                  recipientEmail = "sales@clikkle.com";
                                  break;
                              }

                              const mailtoLink = `mailto:${recipientEmail}?subject=${encodeURIComponent(
                                subject
                              )}&body=${encodeURIComponent(
                                `Full Name: ${fullName.trim()}\n\nEmail: ${email.trim()}\n\nPhone Number: ${
                                  phone.trim() || "Not provided"
                                }\n\nMessage:\n${message.trim()}`
                              )}`;

                              const link = document.createElement("a");
                              link.href = mailtoLink;
                              link.style.display = "none";
                              document.body.appendChild(link);
                              link.click();
                              document.body.removeChild(link);

                              setFullName("");
                              setEmail("");
                              setPhone("");
                              setMessage("");
                              setMessageValue("");
                            } catch {
                              setSubmitError(
                                "Failed to open email client. Please manually email support@clikkle.com"
                              );
                            } finally {
                              setSubmitting(false);
                            }
                          }}
                          disabled={submitting}
                        >
                          {submitting ? "Opening Email..." : "Send Message"}{" "}
                          <Send className="w-6 h-6" />
                        </Button>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
      <SidebarDrawer open={leftDrawerOpen} toggle={toggleLeftDrawer} />
      <Footer />
    </div>
  );
};

export default Contact;
