/* eslint-disable @next/next/no-img-element */
"use client";
/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { useState, useEffect, useRef } from "react";
import Link from "next/link";
import Image from "next/image";
import { useTheme } from "../atoms/theme";
import {
  Search,
  Calendar,
  Clock,
  Filter,
  SortAsc,
  SortDesc,
  ArrowRightIcon,
  Logs,
  Ellipsis,
} from "lucide-react";
import { Button } from "../components/button";
import { Card, CardContent } from "../components/card";
import { BlogPost, blogCategories } from "../constants/blogData";
import ThemeSwitch from "../components/themeSwitch";
import { FaArrowRight } from "react-icons/fa";
import { footerColumns } from "../constants/footerText";
import { apps } from "../constants/products";
import illustration from "../../public/images/banner.png";
import SidebarDrawer from "../components/sidebarDrawer";
import AnnouncementBar from "../components/AnnouncementBar";
import { getCookie } from "../utils/cookies";
import ClikkleLogo from "@/components/ClikkleLogo";
import Footer from "@/components/Footer";
import { FaArrowCircleDown } from "react-icons/fa";
import { iconMap } from "../utils/iconMap";
import MainHeader from "@/components/MainHeader";

const DEFAULT_API_BASE_URL = "https://admin.clikkle.com:5100/api";

const POSTS_PER_PAGE = 6;

const API_BASE_URL = DEFAULT_API_BASE_URL;

const BlogsHome = () => {
  const { theme } = useTheme();
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedCategory, setSelectedCategory] = useState("All");
  const [sortBy, setSortBy] = useState<"date" | "title" | "order">("date");
  const [sortOrder, setSortOrder] = useState<"asc" | "desc">("desc");
  const [currentPage, setCurrentPage] = useState(1);
  const [posts, setPosts] = useState<BlogPost[]>([]);
  const [totalCount, setTotalCount] = useState(0);
  const [totalPages, setTotalPages] = useState(1);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [isVisible] = useState(false);

  useEffect(() => {
    const controller = new AbortController();
    const fetchBlogs = async () => {
      try {
        setLoading(true);
        setError(null);
        const params = new URLSearchParams();
        params.set("page", String(currentPage));
        params.set("limit", String(POSTS_PER_PAGE));
        params.set("status", "published");
        if (searchTerm) params.set("search", searchTerm);
        if (selectedCategory !== "All")
          params.set("category", selectedCategory);
        const sortByServer =
          sortBy === "date"
            ? "publishedDate"
            : sortBy === "order"
            ? "orderIndex"
            : "title";
        params.set("sortBy", sortByServer);
        params.set("sortOrder", sortOrder);

        const res = await fetch(
          `${API_BASE_URL}/blogs/public?${params?.toString()}`,
          {
            signal: controller.signal,
          }
        );
        const json = await res.json();
        if (!res.ok || !json?.success) {
          throw new Error(json?.error || "Failed to fetch blogs");
        }
        const apiBlogs = Array.isArray(json.data) ? json.data : [];
        const mapped: BlogPost[] = apiBlogs.map((b: any) => ({
          id: b._id,
          slug: b.slug,
          title: b.title,
          excerpt: b.excerpt || "",
          content: b.content || "",
          featuredImage:
            b.featuredImage ||
            "https://cdn.clikkle.com/images/clikkle/logo/2025/clikkle.png",
          author: {
            name: b.author?.name || "Unknown Author",
            avatar:
              b.author?.avatar ||
              "https://cdn.clikkle.com/images/clikkle/logo/2025/clikkle.png",
            bio: b.author?.bio || "",
          },
          publishedDate: b.publishedDate || new Date().toISOString(),
          category: b.category || "News",
          tags: Array.isArray(b.tags) ? b.tags : [],
          readTime: b.readTime ?? b.calculatedReadTime ?? 0,
          seoTitle: b.metaTitle || b.title,
          seoDescription: b.metaDescription || b.excerpt || "",
        }));
        if (currentPage === 1) {
          setPosts(mapped);
        } else {
          setPosts((prev) => [...prev, ...mapped]);
        }
        setTotalCount(json.pagination?.total ?? mapped.length);
        setTotalPages(json.pagination?.pages ?? 1);
      } catch (err: any) {
        if (err?.name === "AbortError") return;
        setError(err?.message || "Unexpected error fetching blogs");
      } finally {
        if (!controller.signal.aborted) {
          setLoading(false);
        }
      }
    };

    fetchBlogs();
    return () => controller.abort();
  }, [searchTerm, selectedCategory, sortBy, sortOrder, currentPage]);

  useEffect(() => {
    setCurrentPage(1);
  }, [searchTerm, selectedCategory, sortBy, sortOrder]);

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "long",
      day: "numeric",
    });
  };

  const [leftDrawerOpen, setLeftDrawerOpen] = useState(false);
  const toggleLeftDrawer = () => setLeftDrawerOpen((prev) => !prev);

  const [productsOpen, setProductsOpen] = useState(false);
  const [dotsOpen, setDotsOpen] = useState(false);
  const productsRef = useRef<HTMLDivElement>(null);
  const dotsRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (
        productsRef.current &&
        !productsRef.current.contains(event.target as Node)
      ) {
        setProductsOpen(false);
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (dotsRef.current && !dotsRef.current.contains(event.target as Node)) {
        setDotsOpen(false);
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  const [scrolled, setScrolled] = useState(false);
  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 0);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);
  const [announcementOpen, setAnnouncementOpen] = useState(false);

  useEffect(() => {
    const checkHeight = () => {
      const el = document.querySelector("[data-announcement]") as HTMLElement;
      if (!el) return;
      if (el.offsetHeight === 0) {
        setAnnouncementOpen(false);
      }
    };
    checkHeight();
    const observer = new MutationObserver(checkHeight);
    observer.observe(document.body, { childList: true, subtree: true });
    return () => observer.disconnect();
  }, []);

  useEffect(() => {
    const checkDismissedAnnouncements = () => {
      const dismissed = getCookie("dismissedAnnouncements");
      if (dismissed) {
        try {
          const dismissedIds = JSON.parse(dismissed);
          if (dismissedIds.length > 0) {
            setAnnouncementOpen(false);
          }
        } catch (error) {
          console.error("Error parsing dismissed announcements:", error);
        }
      }
    };

    checkDismissedAnnouncements();

    const interval = setInterval(checkDismissedAnnouncements, 1000);

    return () => clearInterval(interval);
  }, []);
  return (
    <>
      <div className={`min-h-screen bg-white dark:bg-[#0B0C0E]`}>
        <section className="w-full">
          <section className="relative w-full h-28 overflow-hidden">
            <div
              className={`fixed top-0 inset-x-0 z-50 ${
                scrolled
                  ? "backdrop-blur-md bg-white/90 dark:bg-[#0B0C0E]/90"
                  : productsOpen || dotsOpen
                  ? "backdrop-blur-md bg-white/90 dark:bg-[#0B0C0E]/90"
                  : ""
              }`}
            >
              <div
                ref={(el) => {
                  if (el) {
                    const height = el.getBoundingClientRect().height;

                    if (height !== 0) {
                      setAnnouncementOpen(true);
                    }
                  }
                }}
              >
                <AnnouncementBar onClose={() => setAnnouncementOpen(false)} />
              </div>
              <MainHeader/>
            </div>
          </section>
        </section>
        <div
          className={`bg-white dark:bg-[#0B0C0E] border-b ${
            theme === "dark" ? "border-gray-700" : "border-gray-200"
          }`}
        >
          <div
            className={`max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 ${
              announcementOpen ? "pt-6" : ""
            } py-16 pt-4`}
          >
            <div className="text-center">
              <h1
                className={`text-4xl md:text-5xl font-bold text-[#253237] dark:text-white mb-4`}
              >
                Our Blogs
              </h1>
              <p
                className={`text-xl ${
                  theme === "dark" ? "text-gray-300" : "text-gray-600"
                } max-w-3xl mx-auto`}
              >
                Stay updated with the latest insights, trends, and news from the
                world of business technology and innovation.
              </p>
            </div>
          </div>
        </div>

        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
          <div className="mb-8 space-y-4 lg:space-y-0 lg:flex lg:items-center lg:justify-between">
            <div className="relative flex-1 max-w-md">
              <Search
                className={`absolute left-3 top-1/2 transform -translate-y-1/2 ${
                  theme === "dark" ? "text-gray-400" : "text-gray-500"
                } h-5 w-5`}
              />
              <input
                type="text"
                placeholder="Search blog posts..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className={`w-full pl-10 pr-4 py-3 rounded-lg border bg-white dark:bg-[#0B0C0E] ${
                  theme === "dark"
                    ? " border-gray-600 text-white placeholder-gray-400"
                    : "bg-white border-gray-300 text-gray-900 placeholder-gray-500"
                } focus:ring-2 focus:ring-[#253237] dark:focus:ring-white/20 focus:border-transparent`}
              />
            </div>

            <div className="flex flex-wrap gap-4 lg:ml-6">
              <div className="flex items-center gap-2">
                <Filter
                  className={`h-5 w-5 ${
                    theme === "dark" ? "text-gray-400" : "text-gray-500"
                  }`}
                />
                <select
                  value={selectedCategory}
                  onChange={(e) => setSelectedCategory(e.target.value)}
                  className={`px-4 py-2 rounded-lg border bg-white dark:bg-[#0B0C0E] ${
                    theme === "dark"
                      ? " border-gray-600 text-white"
                      : "bg-white border-gray-300 text-gray-900"
                  } focus:ring-2 focus:ring-[#253237] dark:focus:ring-white/20 focus:border-transparent`}
                >
                  {blogCategories.map((category) => (
                    <option key={category} value={category}>
                      {category}
                    </option>
                  ))}
                </select>
              </div>

              <div className="flex items-center gap-2">
                <button
                  onClick={() => {
                    if (sortBy === "date") setSortBy("title");
                    else if (sortBy === "title") setSortBy("order");
                    else setSortBy("date");
                  }}
                  className={`flex items-center gap-2 px-4 py-2 rounded-lg border ${
                    theme === "dark"
                      ? " border-gray-600 text-white hover:bg-gray-700"
                      : "bg-white border-gray-300 text-gray-900 hover:bg-gray-50"
                  } transition-colors`}
                >
                  {sortBy === "date"
                    ? "Date"
                    : sortBy === "order"
                    ? "Latest"
                    : "Title"}
                  {sortOrder === "desc" ? (
                    <SortDesc className="h-4 w-4" />
                  ) : (
                    <SortAsc className="h-4 w-4" />
                  )}
                </button>
                <button
                  onClick={() =>
                    setSortOrder(sortOrder === "desc" ? "asc" : "desc")
                  }
                  className={`p-2 rounded-lg border ${
                    theme === "dark"
                      ? " border-gray-600 text-white hover:bg-gray-700"
                      : "bg-white border-gray-300 text-gray-900 hover:bg-gray-50"
                  } transition-colors`}
                >
                  {sortOrder === "desc" ? (
                    <SortDesc className="h-4 w-4" />
                  ) : (
                    <SortAsc className="h-4 w-4" />
                  )}
                </button>
              </div>
            </div>
          </div>

          <div className="mb-6">
            <p
              className={`text-sm ${
                theme === "dark" ? "text-gray-400" : "text-gray-600"
              }`}
            >
              Showing {posts.length} of {totalCount} posts
              {selectedCategory !== "All" && ` in "${selectedCategory}"`}
              {searchTerm && ` matching "${searchTerm}"`}
            </p>
          </div>

          {loading && currentPage === 1 ? (
            <div className="text-center py-16">
              <div className="flex justify-center mb-6">
                <div className="relative w-16 h-16">
                  {/* Vercel-style circular dashes loader */}
                  <svg className="animate-spin" viewBox="0 0 64 64" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <circle 
                      cx="32" 
                      cy="32" 
                      r="28" 
                      stroke="currentColor" 
                      strokeWidth="4"
                      strokeDasharray="8 8"
                      strokeLinecap="round"
                      className="text-[#253237] dark:text-white opacity-30"
                    />
                    <circle 
                      cx="32" 
                      cy="32" 
                      r="28" 
                      stroke="currentColor" 
                      strokeWidth="4"
                      strokeDasharray="44 132"
                      strokeLinecap="round"
                      className="text-[#253237] dark:text-white"
                      transform="rotate(-90 32 32)"
                    />
                  </svg>
                </div>
              </div>
              <p className="text-gray-900 dark:text-white font-semibold text-lg mb-2">
                Loading blog posts...
              </p>
              <p className="text-gray-600 dark:text-gray-400 text-sm">
                Please wait while we fetch the latest content
              </p>
            </div>
          ) : posts.length > 0 ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mb-12">
              {posts.map((post) => (
                <Card
                  key={post.id}
                  className={`overflow-hidden hover:shadow-lg transition-shadow duration-300  border border-gray-300 dark:border-[#26282E]`}
                >
                  <div className="aspect-video overflow-hidden">
                    <img
                      src={post.featuredImage}
                      alt={post.title}
                      className="w-full h-full object-cover hover:scale-105 transition-transform duration-300"
                    />
                  </div>

                  <CardContent className="p-6">
                    <div className="mb-3">
                      <span
                        className={`inline-block px-3 py-1 text-xs font-medium rounded-full ${
                          theme === "dark"
                            ? "bg-[#26282E] text-white"
                            : "bg-gray-200 text-gray-800"
                        }`}
                      >
                        {post.category}
                      </span>
                    </div>

                    <h3
                      className={`text-xl font-bold mb-3 line-clamp-2 ${
                        theme === "dark" ? "text-white" : "text-gray-900"
                      }`}
                    >
                      {post.title}
                    </h3>

                    <p
                      className={`text-sm mb-4 line-clamp-3 ${
                        theme === "dark" ? "text-gray-300" : "text-gray-600"
                      }`}
                    >
                      {post.excerpt}
                    </p>

                    <div
                      className={`flex items-center justify-between text-xs mb-4 ${
                        theme === "dark" ? "text-gray-400" : "text-gray-500"
                      }`}
                    >
                      <div className="flex items-center gap-4">
                        <div className="flex items-center gap-1">
                          <Calendar className="h-3 w-3" />
                          <span>{formatDate(post.publishedDate)}</span>
                        </div>
                        <div className="flex items-center gap-1">
                          <Clock className="h-3 w-3" />
                          <span>{post.readTime} min read</span>
                        </div>
                      </div>
                    </div>

                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-2">
                        <img
                          src={post.author.avatar}
                          alt={post.author.name}
                          className="w-8 h-8 rounded-full object-cover"
                        />
                        <span
                          className={`text-sm font-medium ${
                            theme === "dark" ? "text-gray-300" : "text-gray-700"
                          }`}
                        >
                          {post.author.name}
                        </span>
                      </div>

                      <Link
                        href={`/blogs/${post.slug}`}
                        className="inline-flex gap-1 items-center px-4 py-2 bg-[#253237] dark:bg-white text-white dark:text-[#253237] text-sm font-medium rounded-lg hover:bg-[#1a2428] dark:hover:bg-gray-100 transition-colors"
                      >
                        Read More
                        <ArrowRightIcon className="w-4" />
                      </Link>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          ) : (
            <div className="text-center py-12">
              <div
                className={`text-6xl mb-4 ${
                  theme === "dark" ? "text-gray-600" : "text-gray-400"
                }`}
              >
                📝
              </div>
              <h3
                className={`text-xl font-semibold mb-2 ${
                  theme === "dark" ? "text-white" : "text-gray-900"
                }`}
              >
                No blog posts found
              </h3>
              <p
                className={`${
                  theme === "dark" ? "text-gray-400" : "text-gray-600"
                }`}
              >
                Try adjusting your search terms or filters to find what
                you&apos;re looking for.
              </p>
              {error && <p className="mt-2 text-red-600">{error}</p>}
            </div>
          )}

          {currentPage < totalPages && (
            <div className="flex items-center justify-center gap-2 mt-8">
              <Button
                variant="outline"
                onClick={() => setCurrentPage((prev) => prev + 1)}
                disabled={loading}
                className={`min-w-[120px] cursor-pointer bg-white dark:bg-[#0B0C0E] ${
                  theme === "dark"
                    ? "border-gray-600 text-gray-300 hover:bg-gray-800"
                    : ""
                }`}
              >
                <FaArrowCircleDown />
                {loading ? "Loading..." : "Read More"}
              </Button>
            </div>
          )}
        </div>
        <SidebarDrawer open={leftDrawerOpen} toggle={toggleLeftDrawer} />
        <Footer />
      </div>

      
    </>
  );
};

export default BlogsHome;
