"use client";
import React, { useState } from "react";
import { useTheme } from "../atoms/theme";
import { FaCheck, FaRocket, FaStar } from "react-icons/fa";
import Link from "next/link";
import Image from "next/image";
import Footer from "../components/Footer";
import MainHeader from "../components/MainHeader";

interface Feature {
  name: string;
  included: boolean;
}

interface PricingPlan {
  name: string;
  description: string;
  monthlyPrice: number;
  yearlyPrice: number;
  features: Feature[];
  buttonText: string;
  featured: boolean;
}

interface Testimonial {
  name: string;
  role: string;
  image: string;
  rating: number;
  text: string;
}

interface FAQ {
  question: string;
  answer: string;
}

interface HeroData {
  badge?: string;
  title: string;
  subtitle?: string;
  description?: string;
  socialProof?: {
    images?: string[];
    userCount?: string;
  };
}

interface OfferData {
  hero?: HeroData;
  pricing?: {
    plans?: PricingPlan[];
  };
  testimonials?: Testimonial[];
  faq?: FAQ[];
  cta?: {
    badge?: string;
    title?: string;
    subtitle?: string;
    description?: string;
  };
}

interface DynamicWurknestOfferPageProps {
  data: OfferData;
}

const DynamicWurknestOfferPage: React.FC<DynamicWurknestOfferPageProps> = ({ data }) => {
  const { theme } = useTheme();
  const [isYearly, setIsYearly] = useState(false);

  // Default user images for social proof
  const defaultUserImages = [
    "https://i.pravatar.cc/150?img=1",
    "https://i.pravatar.cc/150?img=2",
    "https://i.pravatar.cc/150?img=3",
    "https://i.pravatar.cc/150?img=4",
    "https://i.pravatar.cc/150?img=5",
  ];

  const heroData = data.hero || {
    title: "Exclusive Deals Just For You",
    description: "Get access to premium features at unbeatable prices.",
  };

  const pricingPlans = data.pricing?.plans || [];
  const testimonials = data.testimonials || [];
  const faqs = data.faq || [];

  return (
    <div className="min-h-screen bg-white dark:bg-[#0a0a0a] text-gray-900 dark:text-white">
      <MainHeader />
      
      {/* Hero Section */}
      <section className="relative w-full py-20 md:py-32 px-4 overflow-hidden">
        {/* Decorative Background Elements */}
        <div className="absolute inset-0 overflow-hidden">
          <div className="absolute top-20 left-10 w-72 h-72 bg-gray-900/5 dark:bg-white/5 rounded-full blur-3xl" />
          <div className="absolute bottom-20 right-10 w-96 h-96 bg-gray-900/5 dark:bg-white/5 rounded-full blur-3xl" />
          <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-[800px] h-[800px] bg-gray-900/3 dark:bg-white/3 rounded-full blur-3xl" />
        </div>
        
        {/* Gradient Fade to Transparent at Bottom */}
        <div className="absolute bottom-0 left-0 right-0 h-32 bg-gradient-to-b from-transparent to-white dark:to-[#0a0a0a] pointer-events-none z-10" />

        <div className="container mx-auto max-w-6xl relative z-10">
          <div className="text-center space-y-8">
            {/* Badge */}
            {heroData.badge && (
              <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-white/50 dark:bg-black/50 backdrop-blur-sm border border-gray-900/10 dark:border-white/10 mb-4">
                <FaRocket className="text-gray-900 dark:text-white" />
                <span className="text-sm font-semibold text-gray-900 dark:text-white tracking-wider uppercase">
                  {heroData.badge}
                </span>
              </div>
            )}

            {/* Main Heading */}
            <h1 className="text-4xl md:text-6xl lg:text-7xl font-bold text-gray-900 dark:text-white leading-tight">
              {heroData.title}
              {heroData.subtitle && (
                <>
                  <br />
                  <span className="text-gray-900 dark:text-white">{heroData.subtitle}</span>
                </>
              )}
            </h1>

            {/* Description */}
            {heroData.description && (
              <p className="text-lg md:text-xl text-gray-600 dark:text-gray-300 max-w-2xl mx-auto leading-relaxed">
                {heroData.description}
              </p>
            )}

            {/* Social Proof */}
            {heroData.socialProof && (
              <div className="pt-6 flex flex-col items-center gap-4">
                <div className="flex -space-x-3">
                  {(heroData.socialProof.images || defaultUserImages).map((img, i) => (
                    <img
                      key={i}
                      src={img}
                      alt={`User ${i + 1}`}
                      className="w-12 h-12 rounded-full border-2 border-white dark:border-black object-cover"
                    />
                  ))}
                </div>
                <p className="text-sm text-gray-600 dark:text-gray-400">
                  <span className="font-semibold text-gray-900 dark:text-white">
                    {heroData.socialProof.userCount || "51,000+ users"}
                  </span>{" "}
                  already enjoying our services
                </p>
              </div>
            )}
          </div>
        </div>
      </section>

      {/* Pricing Toggle */}
      {pricingPlans.length > 0 && (
        <>
          <section className="pb-12 px-4">
            <div className="container mx-auto max-w-6xl flex justify-center">
              <div className="inline-flex items-center bg-white/50 dark:bg-black/50 backdrop-blur-sm border border-gray-900/10 dark:border-white/10 rounded-2xl p-1.5 shadow-lg">
                <button
                  onClick={() => setIsYearly(false)}
                  className={`px-8 py-3 rounded-xl transition-all font-semibold ${
                    !isYearly
                      ? "bg-gray-900 dark:bg-white text-white dark:text-black shadow-lg"
                      : "text-gray-600 dark:text-gray-400"
                  }`}
                >
                  Monthly
                </button>
                <button
                  onClick={() => setIsYearly(true)}
                  className={`px-8 py-3 rounded-xl transition-all font-semibold flex items-center gap-2 ${
                    isYearly
                      ? "bg-gray-900 dark:bg-white text-white dark:text-black shadow-lg"
                      : "text-gray-600 dark:text-gray-400"
                  }`}
                >
                  Yearly
                  <span className="text-xs bg-gray-900 dark:bg-white text-white dark:text-black px-2 py-1 rounded-full">
                    Save 20%
                  </span>
                </button>
              </div>
            </div>
          </section>

          {/* Pricing Cards */}
          <section className="pb-20 px-4">
            <div className="container mx-auto max-w-7xl">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
                {pricingPlans.map((plan, index) => (
                  <div
                    key={index}
                    className={`relative bg-white/30 dark:bg-black/30 backdrop-blur-xl border border-gray-900/10 dark:border-white/10 rounded-3xl p-8 shadow-2xl transition-all duration-500 hover:scale-105 ${
                      plan.featured ? "md:-mt-4 md:mb-4" : ""
                    }`}
                  >
                    {plan.featured && (
                      <div className="absolute -top-4 left-1/2 transform -translate-x-1/2">
                        <span className="bg-gray-900 dark:bg-white text-white dark:text-black px-6 py-2 rounded-full text-sm font-bold shadow-lg">
                          Most Popular
                        </span>
                      </div>
                    )}

                    {/* Decorative elements */}
                    <div className="absolute top-0 right-0 w-32 h-32 bg-gray-900/5 dark:bg-white/5 rounded-full -mr-16 -mt-16 blur-2xl" />
                    <div className="absolute bottom-0 left-0 w-32 h-32 bg-gray-900/5 dark:bg-white/5 rounded-full -ml-16 -mb-16 blur-2xl" />

                    <div className="relative z-10">
                      <div className="mb-6">
                        <h3 className="text-2xl md:text-3xl font-bold mb-2 text-gray-900 dark:text-white">
                          {plan.name}
                        </h3>
                        <p className="text-gray-600 dark:text-gray-400">
                          {plan.description}
                        </p>
                      </div>

                      <div className="mb-8">
                        <div className="flex items-baseline gap-1">
                          <span className="text-5xl md:text-6xl font-bold text-gray-900 dark:text-white">
                            ${isYearly ? plan.yearlyPrice : plan.monthlyPrice}
                          </span>
                          <span className="text-gray-600 dark:text-gray-400 text-lg">
                            /{isYearly ? "year" : "month"}
                          </span>
                        </div>
                        {isYearly && (
                          <p className="text-sm text-gray-600 dark:text-gray-400 mt-2">
                            Billed annually
                          </p>
                        )}
                      </div>

                      <ul className="space-y-4 mb-8">
                        {plan.features.map((feature, idx) => (
                          <li key={idx} className="flex items-center gap-3">
                            <div
                              className={`flex-shrink-0 w-5 h-5 rounded-full flex items-center justify-center ${
                                feature.included
                                  ? "bg-gray-900 dark:bg-white"
                                  : "bg-gray-300 dark:bg-gray-700"
                              }`}
                            >
                              <FaCheck
                                className={`text-xs ${
                                  feature.included
                                    ? "text-white dark:text-black"
                                    : "text-gray-500"
                                }`}
                              />
                            </div>
                            <span
                              className={
                                feature.included
                                  ? "text-gray-900 dark:text-white"
                                  : "text-gray-500 dark:text-gray-500 line-through"
                              }
                            >
                              {feature.name}
                            </span>
                          </li>
                        ))}
                      </ul>

                      <button
                        onClick={() => {
                          window.location.replace(
                            "https://hub.clikkle.com/signup?theme=" + theme
                          );
                        }}
                        className={`w-full py-4 rounded-xl font-bold transition-all duration-300 shadow-lg hover:shadow-xl hover:scale-105 ${
                          plan.featured
                            ? "bg-gray-900 dark:bg-white text-white dark:text-black"
                            : "bg-white/50 dark:bg-black/50 backdrop-blur-sm border border-gray-900/10 dark:border-white/10 text-gray-900 dark:text-white"
                        }`}
                      >
                        {plan.buttonText}
                      </button>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </section>
        </>
      )}

      {/* Testimonials Section */}
      {testimonials.length > 0 && (
        <section className="py-20 px-4 bg-white/50 dark:bg-black/50">
          <div className="container mx-auto max-w-7xl">
            <div className="text-center mb-16">
              <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-white/50 dark:bg-black/50 backdrop-blur-sm border border-gray-900/10 dark:border-white/10 mb-6">
                <span className="text-sm font-semibold text-gray-900 dark:text-white tracking-wider uppercase">
                  Testimonials
                </span>
              </div>
              <h2 className="text-3xl md:text-5xl lg:text-6xl font-bold mb-6 text-gray-900 dark:text-white">
                What Our Customers Say
              </h2>
              <p className="text-lg md:text-xl text-gray-600 dark:text-gray-400 max-w-2xl mx-auto">
                Trusted by thousands of businesses worldwide
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
              {testimonials.map((testimonial, index) => (
                <div
                  key={index}
                  className="bg-white/30 dark:bg-black/30 backdrop-blur-xl border border-gray-900/10 dark:border-white/10 rounded-3xl p-8 shadow-xl transition-all duration-500 hover:scale-105"
                >
                  {/* Rating */}
                  <div className="flex items-center gap-1 mb-6">
                    {[...Array(testimonial.rating)].map((_, i) => (
                      <FaStar key={i} className="text-gray-900 dark:text-white w-5 h-5" />
                    ))}
                  </div>

                  {/* Testimonial Text */}
                  <p className="text-gray-600 dark:text-gray-300 mb-6 leading-relaxed">
                    &quot;{testimonial.text}&quot;
                  </p>

                  {/* Author */}
                  <div className="flex items-center gap-4">
                    <img
                      src={testimonial.image}
                      alt={testimonial.name}
                      className="w-14 h-14 rounded-full object-cover border-2 border-gray-900/10 dark:border-white/10"
                    />
                    <div>
                      <p className="font-bold text-gray-900 dark:text-white">
                        {testimonial.name}
                      </p>
                      <p className="text-sm text-gray-600 dark:text-gray-400">
                        {testimonial.role}
                      </p>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </section>
      )}

      {/* FAQ Section */}
      {faqs.length > 0 && (
        <section className="py-20 px-4">
          <div className="container mx-auto max-w-4xl">
            <div className="text-center mb-16">
              <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-white/50 dark:bg-black/50 backdrop-blur-sm border border-gray-900/10 dark:border-white/10 mb-6">
                <span className="text-sm font-semibold text-gray-900 dark:text-white tracking-wider uppercase">
                  FAQ
                </span>
              </div>
              <h2 className="text-3xl md:text-5xl lg:text-6xl font-bold text-gray-900 dark:text-white">
                Frequently Asked Questions
              </h2>
            </div>

            <div className="space-y-6">
              {faqs.map((faq, idx) => (
                <div
                  key={idx}
                  className="bg-white/30 dark:bg-black/30 backdrop-blur-xl border border-gray-900/10 dark:border-white/10 rounded-2xl p-6 md:p-8 transition-all duration-300 hover:shadow-xl"
                >
                  <h3 className="text-xl md:text-2xl font-bold mb-3 text-gray-900 dark:text-white">
                    {faq.question}
                  </h3>
                  <p className="text-gray-600 dark:text-gray-300 leading-relaxed">
                    {faq.answer}
                  </p>
                </div>
              ))}
            </div>
          </div>
        </section>
      )}

      {/* CTA Section - Wurknest Branded */}
      <section className="w-full flex justify-center items-center bg-transparent">
        <div className="relative my-24 mx-[1.2rem] md:mx-[50px] w-full max-w-[1062px]">
          {/* Glassmorphic Container */}
          <div className="bg-white/30 dark:bg-black/30 backdrop-blur-xl border border-gray-900/10 dark:border-white/10 rounded-3xl md:rounded-[40px] shadow-2xl transition-all duration-500 p-8 md:p-12 relative overflow-hidden">
            {/* Decorative background elements */}
            <div className="absolute top-0 right-0 w-64 h-64 bg-gray-900/5 dark:bg-white/5 rounded-full -mr-32 -mt-32 blur-3xl" />
            <div className="absolute bottom-0 left-0 w-64 h-64 bg-gray-900/5 dark:bg-white/5 rounded-full -ml-32 -mb-32 blur-3xl" />
            
            <div className="flex flex-col items-center justify-center h-full relative z-10">
              <div className="w-full text-center flex flex-col items-center gap-6 max-w-3xl mx-auto">
                {/* Heading */}
                <div>
                  {/* Desktop badge */}
                  <div className="hidden md:inline-flex items-center gap-2 px-4 py-2 rounded-full bg-white/50 dark:bg-black/50 backdrop-blur-sm border border-gray-900/10 dark:border-white/10 mb-6">
                    <span className="text-xs md:text-sm font-semibold text-gray-900 dark:text-white tracking-wider uppercase">
                      {data.cta?.badge || "Ready to Transform?"}
                    </span>
                  </div>
                  
                  {/* Mobile simplified headline */}
                  <h2 className="md:hidden !text-[28px] font-bold text-gray-900 dark:text-white mb-3 leading-tight">
                    {data.cta?.title || "Scale Your Business Today"}
                  </h2>
                  
                  {/* Desktop full headline */}
                  <div className="hidden md:block">
                    <h2 className="!text-[32px] md:!text-[56px] leading-none font-bold text-gray-900 dark:text-white mb-3">
                      {data.cta?.title || "Transform Your"}{" "}
                      <span className="text-[#253237] dark:text-white">
                        {data.cta?.subtitle || "Workflow"}
                      </span>
                    </h2>
                    <h3 className="!text-[32px] md:!text-[56px] font-bold text-gray-900 dark:text-white leading-tight">
                      Scale Your Business
                    </h3>
                  </div>
                </div>

                <p className="text-gray-600 dark:text-[#94A3B8] text-sm md:text-lg font-medium max-w-2xl mx-auto">
                  {data.cta?.description || "Join thousands of teams streamlining operations and driving growth with our all-in-one platform. Start free, scale fast."}
                </p>

                {/* Form Section */}
                <div className="w-full max-w-2xl flex flex-col items-center gap-4">
                  <div className="flex flex-col sm:flex-row items-center sm:items-stretch gap-3 w-full">
                    <input
                      type="email"
                      placeholder="Enter Your Email"
                      className="flex-1 px-5 py-4 rounded-xl bg-white/80 dark:bg-black/40 backdrop-blur-sm text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-gray-900/20 dark:focus:ring-white/20 w-full border border-gray-900/10 dark:border-white/10 shadow-lg transition-all duration-300 hover:bg-white dark:hover:bg-black/60"
                    />
                    <button
                      onClick={() => {
                        window.location.replace(
                          "https://hub.clikkle.com/signup?theme=" + theme
                        );
                      }}
                      className="cursor-pointer px-8 py-4 bg-[#253237] dark:bg-white text-white dark:text-[#253237] font-bold rounded-xl transition-all duration-300 w-full sm:w-auto shadow-xl hover:shadow-2xl hover:scale-105 hover:bg-[#1a2329] dark:hover:bg-gray-100"
                    >
                      Get Started Free
                    </button>
                  </div>

                  {/* Terms Checkbox */}
                  <div className="flex items-start gap-3 text-xs md:text-sm text-gray-600 dark:text-[#94A3B8]">
                    <label className="relative flex items-center h-full cursor-pointer mt-0.5">
                      <input
                        type="checkbox"
                        id="terms"
                        aria-label="Agree to terms and conditions"
                        defaultChecked
                        className="peer appearance-none h-5 w-5 border-2 border-gray-300 dark:border-gray-600 rounded-md bg-white/50 dark:bg-black/50 backdrop-blur-sm checked:bg-[#253237] dark:checked:bg-white checked:border-[#253237] dark:checked:border-white transition duration-200"
                      />
                      <span className="absolute left-0 top-0 h-5 w-5 flex items-center justify-center pointer-events-none peer-checked:before:content-['✓'] peer-checked:before:text-white dark:peer-checked:before:text-[#253237] peer-checked:before:text-xs peer-checked:before:font-bold"></span>
                    </label>

                    <span className="leading-5 font-medium">
                      I agree with the{" "}
                      <Link
                        href="/terms-of-service"
                        className="underline text-gray-900 dark:text-white hover:text-[#253237] dark:hover:text-gray-300 transition-colors font-semibold"
                        target="_blank"
                        rel="noopener noreferrer"
                      >
                        Terms and Conditions
                      </Link>
                    </span>
                  </div>

                  {/* Trust Indicators */}
                  <div className="flex items-center gap-6 mt-2 flex-wrap justify-center">
                    <div className="flex items-center gap-2">
                      <div className="flex -space-x-2">
                        <img src="https://i.pravatar.cc/150?img=12" alt="User" className="w-8 h-8 rounded-full border-2 border-white dark:border-black object-cover" />
                        <img src="https://i.pravatar.cc/150?img=26" alt="User" className="w-8 h-8 rounded-full border-2 border-white dark:border-black object-cover" />
                        <img src="https://i.pravatar.cc/150?img=47" alt="User" className="w-8 h-8 rounded-full border-2 border-white dark:border-black object-cover" />
                      </div>
                      <span className="text-xs font-semibold text-gray-900 dark:text-white">51,000+ users</span>
                    </div>
                    <div className="flex items-center gap-1">
                      {[...Array(5)].map((_, i) => (
                        <svg key={i} className="w-4 h-4 text-gray-900 dark:text-white" fill="currentColor" viewBox="0 0 20 20">
                          <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                        </svg>
                      ))}
                      <span className="text-xs font-semibold text-gray-900 dark:text-white ml-1">4.9/5</span>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Footer */}
      <Footer />
    </div>
  );
};

export default DynamicWurknestOfferPage;
